/// <reference types="node" />
import * as express from 'express';
import * as firebase from 'firebase-admin';
import { HttpsFunction, Runnable } from '../cloud-functions';
import { DeploymentOptions } from '../function-configuration';
/** @hidden */
export interface Request extends express.Request {
    rawBody: Buffer;
}
/**
 * Handle HTTP requests.
 * @param handler A function that takes a request and response object,
 * same signature as an Express app.
 */
export declare function onRequest(handler: (req: Request, resp: express.Response) => void | Promise<void>): HttpsFunction;
/**
 * Declares a callable method for clients to call using a Firebase SDK.
 * @param handler A method that takes a data and context and returns a value.
 */
export declare function onCall(handler: (data: any, context: CallableContext) => any | Promise<any>): HttpsFunction & Runnable<any>;
/** @hidden */
export declare function _onRequestWithOptions(handler: (req: Request, resp: express.Response) => void | Promise<void>, options: DeploymentOptions): HttpsFunction;
/**
 * The set of Firebase Functions status codes. The codes are the same at the
 * ones exposed by gRPC here:
 * https://github.com/grpc/grpc/blob/master/doc/statuscodes.md
 *
 * Possible values:
 * - 'cancelled': The operation was cancelled (typically by the caller).
 * - 'unknown': Unknown error or an error from a different error domain.
 * - 'invalid-argument': Client specified an invalid argument. Note that this
 *   differs from 'failed-precondition'. 'invalid-argument' indicates
 *   arguments that are problematic regardless of the state of the system
 *   (e.g. an invalid field name).
 * - 'deadline-exceeded': Deadline expired before operation could complete.
 *   For operations that change the state of the system, this error may be
 *   returned even if the operation has completed successfully. For example,
 *   a successful response from a server could have been delayed long enough
 *   for the deadline to expire.
 * - 'not-found': Some requested document was not found.
 * - 'already-exists': Some document that we attempted to create already
 *   exists.
 * - 'permission-denied': The caller does not have permission to execute the
 *   specified operation.
 * - 'resource-exhausted': Some resource has been exhausted, perhaps a
 *   per-user quota, or perhaps the entire file system is out of space.
 * - 'failed-precondition': Operation was rejected because the system is not
 *   in a state required for the operation's execution.
 * - 'aborted': The operation was aborted, typically due to a concurrency
 *   issue like transaction aborts, etc.
 * - 'out-of-range': Operation was attempted past the valid range.
 * - 'unimplemented': Operation is not implemented or not supported/enabled.
 * - 'internal': Internal errors. Means some invariants expected by
 *   underlying system has been broken. If you see one of these errors,
 *   something is very broken.
 * - 'unavailable': The service is currently unavailable. This is most likely
 *   a transient condition and may be corrected by retrying with a backoff.
 * - 'data-loss': Unrecoverable data loss or corruption.
 * - 'unauthenticated': The request does not have valid authentication
 *   credentials for the operation.
 */
export declare type FunctionsErrorCode = 'ok' | 'cancelled' | 'unknown' | 'invalid-argument' | 'deadline-exceeded' | 'not-found' | 'already-exists' | 'permission-denied' | 'resource-exhausted' | 'failed-precondition' | 'aborted' | 'out-of-range' | 'unimplemented' | 'internal' | 'unavailable' | 'data-loss' | 'unauthenticated';
/** @hidden */
export declare type CanonicalErrorCodeName = 'OK' | 'CANCELLED' | 'UNKNOWN' | 'INVALID_ARGUMENT' | 'DEADLINE_EXCEEDED' | 'NOT_FOUND' | 'ALREADY_EXISTS' | 'PERMISSION_DENIED' | 'UNAUTHENTICATED' | 'RESOURCE_EXHAUSTED' | 'FAILED_PRECONDITION' | 'ABORTED' | 'OUT_OF_RANGE' | 'UNIMPLEMENTED' | 'INTERNAL' | 'UNAVAILABLE' | 'DATA_LOSS';
/** @hidden */
interface HttpErrorCode {
    canonicalName: CanonicalErrorCodeName;
    status: number;
}
/** @hidden */
interface HttpErrorWireFormat {
    details?: unknown;
    message: string;
    status: CanonicalErrorCodeName;
}
/**
 * An explicit error that can be thrown from a handler to send an error to the
 * client that called the function.
 */
export declare class HttpsError extends Error {
    /**
     * A standard error code that will be returned to the client. This also
     * determines the HTTP status code of the response, as defined in code.proto.
     */
    readonly code: FunctionsErrorCode;
    /**
     * Extra data to be converted to JSON and included in the error response.
     */
    readonly details: unknown;
    /**
     * A wire format representation of a provided error code.
     *
     * @hidden
     */
    readonly httpErrorCode: HttpErrorCode;
    constructor(code: FunctionsErrorCode, message: string, details?: unknown);
    /** @hidden */
    toJSON(): HttpErrorWireFormat;
}
/**
 * The interface for metadata for the API as passed to the handler.
 */
export interface CallableContext {
    /**
     * The result of decoding and verifying a Firebase Auth ID token.
     */
    auth?: {
        uid: string;
        token: firebase.auth.DecodedIdToken;
    };
    /**
     * An unverified token for a Firebase Instance ID.
     */
    instanceIdToken?: string;
    /**
     * The raw request handled by the callable.
     */
    rawRequest: Request;
}
/**
 * Encodes arbitrary data in our special format for JSON.
 * This is exposed only for testing.
 */
/** @hidden */
export declare function encode(data: any): any;
/**
 * Decodes our special format for JSON into native types.
 * This is exposed only for testing.
 */
/** @hidden */
export declare function decode(data: any): any;
/** @hidden */
export declare function _onCallWithOptions(handler: (data: any, context: CallableContext) => any | Promise<any>, options: DeploymentOptions): HttpsFunction & Runnable<any>;
export {};
