"use strict";
// The MIT License (MIT)
//
// Copyright (c) 2017 Firebase
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
Object.defineProperty(exports, "__esModule", { value: true });
exports.IssueBuilder = exports._issueWithOptions = exports.issue = exports.service = exports.provider = void 0;
const cloud_functions_1 = require("../cloud-functions");
/** @hidden */
exports.provider = 'google.firebase.crashlytics';
/** @hidden */
exports.service = 'fabric.io';
/**
 * Registers a Cloud Function to handle Crashlytics issue events.
 *
 * @returns Crashlytics issue event builder interface.
 */
function issue() {
    return _issueWithOptions({});
}
exports.issue = issue;
/** @hidden */
function _issueWithOptions(options) {
    return new IssueBuilder(() => {
        if (!process.env.GCLOUD_PROJECT) {
            throw new Error('process.env.GCLOUD_PROJECT is not set.');
        }
        return 'projects/' + process.env.GCLOUD_PROJECT;
    }, options);
}
exports._issueWithOptions = _issueWithOptions;
/** The Firebase Crashlytics issue builder interface. */
class IssueBuilder {
    /** @hidden */
    constructor(triggerResource, options) {
        this.triggerResource = triggerResource;
        this.options = options;
    }
    /** @hidden */
    onNewDetected(handler) {
        throw new Error('"onNewDetected" is now deprecated, please use "onNew"');
    }
    /**
     * Event handler that fires every time a new issue occurs in a project.
     *
     * @param handler Event handler that fires every time a new issue event occurs.
     * @example
     * ```javascript
     * exports.postOnNewIssue = functions.crashlytics.issue().onNew(event => {
     *   const { data } = event;
     *   issueId = data.issueId;
     *   issueTitle =  data.issueTitle;
     *   const slackMessage = ` There's a new issue (${issueId}) ` +
     *       `in your app - ${issueTitle}`;
     *   return notifySlack(slackMessage).then(() => {
     *     functions.logger.info(`Posted new issue ${issueId} successfully to Slack`);
     *   });
     * });
     * ```
     */
    onNew(handler) {
        return this.onEvent(handler, 'issue.new');
    }
    /**
     * Event handler that fires every time a regressed issue reoccurs in a project.
     *
     * @param handler Event handler that fires every time a regressed issue event occurs.
     */
    onRegressed(handler) {
        return this.onEvent(handler, 'issue.regressed');
    }
    /**
     * Event handler that fires every time a velocity alert occurs in a project.
     *
     * @param handler handler that fires every time a velocity alert issue event occurs.
     */
    onVelocityAlert(handler) {
        return this.onEvent(handler, 'issue.velocityAlert');
    }
    onEvent(handler, eventType) {
        return cloud_functions_1.makeCloudFunction({
            handler,
            provider: exports.provider,
            eventType,
            service: exports.service,
            legacyEventType: `providers/firebase.crashlytics/eventTypes/${eventType}`,
            triggerResource: this.triggerResource,
            options: this.options,
        });
    }
}
exports.IssueBuilder = IssueBuilder;
