const functions = require("firebase-functions");
const admin = require("firebase-admin");
const https = require("https");
const axios = require("axios");
const path = require('path');
const os = require('os');
const fs = require('fs');
const got = require('got');
const FileType = require('file-type');


admin.initializeApp();

//Chage this to your collection id 
const collectionId = "ImageList";

//Chage this to your storage bucket path to save files 
const bucketPath = "Images"; 

//This function will be called when the document created in the defined collection
exports.firestoreImageDownloadTrigger = functions.firestore.document(`${collectionId}/{id}`).onCreate(
    async (snapshot) => {

        const data = snapshot.data();

        const fileUrl = data.url;
        const name = data.name;
        
        let tempFile = null;

        try {
        
            tempFile = await download(fileUrl, name);
            console.log("File " + tempFile.path + " (" +  tempFile.name + ") dowloaded.");
			
            await upload(tempFile.path, tempFile.name, bucketPath);

            return snapshot.ref.set({
                bucketPath: bucketPath + "/" + tempFile.name
              }, {merge: true});
        }
        catch(error) {
            console.log("Error while downloading and uploading file, " + error);
        }
    }
);


async function download(fileUrl, name) {
 
    let ext = "";
    const fileType = await getFileType(fileUrl);
    if (fileType) {
        ext = "." + fileType.ext;
    }

    const fileName = name + ext;
    const tempFilePath = path.join(os.tmpdir(), (fileName));

    const writer = fs.createWriteStream(tempFilePath);

    return axios({
        method: 'get',
        url: fileUrl,
        responseType: 'stream',
    }).then(response => {
        
        return new Promise((resolve, reject) => {
            response.data.pipe(writer);
            let error = null;
            writer.on('error', err => {
                error = err;
                writer.close();
                throw new Error(err);
            });
            writer.on('close', () => {
                if (!error) {
                    resolve({
                        "path": tempFilePath, 
                        "name": fileName
                    });
                }
            });
        });
    });
}


async function getFileType(fileUrl) {
    const stream = got.stream(fileUrl);
    try {
        const fileType = await FileType.fromStream(stream);
        return fileType;
    }
    catch(e) {
        console.error("Error to get file type: ", e);
        return null
    }
}


async function upload(filePath, fileName, bucketPath) {
    
    const bucket = admin.storage().bucket();

    try {
        await bucket.upload(filePath, {
            destination: `${bucketPath}/${fileName}`,
            gzip: true,
            metadata: {
                cacheControl: 'public, max-age=31536000',
            },
        });

        console.log(`${fileName} uploaded`);
		
		try {
			fs.unlinkSync(filePath)
        } catch(err) {
			console.error("Error while delete the temp file, ", err)
        }
    }
    catch (e) {
        throw new Error("File upload failed, " + e);
    }
}