import { CloudFunction, EventContext } from '../cloud-functions';
import { DeploymentOptions } from '../function-configuration';
/** @hidden */
export declare const provider = "google.firebase.crashlytics";
/** @hidden */
export declare const service = "fabric.io";
/**
 * Registers a Cloud Function to handle Crashlytics issue events.
 *
 * @returns Crashlytics issue event builder interface.
 */
export declare function issue(): IssueBuilder;
/** @hidden */
export declare function _issueWithOptions(options: DeploymentOptions): IssueBuilder;
/** The Firebase Crashlytics issue builder interface. */
export declare class IssueBuilder {
    private triggerResource;
    private options;
    /** @hidden */
    constructor(triggerResource: () => string, options: DeploymentOptions);
    /** @hidden */
    onNewDetected(handler: any): Error;
    /**
     * Event handler that fires every time a new issue occurs in a project.
     *
     * @param handler Event handler that fires every time a new issue event occurs.
     * @example
     * ```javascript
     * exports.postOnNewIssue = functions.crashlytics.issue().onNew(event => {
     *   const { data } = event;
     *   issueId = data.issueId;
     *   issueTitle =  data.issueTitle;
     *   const slackMessage = ` There's a new issue (${issueId}) ` +
     *       `in your app - ${issueTitle}`;
     *   return notifySlack(slackMessage).then(() => {
     *     functions.logger.info(`Posted new issue ${issueId} successfully to Slack`);
     *   });
     * });
     * ```
     */
    onNew(handler: (issue: Issue, context: EventContext) => PromiseLike<any> | any): CloudFunction<Issue>;
    /**
     * Event handler that fires every time a regressed issue reoccurs in a project.
     *
     * @param handler Event handler that fires every time a regressed issue event occurs.
     */
    onRegressed(handler: (issue: Issue, context: EventContext) => PromiseLike<any> | any): CloudFunction<Issue>;
    /**
     * Event handler that fires every time a velocity alert occurs in a project.
     *
     * @param handler handler that fires every time a velocity alert issue event occurs.
     */
    onVelocityAlert(handler: (issue: Issue, context: EventContext) => PromiseLike<any> | any): CloudFunction<Issue>;
    private onEvent;
}
/**
 * Interface representing a Firebase Crashlytics event that was logged for a specific issue.
 */
export interface Issue {
    /** Crashlytics-provided issue ID. */
    issueId: string;
    /** Crashlytics-provided issue title. */
    issueTitle: string;
    /** AppInfo interface describing the App. */
    appInfo: AppInfo;
    /**
     * UTC when the issue occurred in ISO8601 standard representation.
     *
     * Example: 1970-01-17T10:52:15.661-08:00
     */
    createTime: string;
    /**
     * UTC When the issue was closed in ISO8601 standard representation.
     *
     * Example: 1970-01-17T10:52:15.661-08:00
     */
    resolvedTime?: string;
    /** Information about the velocity alert, like number of crashes and percentage of users affected by the issue. */
    velocityAlert?: VelocityAlert;
}
/** Interface representing Firebase Crashlytics VelocityAlert data. */
export interface VelocityAlert {
    /**
     * The percentage of sessions which have been impacted by this issue.
     *
     * Example: .04
     */
    crashPercentage: number;
    /** The number of crashes that this issue has caused. */
    crashes: number;
}
/** Interface representing Firebase Crashlytics AppInfo data. */
export interface AppInfo {
    /**
     * The app's name.
     *
     * Example: "My Awesome App".
     */
    appName: string;
    /** The app's platform.
     *
     * Examples: "android", "ios".
     */
    appPlatform: string;
    /** Unique application identifier within an app store, either the Android package name or the iOS bundle id. */
    appId: string;
    /**
     * The app's version name.
     *
     * Examples: "1.0", "4.3.1.1.213361", "2.3 (1824253)", "v1.8b22p6".
     */
    latestAppVersion: string;
}
