#ifndef SCHNEIDER_MODBUS_TCP_H
#define SCHNEIDER_MODBUS_TCP_H

#include <Arduino.h>
#include <WiFi.h> // Ensure WiFi.h is included for WiFiClient usage in .cpp

class SchneiderModbusTCP {
public:
    // Public functions for reading registers
    bool readHoldingRegister16(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint16_t& value);
    bool readHoldingRegister32(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint32_t& value);
    bool readHoldingRegister16s(const char* host, uint16_t port, uint8_t slave, uint16_t reg, int16_t& value);
    bool readHoldingRegister32s(const char* host, uint16_t port, uint8_t slave, uint16_t reg, int32_t& value);

    // Public functions for writing registers
    bool writeSingleRegister16(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint16_t value);
    bool writeSingleRegister32(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint32_t value);

    // New: Public functions for reading and writing Coils (binary states)

    /*
     * @brief Reads the state of a single Modbus Coil (Function Code 0x01).
     * @param host The IP address of the Modbus Gateway.
     * @param port The Modbus TCP port (usually 503).
     * @param slave The Modbus Slave ID of the target device.
     * @param coil The coil address (0-based).
     * @param value Output parameter to store the coil state (true for ON, false for OFF).
     * @return True if the read was successful, false otherwise.
     *
     * Example Use Case: Check if Generator is running (Coil 0x0001 on AGS, Slave ID 50 - example)
     * bool generatorRunning;
     * if (modbus.readCoil(inverterIP, inverterPort, 50, 0x0001, generatorRunning)) {
     * Serial.printf("Generator Running: %s\n", generatorRunning ? "YES" : "NO");
     * } else {
     * Serial.println("Failed to read generator status.");
     * }
     */
    bool readCoil(const char* host, uint16_t port, uint8_t slave, uint16_t coil, bool& value);

    /*
     * @brief Writes a single Modbus Coil (Function Code 0x05) to set its state.
     * @param host The IP address of the Modbus Gateway.
     * @param port The Modbus TCP port (usually 503).
     * @param slave The Modbus Slave ID of the target device.
     * @param coil The coil address (0-based).
     * @param value The desired state for the coil (true for ON/SET, false for OFF/CLEAR).
     * @return True if the write was successful, false otherwise.
     *
     * Example Use Case: Toggle Generator ON/OFF (Coil 0x0001 on AGS, Slave ID 50 - example)
     * // To turn generator ON
     * if (modbus.writeSingleCoil(inverterIP, inverterPort, 50, 0x0001, true)) {
     * Serial.println("Generator commanded ON successfully.");
     * } else {
     * Serial.println("Failed to command generator ON.");
     * }
     * // To turn generator OFF
     * if (modbus.writeSingleCoil(inverterIP, inverterPort, 50, 0x0001, false)) {
     * Serial.println("Generator commanded OFF successfully.");
     * } else {
     * Serial.println("Failed to command generator OFF.");
     * }
     */
    bool writeSingleCoil(const char* host, uint16_t port, uint8_t slave, uint16_t coil, bool value);

private:
    // Private helper function for sending Modbus requests
    // Parameter 'quantity' now also serves as the value for writeSingleCoil (FC 0x05)
    bool sendModbusRequest(const char* host, uint16_t port, uint8_t slave, uint8_t functionCode, uint16_t startRegOrCoil, uint16_t quantityOrValue, uint8_t* sendData = nullptr, size_t sendDataLen = 0, uint8_t* responseBuffer = nullptr, size_t bufferSize = 0, size_t* bytesRead = nullptr);
};

#endif
