#include "SchneiderModbusTCP.h"
#include <WiFiClient.h> // Include for WiFiClient
#include <Arduino.h>    // For Serial.printf and millis()

/**
 * @brief Helper function to send Modbus TCP requests and receive responses.
 * @param host The IP address of the Modbus Gateway.
 * @param port The Modbus TCP port (usually 503).
 * @param slave The Modbus Slave ID of the target device.
 * @param functionCode The Modbus function code (e.g., 0x01 Read Coils, 0x03 Read Holding, 0x05 Write Single Coil, 0x06 Write Single Reg, 0x10 Write Multiple Reg).
 * @param startRegOrCoil The starting register or coil address (0-based).
 * @param quantityOrValue For read operations, this is the quantity of registers/coils. For FC 0x05/0x06, it's the value to write.
 * @param sendData Optional pointer to data bytes to send (for FC 0x10).
 * @param sendDataLen Length of sendData.
 * @param responseBuffer Optional buffer to store the received data bytes.
 * @param bufferSize The maximum size of the responseBuffer.
 * @param bytesRead Optional output parameter to store the actual number of data bytes read into responseBuffer.
 * @return True if the request was successful, false otherwise.
 */
bool SchneiderModbusTCP::sendModbusRequest(const char* host, uint16_t port, uint8_t slave, uint8_t functionCode, uint16_t startRegOrCoil, uint16_t quantityOrValue, uint8_t* sendData, size_t sendDataLen, uint8_t* responseBuffer, size_t bufferSize, size_t* bytesRead) {
    WiFiClient client;
    if (!client.connect(host, port)) {
        Serial.printf("Failed to connect to Modbus server %s:%d\n", host, port);
        return false;
    }

    uint16_t transactionID = millis() & 0xFFFF; // Simple transaction ID
    uint16_t protocolID = 0; // Modbus TCP protocol identifier (always 0)
    
    uint16_t pduLen = 0;
    if (functionCode == 0x01 || functionCode == 0x03) { // Read Coils / Read Holding Registers
        pduLen = 5; // FC (1) + Start Addr (2) + Quantity (2)
    } else if (functionCode == 0x05 || functionCode == 0x06) { // Write Single Coil / Write Single Register
        pduLen = 5; // FC (1) + Address (2) + Value (2)
    } else if (functionCode == 0x10) { // Write Multiple Registers
        pduLen = 6 + sendDataLen; // FC (1) + Start Addr (2) + Quantity (2) + Byte Count (1) + Data (N)
    } else {
        Serial.printf("Unsupported Modbus function code: 0x%02X\n", functionCode);
        client.stop();
        return false;
    }

    uint16_t totalLength = pduLen + 1; // PDU length + Unit ID (1 byte)

    uint8_t request[260]; // Max Modbus ADU size for TCP (253 bytes PDU + 7 bytes MBAP)
    
    request[0] = (transactionID >> 8) & 0xFF;
    request[1] = transactionID & 0xFF;
    request[2] = (protocolID >> 8) & 0xFF;
    request[3] = protocolID & 0xFF;
    request[4] = (totalLength >> 8) & 0xFF; // Length of following bytes (PDU + Unit ID)
    request[5] = totalLength & 0xFF;
    request[6] = slave;             // Unit ID (Slave ID)

    // Populate PDU
    request[7] = functionCode;
    request[8] = (startRegOrCoil >> 8) & 0xFF;
    request[9] = startRegOrCoil & 0xFF;

    if (functionCode == 0x01 || functionCode == 0x03) { // Read Coils / Read Holding Registers
        request[10] = (quantityOrValue >> 8) & 0xFF; // Quantity
        request[11] = quantityOrValue & 0xFF;
    } else if (functionCode == 0x05 || functionCode == 0x06) { // Write Single Coil / Write Single Register
        request[10] = (quantityOrValue >> 8) & 0xFF; // Value
        request[11] = quantityOrValue & 0xFF;        // Value
    } else if (functionCode == 0x10) { // Write Multiple Registers
        request[10] = (quantityOrValue >> 8) & 0xFF; // Quantity of registers
        request[11] = quantityOrValue & 0xFF;
        request[12] = sendDataLen;                   // Byte Count
        memcpy(&request[13], sendData, sendDataLen); // Copy data
    }
    
    size_t requestLen = 7 + pduLen; // MBAP header (7) + PDU

    client.write(request, requestLen);

    // Determine expected minimum response size and timeout
    size_t minExpectedResponseLen = 0;
    if (functionCode == 0x01 || functionCode == 0x03) { // Read response: MBAP (7) + FC (1) + Byte Count (1) + Data (N)
        minExpectedResponseLen = 9;
    } else if (functionCode == 0x05 || functionCode == 0x06 || functionCode == 0x10) { // Write response (echoes request or fixed structure)
        minExpectedResponseLen = 12; // MBAP (7) + FC (1) + Addr (2) + Value/Quantity (2)
    } else {
        client.stop();
        return false; // Should not happen due to earlier check
    }

    unsigned long startTime = millis();
    while (client.available() < minExpectedResponseLen && (millis() - startTime < 1000)) { // 1 second timeout
        delay(1);
    }

    if (client.available() < minExpectedResponseLen) {
        Serial.printf("No complete Modbus response header received within timeout for FC 0x%02X.\n", functionCode);
        client.stop();
        return false;
    }

    // Read the initial part of the response (MBAP header + FC + first data byte/byte count)
    uint8_t responseHeader[minExpectedResponseLen]; // Adjust size to read echo response fully
    client.readBytes(responseHeader, minExpectedResponseLen);

    // Verify transaction ID
    uint16_t r_transactionID = (responseHeader[0] << 8) | responseHeader[1];
    if (r_transactionID != transactionID) {
        Serial.printf("Modbus Transaction ID mismatch. Sent: %04X, Received: %04X\n", transactionID, r_transactionID);
        client.stop();
        return false;
    }

    uint8_t r_functionCode = responseHeader[7];
    if (r_functionCode > 0x80) { // Check for Modbus exception response
        uint8_t exceptionCode = responseHeader[8];
        Serial.printf("Modbus Exception Response from Slave %d: FC 0x%02X, Code 0x%02X\n", slave, r_functionCode, exceptionCode);
        client.stop();
        return false;
    }

    bool success = true;
    if (functionCode == 0x01 || functionCode == 0x03) { // Read operations
        uint8_t byteCount = responseHeader[8]; // This byte holds the number of data bytes for FC01/03
        if (responseBuffer != nullptr && bytesRead != nullptr) {
            if (byteCount > bufferSize) {
                Serial.printf("Response data size (%d) exceeds buffer capacity (%d) for FC 0x%02X.\n", byteCount, bufferSize, functionCode);
                success = false;
            } else {
                startTime = millis();
                while (client.available() < byteCount && (millis() - startTime < 1000)) { // Wait for all data bytes
                    delay(1);
                }
                if (client.available() < byteCount) {
                    Serial.printf("Incomplete Modbus response data. Expected %d, got %d for FC 0x%02X.\n", byteCount, client.available(), functionCode);
                    success = false;
                } else {
                    *bytesRead = client.readBytes(responseBuffer, byteCount);
                    if (*bytesRead != byteCount) {
                        Serial.printf("Error reading all data bytes. Expected %d, Read %d for FC 0x%02X.\n", byteCount, *bytesRead, functionCode);
                        success = false;
                    }
                }
            }
        }
    } else { // Write operations (FC 0x05, 0x06, 0x10) - response is typically an echo of the request
        // For writes, if we reached here without exception and the response length is minExpectedResponseLen, it's generally good.
        Serial.printf("Modbus Write (FC 0x%02X) successful for slave %d, address 0x%04X.\n", functionCode, slave, startRegOrCoil);
    }

    // Flush any remaining bytes in the buffer (important if response was larger or unexpected)
    while(client.available()) {
        client.read();
    }
    client.stop();
    return success;
}


// --- Register Read Functions ---

bool SchneiderModbusTCP::readHoldingRegister16(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint16_t& value) {
    uint8_t responseData[2]; // 16-bit value
    size_t bytesRead = 0;

    if (sendModbusRequest(host, port, slave, 0x03, reg, 1, nullptr, 0, responseData, sizeof(responseData), &bytesRead)) {
        if (bytesRead == 2) {
            value = (responseData[0] << 8) | responseData[1]; // Big-Endian
            return true;
        } else {
            Serial.printf("readHoldingRegister16: Expected 2 data bytes, got %d for reg 0x%04X, slave %d\n", bytesRead, reg, slave);
        }
    }
    return false;
}

bool SchneiderModbusTCP::readHoldingRegister32(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint32_t& value) {
    uint8_t responseData[4]; // 32-bit value (2 Modbus registers)
    size_t bytesRead = 0;

    if (sendModbusRequest(host, port, slave, 0x03, reg, 2, nullptr, 0, responseData, sizeof(responseData), &bytesRead)) {
        if (bytesRead == 4) {
            // Schneider devices store 32-bit values as two 16-bit registers with MSW first.
            // Each 16-bit word is also Big-Endian.
            // [MSB_MSW, LSB_MSW, MSB_LSW, LSB_LSW]
            value = ((uint32_t)responseData[0] << 24) |
                    ((uint32_t)responseData[1] << 16) |
                    ((uint32_t)responseData[2] << 8)  |
                    ((uint32_t)responseData[3]);
            return true;
        } else {
            Serial.printf("readHoldingRegister32: Expected 4 data bytes, got %d for reg 0x%04X, slave %d\n", bytesRead, reg, slave);
        }
    }
    return false;
}

bool SchneiderModbusTCP::readHoldingRegister16s(const char* host, uint16_t port, uint8_t slave, uint16_t reg, int16_t& value) {
    uint16_t raw;
    bool ok = readHoldingRegister16(host, port, slave, reg, raw);
    if (ok) value = (int16_t)raw;
    return ok;
}

bool SchneiderModbusTCP::readHoldingRegister32s(const char* host, uint16_t port, uint8_t slave, uint16_t reg, int32_t& value) {
    uint32_t raw;
    bool ok = readHoldingRegister32(host, port, slave, reg, raw);
    if (ok) value = (int32_t)raw;
    return ok;
}

// --- Register Write Functions ---

bool SchneiderModbusTCP::writeSingleRegister16(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint16_t value) {
    bool ok = sendModbusRequest(host, port, slave, 0x06, reg, value);
    if (ok) {
        delay(200); // Important: Add a delay after writes for Schneider devices as per documentation
        Serial.printf("Successfully wrote 16-bit value %u to register 0x%04X on slave %d.\n", value, reg, slave);
    } else {
        Serial.printf("Failed to write 16-bit value %u to register 0x%04X on slave %d.\n", value, reg, slave);
    }
    return ok;
}

bool SchneiderModbusTCP::writeSingleRegister32(const char* host, uint16_t port, uint8_t slave, uint16_t reg, uint32_t value) {
    uint8_t writeData[4]; // Two 16-bit registers = 4 bytes

    uint16_t msw = (uint16_t)((value >> 16) & 0xFFFF);
    uint16_t lsw = (uint16_t)(value & 0xFFFF);

    writeData[0] = (msw >> 8) & 0xFF;
    writeData[1] = msw & 0xFF;
    writeData[2] = (lsw >> 8) & 0xFF;
    writeData[3] = lsw & 0xFF;

    bool ok = sendModbusRequest(host, port, slave, 0x10, reg, 2, writeData, sizeof(writeData));
    if (ok) {
        delay(200); // Important: Add a delay after writes for Schneider devices
        Serial.printf("Successfully wrote 32-bit value %lu to registers 0x%04X, 0x%04X on slave %d.\n", value, reg, reg + 1, slave);
    } else {
        Serial.printf("Failed to write 32-bit value %lu to registers 0x%04X on slave %d.\n", value, reg, reg + 1, slave);
    }
    return ok;
}

// --- New Coil Functions Implementations ---

/**
 * @brief Reads the state of a single Modbus Coil (Function Code 0x01).
 * @param host The IP address of the Modbus Gateway.
 * @param port The Modbus TCP port.
 * @param slave The Modbus Slave ID.
 * @param coil The coil address (0-based).
 * @param value Output parameter to store the coil state (true for ON, false for OFF).
 * @return True if the read was successful, false otherwise.
 */
bool SchneiderModbusTCP::readCoil(const char* host, uint16_t port, uint8_t slave, uint16_t coil, bool& value) {
    uint8_t responseData[1]; // Coil read response data: 1 byte representing 8 coils (even if reading one)
    size_t bytesRead = 0;

    // For FC 0x01, quantity is 1 coil
    if (sendModbusRequest(host, port, slave, 0x01, coil, 1, nullptr, 0, responseData, sizeof(responseData), &bytesRead)) {
        if (bytesRead == 1) { // Expect 1 byte of data for 1 coil
            value = (responseData[0] & 0x01) != 0; // Check the first bit of the byte
            return true;
        } else {
            Serial.printf("readCoil: Expected 1 data byte, got %d for coil 0x%04X, slave %d\n", bytesRead, coil, slave);
        }
    }
    return false;
}

/**
 * @brief Writes a single Modbus Coil (Function Code 0x05) to set its state.
 * @param host The IP address of the Modbus Gateway.
 * @param port The Modbus TCP port.
 * @param slave The Modbus Slave ID.
 * @param coil The coil address (0-based).
 * @param value The desired state for the coil (true for ON/SET, false for OFF/CLEAR).
 * @return True if the write was successful, false otherwise.
 */
bool SchneiderModbusTCP::writeSingleCoil(const char* host, uint16_t port, uint8_t slave, uint16_t coil, bool value) {
    // For FC 0x05, the value is 0xFF00 for ON, 0x0000 for OFF
    uint16_t coilValue = value ? 0xFF00 : 0x0000;
    
    // sendModbusRequest's quantityOrValue parameter will be used as the value for FC 0x05
    bool ok = sendModbusRequest(host, port, slave, 0x05, coil, coilValue);
    if (ok) {
        delay(200); // Important: Add a delay after writes for Schneider devices
        Serial.printf("Successfully wrote coil 0x%04X to %s on slave %d.\n", coil, value ? "ON" : "OFF", slave);
    } else {
        Serial.printf("Failed to write coil 0x%04X to %s on slave %d.\n", coil, value ? "ON" : "OFF", slave);
    }
    return ok;
}
