# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Repository Overview

WAMR-ESP32-Arduino is a standalone Arduino/PlatformIO library that packages WebAssembly Micro Runtime (WAMR) for ESP32 and ESP32-S3 microcontrollers. This library provides an Arduino-friendly API wrapper around WAMR's core functionality.

## Repository Structure

```
wamr-esp32-arduino/
├── src/
│   ├── WAMR.h                    # Main Arduino API header
│   ├── WAMR.cpp                  # Arduino API implementation
│   └── wamr/                     # WAMR core sources
│       ├── build_config.h        # Build configuration flags
│       ├── WAMR_VERSION.txt      # WAMR version info
│       └── core/                 # WAMR source files
│           ├── iwasm/            # WASM interpreter and libraries
│           │   ├── include/      # Public WAMR headers
│           │   ├── common/       # Common runtime code
│           │   ├── interpreter/  # WASM interpreter
│           │   └── libraries/    # Built-in libraries
│           └── shared/           # Shared utilities
│               ├── platform/esp-idf/  # ESP32 platform layer
│               ├── mem-alloc/    # Memory allocators
│               └── utils/        # Utility functions
├── examples/                     # Arduino example sketches
│   ├── basic_wasm/              # Basic usage example
│   ├── native_functions/        # Native function export example
│   ├── memory_test/             # Memory profiling example
│   └── threading/               # Threading and API comparison example
├── docs/                        # Documentation
│   ├── BUILDING_WASM.md        # Guide to compiling WASM
│   ├── API_REFERENCE.md        # Complete API documentation
│   ├── TROUBLESHOOTING.md      # Common issues and solutions
│   └── SOURCE_FILES.md         # List of included WAMR files
├── tools/                       # Build and maintenance tools
│   ├── collect_sources.py      # Script to update WAMR sources
│   └── wasm_examples/          # Example WASM source code
├── library.json                 # PlatformIO library manifest
├── library.properties           # Arduino library manifest
└── README.md                    # Main documentation
```

## Architecture

### Arduino Wrapper Layer

**Files:** `src/WAMR.h`, `src/WAMR.cpp`

Provides two main classes:
- `WamrRuntime`: Static class for runtime initialization and management
- `WamrModule`: Instance class for loading and executing WASM modules

The wrapper handles:
- PSRAM detection and allocation
- Error handling and reporting
- Serial debugging output
- Arduino-style initialization
- **Pthread safety**: Automatic pthread wrapping for WASM execution (Arduino tasks are not pthreads)

### WAMR Core Layer

**Directory:** `src/wamr/core/`

Contains selected WAMR sources configured for minimal interpreter build:
- Fast interpreter only (no AOT, no JIT)
- libc-builtin only (no WASI)
- ESP-IDF platform layer
- Xtensa architecture support

### Build Configuration

**File:** `src/wamr/build_config.h`

Centralized configuration for WAMR features. Current settings:
- `BUILD_TARGET_XTENSA=1` - Xtensa architecture (ESP32/ESP32-S3)
- `WASM_ENABLE_FAST_INTERP=1` - Fast interpreter mode
- `WASM_ENABLE_LIBC_BUILTIN=1` - Built-in libc
- `WASM_ENABLE_AOT=0` - AOT disabled (can be enabled)
- `WASM_ENABLE_LIBC_WASI=0` - WASI disabled (can be enabled)

To add features: modify this file and ensure required source files are present.

## Build Commands

### Compiling the Library

**PlatformIO:**
```bash
# Build example for ESP32
pio ci --lib="." --board=esp32dev examples/basic_wasm/basic_wasm.ino

# Build example for ESP32-S3
pio ci --lib="." --board=esp32-s3-devkitc-1 examples/basic_wasm/basic_wasm.ino

# Build all examples
pio ci --lib="." --board=esp32dev examples/basic_wasm/basic_wasm.ino
pio ci --lib="." --board=esp32dev examples/native_functions/native_functions.ino
pio ci --lib="." --board=esp32dev examples/memory_test/memory_test.ino
pio ci --lib="." --board=esp32dev examples/threading/threading.ino

# Install library locally for testing
pio lib install file://.
```

**Arduino IDE:**
- Sketch → Verify/Compile (Ctrl+R)
- Tools → Board → Select ESP32/ESP32-S3 variant
- No separate test command available

### Compiling WASM Modules

**Using WASI SDK:**
```bash
# Install WASI SDK 21
wget https://github.com/WebAssembly/wasi-sdk/releases/download/wasi-sdk-21/wasi-sdk-21.0-linux.tar.gz
tar xf wasi-sdk-21.0-linux.tar.gz

# Compile C to standalone WASM (no WASI)
wasi-sdk-21.0/bin/clang --target=wasm32 -O3 \
    -nostdlib -Wl,--no-entry \
    -Wl,--export=function_name \
    -o output.wasm input.c

# Convert to C byte array for embedding
xxd -i output.wasm > output.wasm.h
```

See `tools/wasm_examples/` for example WASM source code and `docs/BUILDING_WASM.md` for details.

## Development Workflows

### Updating WAMR Sources

When updating to a newer WAMR version:

```bash
# From repository root
cd tools
python3 collect_sources.py /path/to/wamr-micro-runtime

# This will:
# 1. Copy required sources to src/wamr/core/
# 2. Generate docs/SOURCE_FILES.md
# 3. Write version to src/wamr/WAMR_VERSION.txt
```

The script (`tools/collect_sources.py`) knows which files are needed based on `SOURCE_PATTERNS` dictionary. Modify this if adding new features.

### Adding New Features

To enable additional WAMR features (e.g., AOT, WASI):

1. **Modify build configuration:**
   Edit `src/wamr/build_config.h` to enable feature flags

2. **Add required sources:**
   Update `tools/collect_sources.py` SOURCE_PATTERNS to include needed files

3. **Run collection script:**
   ```bash
   python3 tools/collect_sources.py /path/to/wamr
   ```

4. **Update library manifests:**
   Add any new build flags to `library.json` if needed

5. **Test thoroughly:**
   Test on real hardware with examples

6. **Update documentation:**
   Document new features in README.md and API_REFERENCE.md

### Testing Changes

**Compilation test:**
```bash
# Test all examples compile for ESP32
pio ci --board=esp32dev examples/basic_wasm/basic_wasm.ino
pio ci --board=esp32dev examples/native_functions/native_functions.ino
pio ci --board=esp32dev examples/memory_test/memory_test.ino
pio ci --board=esp32dev examples/threading/threading.ino

# Test for ESP32-S3
pio ci --board=esp32-s3-devkitc-1 examples/basic_wasm/basic_wasm.ino
```

**Hardware testing checklist:**
- Upload each example to ESP32/ESP32-S3
- Verify Serial output shows expected results
- Check memory usage is within limits (<200KB for basic example)
- Test error handling with invalid WASM
- Test load/unload cycle for memory leaks

## Common Development Tasks

### Adding a New Example

1. Create directory in `examples/`
2. Create `.ino` file with same name as directory
3. Include comprehensive comments
4. Test on hardware
5. Add to `library.json` examples list
6. Document in README.md

### Modifying Arduino API

**Files to modify:**
- `src/WAMR.h` - Add new methods/classes
- `src/WAMR.cpp` - Implement functionality
- `docs/API_REFERENCE.md` - Document new API
- Create example demonstrating usage

### Debugging Build Issues

**Common issues:**

1. **Missing source files:**
   - Check `docs/SOURCE_FILES.md` for list
   - Run `collect_sources.py` again
   - Verify file paths in error messages

2. **Undefined references:**
   - Feature enabled in `build_config.h` but source files missing
   - Check WAMR source file dependencies
   - May need additional source files

3. **Conflicting symbols:**
   - Usually BH_MALLOC/BH_FREE redefinition
   - Ensure flags set correctly in `library.json`

## Memory Management

### ESP32 Memory Layout

- **Internal RAM**: ~520KB SRAM, shared between heap and stack
- **PSRAM** (ESP32-S3): Up to 8MB external PSRAM
- **Flash**: Code storage

### WAMR Memory Usage

Three main memory pools:
1. **Runtime heap** (`WamrRuntime::begin(size)`): Global pool for WAMR runtime
2. **Module heap** (`module.load(..., heap_size)`): Per-module WASM linear memory
3. **Stack** (`module.load(..., stack_size, ...)`): Execution stack

### PSRAM Strategy

The wrapper (`WAMR.cpp`) automatically:
1. Tries to allocate from PSRAM (MALLOC_CAP_SPIRAM)
2. Falls back to internal RAM if PSRAM unavailable
3. Prints memory source to Serial

## Threading and Pthread Safety

### The Pthread Requirement Problem

**Critical concept:** WAMR requires execution to happen in a pthread context, but Arduino's main task (setup/loop) is NOT a pthread.

**Symptoms without pthread:**
```
assertion "pthread != NULL" failed: file "esp-idf/espidf_thread.c", line 123
Guru Meditation Error: Core 1 panic'ed (abort).
```

### Dual API Design

The library provides two APIs to solve this:

**1. Safe API (callFunction) - Recommended**
- Automatically creates pthread, executes WASM, returns result
- Safe to call from setup(), loop(), any Arduino code
- Overhead: ~100-500μs per call for pthread creation
- Use for: General use, <100 calls/second

**2. Raw API (callFunctionRaw) - Advanced**
- Direct WASM execution without pthread wrapper
- **MUST** be called from existing pthread context
- **WILL CRASH** if called from Arduino main task
- Zero pthread overhead
- Use for: High-frequency calls (>100 Hz), custom thread pools

### Implementation Details

**Files:** `src/WAMR.h`, `src/WAMR.cpp`

The implementation uses:
- `WasmCallContext` struct to pass data to pthread
- `wasm_thread_wrapper()` as pthread entry point
- `pthread_create()` and `pthread_join()` for synchronous execution
- Static `thread_stack_size` for configurable pthread stack

**Key methods:**
```cpp
// Public safe API - auto pthread wrapper
bool callFunction(const char* func_name, uint32_t argc, uint32_t* argv);

// Public raw API - must be in pthread
bool callFunctionRaw(const char* func_name, uint32_t argc, uint32_t* argv);

// Configure pthread stack size
static void setThreadStackSize(size_t stack_size);

// Private internal implementation
bool callFunctionInternal(const char* func_name, uint32_t argc, uint32_t* argv);
```

### Example Usage Patterns

**Pattern 1: Simple Arduino usage (safe API)**
```cpp
void loop() {
  uint32_t args[2] = {42, 58};
  module.callFunction("add", 2, args);  // Safe - auto pthread
  delay(100);
}
```

**Pattern 2: High-frequency calls (raw API)**
```cpp
void* worker_thread(void* arg) {
  WamrModule* module = (WamrModule*)arg;

  // In pthread context, use raw API
  for (int i = 0; i < 1000; i++) {
    uint32_t args[2] = {i, i+1};
    module->callFunctionRaw("add", 2, args);  // No pthread overhead
  }
  return nullptr;
}

void setup() {
  // ... init WAMR ...
  pthread_t thread;
  pthread_create(&thread, nullptr, worker_thread, &module);
  pthread_join(thread, nullptr);
}
```

**Pattern 3: Custom pthread attributes**
```cpp
pthread_t thread;
pthread_attr_t attr;
pthread_attr_init(&attr);
pthread_attr_setstacksize(&attr, 64 * 1024);  // Custom stack
pthread_create(&thread, &attr, worker_thread, &module);
pthread_join(thread, nullptr);
pthread_attr_destroy(&attr);
```

### Performance Considerations

**Benchmarks (ESP32 @ 240MHz):**
- `callFunction()`: ~100-500μs overhead (pthread creation)
- `callFunctionRaw()`: ~10-50μs overhead (function lookup only)
- Actual WASM execution: varies by function complexity

**When to use each API:**
- **Safe API**: Default choice, occasional calls, simplicity
- **Raw API**: >100 calls/sec, thread pool management, latency critical

See `examples/threading/` for comprehensive demonstrations.

## Platform Support

### Current: ESP32/ESP32-S3 (Xtensa)

**Architecture:** Xtensa dual-core
**Assembly:** `core/iwasm/common/arch/invokeNative_xtensa.s`
**Platform:** ESP-IDF layer compatible with Arduino-ESP32

### Planned: ESP32-C3/C6 (RISC-V)

To add RISC-V support:
1. Change `build_config.h`: `BUILD_TARGET_RISCV32=1`
2. Update `collect_sources.py` to include RISC-V assembly
3. Test native function calling
4. Update README.md platform support section

## Build System Integration

### PlatformIO

Library is discovered via:
- `library.json` manifest
- Build flags in `build.flags`
- Source filter in `build.srcFilter`

### Arduino IDE

Library is discovered via:
- `library.properties` manifest
- `keywords.txt` for syntax highlighting
- All sources in `src/` compiled automatically

## Documentation Maintenance

When making changes, update relevant docs:

- **README.md** - High-level changes, new features, compatibility
- **API_REFERENCE.md** - API changes, new methods, signatures
- **BUILDING_WASM.md** - Compilation changes, new requirements
- **TROUBLESHOOTING.md** - New issues, solutions, workarounds
- **CLAUDE.md** - Architecture changes, workflow updates

## Version Management and Release

**Version tracked in:**
- `library.json` - PlatformIO version
- `library.properties` - Arduino version
- `src/wamr/WAMR_VERSION.txt` - WAMR source version (auto-generated by collect_sources.py)

**Release process:**
1. Update version in `library.json` and `library.properties` (keep synchronized)
2. Test all examples on ESP32/ESP32-S3 hardware
3. Tag release: `git tag v1.0.0`
4. Push with tags: `git push --tags`
5. Create GitHub release with notes

## Useful References

- [WAMR GitHub](https://github.com/bytecodealliance/wasm-micro-runtime)
- [WAMR Documentation](https://wamr.gitbook.io/)
- [Arduino-ESP32](https://github.com/espressif/arduino-esp32)
- [PlatformIO Library Docs](https://docs.platformio.org/en/latest/librarymanager/)
- [WebAssembly Spec](https://webassembly.github.io/spec/)

## Key Files for Future Modifications

| Task | Files to Modify |
|------|-----------------|
| Add WAMR feature | `build_config.h`, `collect_sources.py` |
| Change API | `WAMR.h`, `WAMR.cpp`, `API_REFERENCE.md` |
| Add example | `examples/*/`, `library.json`, `README.md` |
| Fix platform issue | `WAMR.cpp`, platform-specific sources |
| Update WAMR | Run `collect_sources.py` |
| Add architecture | `build_config.h`, architecture-specific assembly |

## Code Style

- Follow existing Arduino-style naming (camelCase for methods)
- Use C++11 features (Arduino-ESP32 supports it)
- Keep Serial debugging optional but helpful
- Comment non-obvious WAMR API usage
- Provide examples for new functionality
