# XPT2046 Driver

<a href="./"><img alt="XPT2046 Driver Banner" src="./Assets/Banner.svg" style="width:100%"></a>

<p>
  <a href="https://github.com/median-dispersion/XPT2046-Driver/releases/latest"><img alt="GitHub Release" src="https://img.shields.io/github/v/release/median-dispersion/XPT2046-Driver?style=for-the-badge"></a>
  <a href="./LICENSE"><img alt="GitHub License" src="https://img.shields.io/github/license/median-dispersion/XPT2046-Driver?style=for-the-badge"></a>
</p>

An Arduino library for driving an XPT2046 touch controller. It can be used to get the X and Y positions of a touch event either as a raw 12-Bit value between 0 and 4095 or as a value mapped to a display's pixel grid using a calibration matrix generated by performing a 3-point calibration.

## ✨ Examples

### Example Sketch

This is a basic example showing all the functions of this library. For more detailed explanations, look at the included examples in `examples/`.

```c++
// Include the library
#include "XPT2046.h"

// Configuration
#define TOUCH_CS_PIN  2
#define TOUCH_IRQ_PIN 3
#define CALIBRATION   {0.0644197,-0.0003023,-11.8842955,-0.0002274,0.0898521,-14.7727499,240,320,0};

// Create the XPT2046 touch object
XPT2046 touch(TOUCH_CS_PIN, TOUCH_IRQ_PIN);

void setup() {
  
  Serial.begin(115200);              // Initialize serial communication
   
  touch.begin();                     // Initialize
  touch.setRotation(0);              // Set the rotation
  touch.setCalibration(CALIBRATION); // Set the calibration matrix
  touch.setSampleCount(20);          // Set the sample count
  touch.setDebounceTimeout(100);     // Set the debounce timeout
  touch.setTouchPressure(3.5);       // Set the touch pressure
  touch.setDeadZone(50);             // Set the dead zone
  touch.setPowerDown(true);          // Set the power-down state

}

void loop() {

  // Check if touch event is occurring and if the last event has been released
  if (touch.touched() && touch.released()) {

    // Get the touch position
    XPT2046::Point position = touch.getTouchPosition();

    // Check if the touch position is valid
    if (touch.valid(position)) {

      // Print the position to the Serial Monitor
      Serial.println("X: " + String(position.x) + ", Y: " + String(position.y));

    }

  }

}
```

### Basic Usage

The most basic example of this library can be found in the `examples/BasicUsage/BasicUsage.ino` sketch file. It initializes everything and then checks if a touch event occurred. If a touch event is captured, the touch position is printed to the Serial Monitor. The X and Y positions are a 12-Bit integer between 0 and 4095.

### Touchscreen Usage

The `examples/TouchscreenUsage/TouchscreenUsage.ino` is an example of how to use the XPT2046 Driver library with a touchscreen. The touchscreen used in this case is an ILI9341 TFT display driven by the Adafruit GFX & ILI9341 library. It maps the touch X and Y positions to the pixel grid of the ILI9341 TFT using a calibration matrix. The sketch initializes the touchscreen and display, then checks if a touch event occurred. It takes the touch position mapped to the pixel grid of the display and draws a crosshair at that point.

### 3 Point Calibration

The `examples/3PointCalibration/3PointCalibration.ino` sketch performs a 3-point calibration on a touchscreen that uses an XPT2046 touch controller. The touchscreen used in this case is an ILI9341 TFT display driven by the Adafruit GFX & ILI9341 library. The resulting calibration matrix can then be used in other sketches to calibrate and map the touch positions to the display's pixel grid.

## 🛠️ Library Functions

These are the functions this library provides. For more detailed examples on how to use these functions, look at the [Examples](#-examples) section.

Create an XPT2046 object.
```c++
XPT2046 touch(TOUCH_CS_PIN, TOUCH_IRQ_PIN);
```

Initialize the object.
```c++
touch.begin();
```

Set the touchscreen rotation (optional, default = 0, values = 0, 1, 2, 3, 0 = 0°, 1 = 90°, 2 = 180°, 3 = 270°). On some screens, one or both axes might be flipped, this depends on how that specific touchscreen was manufactured. The flipped axes should be fixed when using the calibration matrix generated by performing a 3-point calibration.
```c++
touch.setRotation(0);
```

Set the display calibration (optional, default = unset). This will map the touch position to the display's pixel grid, i.e., touchX = pixelX and touchY = pixelY. To get the calibration matrix, use the sketch: `examples/3PointCalibration/3PointCalibration.ino`.
```c++
touch.setCalibration({/*A*/, /*B*/, /*C*/, /*D*/, /*E*/, /*F*/, /*W*/, /*H*/, /*R*/});
```

Set the number of samples to average over before returning the touch position (optional, default = 20, range = 1-255). A higher number of samples will result in a more consistent touch position but, as a consequence, will take a longer time to process.
```c++
touch.setSampleCount(20);
```

Set the debounce timeout (optional, default = 100, time = milliseconds). This can help mitigate erroneous double-touch events when lifting from the touch area. Keep in mind that this is a **TIMEOUT** meaning if set to 1000 ms, only after waiting for 1 second another touch event is registered. For something like a paint program, this value should be as low as possible, or even set to 0.
```c++
touch.setDebounceTimeout(100);
```

Set the pressure at which a touch event will be registered (optional, default = 3.5). A lower value means a higher pressure needs to be applied before counting as a touch event. Too high of a value might make touch readings unstable because there is not enough pressure applied to make them settle down.
```c++
touchscreen.setTouchPressure(3.5);
```

Set the dead zone (optional, default = 50, range = 0-2047). All touch values below this value will be disregarded as invalid. Most touchscreens have a dead zone of around 200, so disregarding all values below that should be fine.
```c++
touchscreen.setDeadZone(50);
```

Set the power-down state (optional, default = true, values = true, false). Enabling this will send an extra command to the XPT2046 to power down its reference and ADCs after each SPI transaction. Reducing overall power draw but sending an extra 8 bits over the SPI bus.
```c++
touchscreen.setPowerDown(true);
```

Returns if the touch area is being touched (true / false).
```c++
touch.touched();
```

Returns if a touch event has been released, i.e., a pen or finger was lifted from the touch area. Can be used in combination with `touch.touched()` to check for single button presses.
```c++
touch.released();
```

Returns the X and Y positions of a touch event of type `XPT2046::Point`. If calibration is not set, this will be a 12-Bit integer between `0` and `4095`. If calibration is set, this will be a pixel position on the display. If the touch event was lifted during position acquisition, it can return a value of `65535` for both X and Y, indicating that there were no samples taken and the position is invalid.
```c++
touch.getTouchPosition();
```

Returns if a touch position is valid. The touch position can become invalid if the touch event was lifted before or during the sampling process. This will result in a value of `65535` for both X and Y, indicating that there were no samples taken and the position is invalid.
```c++
touch.valid(position);
```