///////////////////////////////////////////////////////////////////////////////////////////////////
// TestRainGauge.cpp
//
// CppUTest unit tests for RainGauge - real world test cases
// Test data from https://data.world/datagov-uk/37334c93-4584-452a-b0f0-1be9e22edacd -
// Pottery Fields rain gauge rainfall data
//
// https://github.com/matthias-bs/BresserWeatherSensorReceiver
//
//
// created: 09/2022
//
//
// MIT License
//
// Copyright (c) 2022 Matthias Prinke
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//
// History:
//
// 20220912 Created
//
// ToDo: 
// -
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include "CppUTest/TestHarness.h"

#define TOLERANCE 0.2
#include "RainGauge.h"

#if defined(_DEBUG_CIRCULAR_BUFFER_)
    #define DEBUG_CB() { rainGauge.printCircularBuffer(); }

#else
  #define DEBUG_CB() {}
#endif


/**
 * \example
 * struct tm tm;
 * time_t t;
 * strptime("6 Dec 2001 12:33:45", "%d %b %Y %H:%M:%S", &tm);
 * tm.tm_isdst = -1;      // Not set by strptime(); tells mktime()
 *                        // to determine whether daylight saving time
 *                        // is in effect
 * t = mktime(&tm);
 */

static void setTime(const char *time, tm &tm, time_t &ts)
{
  tm = {0};
  strptime(time, "%Y-%m-%d %H:%M", &tm);
  tm.tm_isdst = -1;
  ts = mktime(&tm);
}

TEST_GROUP(TestRainGaugePotteryFields) {
  void setup() {
  }

  void teardown() {
  }
};


/*
 * Test rainfall during past hour (no rain gauge overflow)
 */
TEST(TestRainGaugePotteryFields, Test_PotteryFields) {
  RainGauge rainGauge(100);
  rainGauge.reset();
  tm        tm;
  time_t    ts;

  printf("< PotteryFields >\n");
  // 2013-06-12 00:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 00:00", tm, ts);
  rainGauge.hist_init();
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 00:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 00:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 00:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 00:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 00:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 00:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 01:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 01:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 01:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 01:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 01:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 01:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 01:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 01:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 02:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 02:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 02:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 02:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 02:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 02:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 02:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 02:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 03:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 03:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 03:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 03:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 03:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 03:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 03:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 03:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 04:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 04:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 04:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 04:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 04:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 04:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 04:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 04:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 05:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 05:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 05:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 05:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 05:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 05:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 05:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 05:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 06:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 06:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 06:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 06:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 06:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 06:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 06:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 06:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 07:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 07:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 07:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 07:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 07:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 07:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 07:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 07:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 08:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 08:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 08:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 08:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 08:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 08:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 08:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 08:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 09:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 09:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 09:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 09:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 09:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 09:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 09:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 09:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 10:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 10:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 10:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 10:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 10:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 10:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 10:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 10:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 11:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 11:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 11:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 11:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 11:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 11:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 11:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 11:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 12:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 12:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 12:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 12:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 12:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 12:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 12:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 12:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 13:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 13:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 13:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 13:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 13:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 13:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 13:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 13:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 14:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 14:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 14:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 14:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 14:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 14:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 14:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 14:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 15:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 15:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 15:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 15:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 15:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 15:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 15:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 15:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 16:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 16:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 16:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 16:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 16:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 16:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 16:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 16:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 17:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 17:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 17:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 17:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 17:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 17:30", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 17:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 17:45", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 18:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 18:00", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 18:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-06-12 18:15", tm, ts);
  rainGauge.update(ts, 0);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 18:30 -> 0.2; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 18:30", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 18:45 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 18:45", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 19:00 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 19:00", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 19:15 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 19:15", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 19:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 19:30", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 19:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 19:45", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 20:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-06-12 20:00", tm, ts);
  rainGauge.update(ts, 0.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 20:15 -> 0.2; H: 0.2; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 20:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 20:30 -> 0; H: 0.2; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 20:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 20:45 -> 0; H: 0.2; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 20:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 21:00 -> 0; H: 0.2; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 21:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 21:15 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 21:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 21:30 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 21:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 21:45 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 21:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 22:00 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 22:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 22:15 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 22:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 22:30 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 22:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 22:45 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 22:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 23:00 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 23:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 23:15 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 23:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 23:30 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 23:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-12 23:45 -> 0; H: 0; D: 0.4; W: 0.4; M: 0.4
  setTime("2013-06-12 23:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 00:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 00:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 00:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 00:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 00:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 00:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 00:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 00:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 01:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 01:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 01:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 01:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 01:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 01:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 01:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 01:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 02:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 02:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 02:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 02:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 02:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 02:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 02:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 02:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 03:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 03:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 03:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 03:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 03:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 03:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 03:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 03:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 04:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 04:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 04:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 04:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 04:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 04:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 04:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 04:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 05:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 05:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 05:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 05:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 05:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 05:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 05:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 05:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 06:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 06:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 06:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 06:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 06:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 06:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 06:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 06:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 07:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 07:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 07:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 07:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 07:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 07:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 07:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 07:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 08:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 08:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 08:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 08:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 08:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 08:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 08:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 08:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 09:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 09:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 09:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 09:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 09:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 09:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 09:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 09:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 10:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 10:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 10:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 10:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 10:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 10:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 10:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 10:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 11:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 11:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 11:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 11:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 11:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 11:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 11:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 11:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 12:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 12:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 12:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 12:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 12:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 12:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 12:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 12:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 13:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 13:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 13:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 13:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 13:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 13:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 13:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 13:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 14:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 14:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 14:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 14:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 14:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 14:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 14:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 14:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 15:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 15:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 15:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 15:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 15:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 15:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 15:45 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 15:45", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 16:00 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 16:00", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 16:15 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 16:15", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 16:30 -> 0; H: 0; D: 0; W: 0.4; M: 0.4
  setTime("2013-06-13 16:30", tm, ts);
  rainGauge.update(ts, 0.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 16:45 -> 0.2; H: 0.2; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 16:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 17:00 -> 0; H: 0.2; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 17:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 17:15 -> 0; H: 0.2; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 17:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 17:30 -> 0; H: 0.2; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 17:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 17:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 17:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 18:00 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 18:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 18:15 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 18:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 18:30 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 18:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 18:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 18:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 19:00 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 19:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 19:15 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 19:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 19:30 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 19:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 19:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 19:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 20:00 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 20:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 20:15 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 20:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 20:30 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 20:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 20:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 20:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 21:00 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 21:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 21:15 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 21:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 21:30 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 21:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 21:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 21:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 22:00 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 22:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 22:15 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 22:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 22:30 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 22:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 22:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 22:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 23:00 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 23:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 23:15 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 23:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 23:30 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 23:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-13 23:45 -> 0; H: 0; D: 0.2; W: 0.6; M: 0.6
  setTime("2013-06-13 23:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 00:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 00:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 00:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 00:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 00:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 00:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 00:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 00:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 01:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 01:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 01:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 01:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 01:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 01:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 01:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 01:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 02:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 02:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 02:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 02:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 02:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 02:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 02:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 02:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 03:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 03:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 03:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 03:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 03:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 03:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 03:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 03:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 04:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 04:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 04:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 04:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 04:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 04:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 04:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 04:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 05:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 05:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 05:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 05:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 05:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 05:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 05:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 05:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 06:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 06:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 06:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 06:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 06:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 06:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 06:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 06:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 07:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 07:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 07:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 07:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 07:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 07:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 07:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 07:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 08:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 08:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 08:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 08:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 08:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 08:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 08:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 08:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 09:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 09:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 09:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 09:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 09:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 09:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 09:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 09:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 10:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 10:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 10:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 10:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 10:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 10:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 10:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 10:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 11:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 11:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 11:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 11:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 11:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 11:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 11:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 11:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 12:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 12:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 12:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 12:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 12:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 12:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 12:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 12:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 13:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 13:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 13:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 13:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 13:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 13:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 13:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 13:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 14:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 14:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 14:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 14:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 14:30 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 14:30", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 14:45 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 14:45", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 15:00 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 15:00", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 15:15 -> 0; H: 0; D: 0; W: 0.6; M: 0.6
  setTime("2013-06-14 15:15", tm, ts);
  rainGauge.update(ts, 0.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 15:30 -> 1.2; H: 1.2; D: 1.2; W: 1.8; M: 1.8
  setTime("2013-06-14 15:30", tm, ts);
  rainGauge.update(ts, 1.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 15:45 -> 0.4; H: 1.6; D: 1.6; W: 2.2; M: 2.2
  setTime("2013-06-14 15:45", tm, ts);
  rainGauge.update(ts, 2.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 16:00 -> 0.2; H: 1.8; D: 1.8; W: 2.4; M: 2.4
  setTime("2013-06-14 16:00", tm, ts);
  rainGauge.update(ts, 2.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 16:15 -> 0; H: 1.8; D: 1.8; W: 2.4; M: 2.4
  setTime("2013-06-14 16:15", tm, ts);
  rainGauge.update(ts, 2.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 16:30 -> 0; H: 0.6; D: 1.8; W: 2.4; M: 2.4
  setTime("2013-06-14 16:30", tm, ts);
  rainGauge.update(ts, 2.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 16:45 -> 0; H: 0.2; D: 1.8; W: 2.4; M: 2.4
  setTime("2013-06-14 16:45", tm, ts);
  rainGauge.update(ts, 2.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 17:00 -> 0.8; H: 0.8; D: 2.6; W: 3.2; M: 3.2
  setTime("2013-06-14 17:00", tm, ts);
  rainGauge.update(ts, 3.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 17:15 -> 0.2; H: 1; D: 2.8; W: 3.4; M: 3.4
  setTime("2013-06-14 17:15", tm, ts);
  rainGauge.update(ts, 3.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 17:30 -> 0; H: 1; D: 2.8; W: 3.4; M: 3.4
  setTime("2013-06-14 17:30", tm, ts);
  rainGauge.update(ts, 3.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 17:45 -> 0.8; H: 1.8; D: 3.6; W: 4.2; M: 4.2
  setTime("2013-06-14 17:45", tm, ts);
  rainGauge.update(ts, 4.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 18:00 -> 0; H: 1; D: 3.6; W: 4.2; M: 4.2
  setTime("2013-06-14 18:00", tm, ts);
  rainGauge.update(ts, 4.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 18:15 -> 0.2; H: 1; D: 3.8; W: 4.4; M: 4.4
  setTime("2013-06-14 18:15", tm, ts);
  rainGauge.update(ts, 4.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 18:30 -> 0; H: 1; D: 3.8; W: 4.4; M: 4.4
  setTime("2013-06-14 18:30", tm, ts);
  rainGauge.update(ts, 4.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 18:45 -> 0.4; H: 0.6; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 18:45", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 19:00 -> 0; H: 0.6; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 19:00", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 19:15 -> 0; H: 0.4; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 19:15", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 19:30 -> 0; H: 0.4; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 19:30", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 19:45 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 19:45", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 20:00 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 20:00", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 20:15 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 20:15", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 20:30 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 20:30", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 20:45 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 20:45", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 21:00 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 21:00", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 21:15 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 21:15", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 21:30 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 21:30", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 21:45 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 21:45", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 22:00 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 22:00", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 22:15 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 22:15", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 22:30 -> 0; H: 0; D: 4.2; W: 4.8; M: 4.8
  setTime("2013-06-14 22:30", tm, ts);
  rainGauge.update(ts, 4.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 22:45 -> 0.2; H: 0.2; D: 4.4; W: 5; M: 5
  setTime("2013-06-14 22:45", tm, ts);
  rainGauge.update(ts, 5);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 23:00 -> 0; H: 0.2; D: 4.4; W: 5; M: 5
  setTime("2013-06-14 23:00", tm, ts);
  rainGauge.update(ts, 5);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 23:15 -> 0; H: 0.2; D: 4.4; W: 5; M: 5
  setTime("2013-06-14 23:15", tm, ts);
  rainGauge.update(ts, 5);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 23:30 -> 0.2; H: 0.4; D: 4.6; W: 5.2; M: 5.2
  setTime("2013-06-14 23:30", tm, ts);
  rainGauge.update(ts, 5.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-14 23:45 -> 0; H: 0.2; D: 4.6; W: 5.2; M: 5.2
  setTime("2013-06-14 23:45", tm, ts);
  rainGauge.update(ts, 5.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 00:00 -> 0; H: 0.2; D: 0; W: 5.2; M: 5.2
  setTime("2013-06-15 00:00", tm, ts);
  rainGauge.update(ts, 5.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 00:15 -> 0; H: 0.2; D: 0; W: 5.2; M: 5.2
  setTime("2013-06-15 00:15", tm, ts);
  rainGauge.update(ts, 5.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 00:30 -> 0; H: 0; D: 0; W: 5.2; M: 5.2
  setTime("2013-06-15 00:30", tm, ts);
  rainGauge.update(ts, 5.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 00:45 -> 0.2; H: 0.2; D: 0.2; W: 5.4; M: 5.4
  setTime("2013-06-15 00:45", tm, ts);
  rainGauge.update(ts, 5.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 01:00 -> 0.2; H: 0.4; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 01:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 01:15 -> 0; H: 0.4; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 01:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 01:30 -> 0; H: 0.4; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 01:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 01:45 -> 0; H: 0.2; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 01:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 02:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 02:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 02:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 02:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 02:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 02:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 02:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 02:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 03:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 03:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 03:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 03:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 03:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 03:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 03:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 03:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 04:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 04:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 04:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 04:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 04:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 04:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 04:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 04:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 05:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 05:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 05:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 05:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 05:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 05:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 05:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 05:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 06:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 06:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 06:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 06:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 06:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 06:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 06:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 06:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 07:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 07:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 07:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 07:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 07:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 07:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 07:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 07:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 08:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 08:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 08:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 08:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 08:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 08:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 08:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 08:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 09:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 09:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 09:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 09:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 09:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 09:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 09:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 09:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 10:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 10:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 10:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 10:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 10:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 10:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 10:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 10:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 11:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 11:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 11:15 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 11:15", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 11:30 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 11:30", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 11:45 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 11:45", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 12:00 -> 0; H: 0; D: 0.4; W: 5.6; M: 5.6
  setTime("2013-06-15 12:00", tm, ts);
  rainGauge.update(ts, 5.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 12:15 -> 0.2; H: 0.2; D: 0.6; W: 5.8; M: 5.8
  setTime("2013-06-15 12:15", tm, ts);
  rainGauge.update(ts, 5.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 12:30 -> 0; H: 0.2; D: 0.6; W: 5.8; M: 5.8
  setTime("2013-06-15 12:30", tm, ts);
  rainGauge.update(ts, 5.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 12:45 -> 0; H: 0.2; D: 0.6; W: 5.8; M: 5.8
  setTime("2013-06-15 12:45", tm, ts);
  rainGauge.update(ts, 5.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 13:00 -> 0.8; H: 1; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 13:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 13:15 -> 0; H: 0.8; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 13:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 13:30 -> 0; H: 0.8; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 13:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 13:45 -> 0; H: 0.8; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 13:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 14:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 14:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 14:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 14:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 14:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 14:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 14:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 14:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 15:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 15:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 15:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 15:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 15:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 15:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 15:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 15:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 16:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 16:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 16:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 16:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 16:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 16:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 16:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 16:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 17:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 17:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 17:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 17:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 17:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 17:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 17:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 17:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 18:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 18:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 18:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 18:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 18:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 18:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 18:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 18:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 19:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 19:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 19:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 19:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 19:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 19:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 19:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 19:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 20:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 20:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 20:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 20:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 20:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 20:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 20:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 20:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 21:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 21:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 21:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 21:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 21:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 21:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 21:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 21:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 22:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 22:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 22:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 22:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 22:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 22:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 22:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 22:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 23:00 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 23:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 23:15 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 23:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 23:30 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 23:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-15 23:45 -> 0; H: 0; D: 1.4; W: 6.6; M: 6.6
  setTime("2013-06-15 23:45", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 00:00 -> 0; H: 0; D: 0; W: 6.6; M: 6.6
  setTime("2013-06-16 00:00", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 00:15 -> 0; H: 0; D: 0; W: 6.6; M: 6.6
  setTime("2013-06-16 00:15", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 00:30 -> 0; H: 0; D: 0; W: 6.6; M: 6.6
  setTime("2013-06-16 00:30", tm, ts);
  rainGauge.update(ts, 6.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    6.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 00:45 -> 1.2; H: 1.2; D: 1.2; W: 7.8; M: 7.8
  setTime("2013-06-16 00:45", tm, ts);
  rainGauge.update(ts, 7.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    7.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    7.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 01:00 -> 0; H: 1.2; D: 1.2; W: 7.8; M: 7.8
  setTime("2013-06-16 01:00", tm, ts);
  rainGauge.update(ts, 7.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    7.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    7.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 01:15 -> 0.2; H: 1.4; D: 1.4; W: 8; M: 8
  setTime("2013-06-16 01:15", tm, ts);
  rainGauge.update(ts, 8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 01:30 -> 0.4; H: 1.8; D: 1.8; W: 8.4; M: 8.4
  setTime("2013-06-16 01:30", tm, ts);
  rainGauge.update(ts, 8.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 01:45 -> 0.2; H: 0.8; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 01:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 02:00 -> 0; H: 0.8; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 02:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 02:15 -> 0; H: 0.6; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 02:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 02:30 -> 0; H: 0.2; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 02:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 02:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 02:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 03:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 03:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 03:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 03:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 03:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 03:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 03:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 03:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 04:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 04:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 04:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 04:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 04:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 04:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 04:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 04:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 05:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 05:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 05:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 05:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 05:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 05:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 05:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 05:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 06:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 06:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 06:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 06:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 06:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 06:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 06:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 06:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 07:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 07:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 07:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 07:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 07:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 07:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 07:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 07:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 08:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 08:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 08:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 08:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 08:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 08:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 08:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 08:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 09:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 09:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 09:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 09:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 09:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 09:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 09:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 09:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 10:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 10:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 10:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 10:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 10:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 10:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 10:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 10:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 11:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 11:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 11:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 11:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 11:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 11:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 11:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 11:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 12:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 12:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 12:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 12:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 12:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 12:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 12:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 12:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 13:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 13:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 13:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 13:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 13:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 13:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 13:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 13:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 14:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 14:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 14:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 14:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 14:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 14:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 14:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 14:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 15:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 15:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 15:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 15:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 15:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 15:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 15:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 15:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 16:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 16:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 16:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 16:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 16:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 16:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 16:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 16:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 17:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 17:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 17:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 17:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 17:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 17:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 17:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 17:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 18:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 18:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 18:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 18:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 18:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 18:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 18:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 18:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 19:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 19:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 19:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 19:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 19:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 19:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 19:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 19:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 20:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 20:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 20:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 20:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 20:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 20:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 20:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 20:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 21:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 21:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 21:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 21:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 21:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 21:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 21:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 21:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 22:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 22:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 22:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 22:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 22:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 22:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 22:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 22:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 23:00 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 23:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 23:15 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 23:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 23:30 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 23:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-16 23:45 -> 0; H: 0; D: 2; W: 8.6; M: 8.6
  setTime("2013-06-16 23:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 00:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 00:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 00:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 00:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 00:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 00:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 00:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 00:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 01:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 01:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 01:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 01:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 01:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 01:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 01:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 01:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 02:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 02:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 02:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 02:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 02:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 02:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 02:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 02:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 03:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 03:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 03:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 03:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 03:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 03:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 03:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 03:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 04:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 04:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 04:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 04:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 04:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 04:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 04:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 04:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 05:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 05:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 05:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 05:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 05:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 05:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 05:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 05:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 06:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 06:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 06:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 06:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 06:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 06:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 06:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 06:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 07:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 07:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 07:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 07:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 07:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 07:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 07:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 07:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 08:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 08:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 08:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 08:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 08:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 08:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 08:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 08:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 09:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 09:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 09:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 09:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 09:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 09:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 09:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 09:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 10:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 10:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 10:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 10:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 10:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 10:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 10:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 10:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 11:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 11:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 11:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 11:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 11:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 11:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 11:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 11:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 12:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 12:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 12:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 12:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 12:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 12:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 12:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 12:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 13:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 13:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 13:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 13:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 13:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 13:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 13:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 13:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 14:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 14:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 14:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 14:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 14:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 14:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 14:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 14:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 15:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 15:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 15:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 15:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 15:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 15:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 15:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 15:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 16:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 16:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 16:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 16:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 16:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 16:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 16:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 16:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 17:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 17:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 17:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 17:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 17:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 17:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 17:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 17:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 18:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 18:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 18:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 18:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 18:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 18:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 18:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 18:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 19:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 19:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 19:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 19:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 19:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 19:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 19:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 19:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 20:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 20:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 20:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 20:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 20:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 20:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 20:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 20:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 21:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 21:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 21:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 21:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 21:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 21:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 21:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 21:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 22:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 22:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 22:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 22:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 22:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 22:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 22:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 22:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 23:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 23:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 23:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 23:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 23:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 23:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-17 23:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-17 23:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 00:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 00:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 00:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 00:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 00:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 00:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 00:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 00:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 01:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 01:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 01:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 01:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 01:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 01:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 01:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 01:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 02:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 02:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 02:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 02:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 02:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 02:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 02:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 02:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 03:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 03:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 03:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 03:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 03:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 03:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 03:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 03:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 04:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 04:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 04:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 04:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 04:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 04:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 04:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 04:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 05:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 05:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 05:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 05:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 05:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 05:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 05:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 05:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 06:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 06:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 06:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 06:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 06:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 06:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 06:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 06:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 07:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 07:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 07:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 07:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 07:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 07:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 07:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 07:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 08:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 08:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 08:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 08:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 08:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 08:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 08:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 08:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 09:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 09:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 09:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 09:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 09:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 09:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 09:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 09:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 10:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 10:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 10:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 10:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 10:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 10:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 10:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 10:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 11:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 11:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 11:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 11:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 11:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 11:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 11:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 11:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 12:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 12:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 12:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 12:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 12:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 12:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 12:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 12:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 13:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 13:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 13:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 13:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 13:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 13:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 13:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 13:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 14:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 14:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 14:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 14:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 14:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 14:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 14:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 14:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 15:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 15:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 15:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 15:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 15:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 15:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 15:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 15:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 16:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 16:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 16:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 16:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 16:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 16:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 16:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 16:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 17:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 17:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 17:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 17:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 17:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 17:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 17:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 17:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 18:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 18:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 18:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 18:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 18:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 18:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 18:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 18:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 19:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 19:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 19:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 19:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 19:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 19:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 19:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 19:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 20:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 20:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 20:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 20:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 20:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 20:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 20:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 20:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 21:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 21:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 21:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 21:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 21:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 21:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 21:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 21:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 22:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 22:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 22:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 22:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 22:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 22:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 22:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 22:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 23:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 23:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 23:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 23:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 23:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 23:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-18 23:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-18 23:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 00:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 00:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 00:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 00:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 00:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 00:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 00:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 00:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 01:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 01:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 01:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 01:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 01:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 01:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 01:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 01:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 02:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 02:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 02:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 02:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 02:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 02:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 02:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 02:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 03:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 03:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 03:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 03:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 03:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 03:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 03:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 03:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 04:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 04:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 04:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 04:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 04:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 04:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 04:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 04:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 05:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 05:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 05:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 05:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 05:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 05:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 05:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 05:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 06:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 06:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 06:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 06:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 06:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 06:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 06:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 06:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 07:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 07:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 07:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 07:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 07:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 07:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 07:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 07:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 08:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 08:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 08:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 08:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 08:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 08:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 08:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 08:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 09:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 09:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 09:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 09:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 09:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 09:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 09:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 09:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 10:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 10:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 10:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 10:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 10:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 10:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 10:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 10:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 11:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 11:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 11:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 11:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 11:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 11:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 11:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 11:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 12:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 12:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 12:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 12:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 12:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 12:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 12:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 12:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 13:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 13:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 13:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 13:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 13:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 13:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 13:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 13:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 14:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 14:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 14:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 14:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 14:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 14:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 14:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 14:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 15:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 15:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 15:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 15:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 15:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 15:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 15:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 15:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 16:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 16:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 16:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 16:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 16:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 16:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 16:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 16:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 17:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 17:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 17:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 17:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 17:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 17:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 17:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 17:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 18:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 18:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 18:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 18:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 18:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 18:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 18:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 18:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 19:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 19:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 19:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 19:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 19:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 19:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 19:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 19:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 20:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 20:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 20:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 20:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 20:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 20:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 20:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 20:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 21:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 21:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 21:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 21:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 21:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 21:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 21:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 21:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 22:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 22:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 22:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 22:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 22:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 22:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 22:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 22:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 23:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 23:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 23:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 23:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 23:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 23:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-19 23:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-19 23:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 00:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 00:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 00:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 00:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 00:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 00:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 00:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 00:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 01:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 01:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 01:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 01:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 01:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 01:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 01:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 01:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 02:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 02:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 02:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 02:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 02:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 02:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 02:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 02:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 03:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 03:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 03:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 03:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 03:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 03:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 03:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 03:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 04:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 04:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 04:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 04:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 04:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 04:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 04:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 04:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 05:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 05:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 05:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 05:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 05:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 05:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 05:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 05:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 06:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 06:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 06:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 06:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 06:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 06:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 06:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 06:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 07:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 07:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 07:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 07:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 07:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 07:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 07:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 07:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 08:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 08:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 08:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 08:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 08:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 08:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 08:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 08:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 09:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 09:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 09:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 09:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 09:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 09:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 09:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 09:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 10:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 10:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 10:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 10:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 10:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 10:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 10:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 10:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 11:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 11:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 11:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 11:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 11:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 11:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 11:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 11:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 12:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 12:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 12:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 12:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 12:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 12:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 12:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 12:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 13:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 13:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 13:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 13:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 13:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 13:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 13:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 13:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 14:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 14:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 14:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 14:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 14:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 14:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 14:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 14:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 15:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 15:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 15:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 15:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 15:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 15:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 15:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 15:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 16:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 16:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 16:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 16:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 16:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 16:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 16:45 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 16:45", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 17:00 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 17:00", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 17:15 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 17:15", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 17:30 -> 0; H: 0; D: 0; W: 0; M: 8.6
  setTime("2013-06-20 17:30", tm, ts);
  rainGauge.update(ts, 8.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 17:45 -> 0.2; H: 0.2; D: 0.2; W: 0.2; M: 8.8
  setTime("2013-06-20 17:45", tm, ts);
  rainGauge.update(ts, 8.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 18:00 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 8.8
  setTime("2013-06-20 18:00", tm, ts);
  rainGauge.update(ts, 8.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 18:15 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 8.8
  setTime("2013-06-20 18:15", tm, ts);
  rainGauge.update(ts, 8.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 18:30 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 8.8
  setTime("2013-06-20 18:30", tm, ts);
  rainGauge.update(ts, 8.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    8.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 18:45 -> 0.2; H: 0.2; D: 0.4; W: 0.4; M: 9
  setTime("2013-06-20 18:45", tm, ts);
  rainGauge.update(ts, 9);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    9.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 19:00 -> 0; H: 0.2; D: 0.4; W: 0.4; M: 9
  setTime("2013-06-20 19:00", tm, ts);
  rainGauge.update(ts, 9);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    9.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 19:15 -> 0.2; H: 0.4; D: 0.6; W: 0.6; M: 9.2
  setTime("2013-06-20 19:15", tm, ts);
  rainGauge.update(ts, 9.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    9.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 19:30 -> 0.4; H: 0.8; D: 1; W: 1; M: 9.6
  setTime("2013-06-20 19:30", tm, ts);
  rainGauge.update(ts, 9.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 19:45 -> 0.6; H: 1.2; D: 1.6; W: 1.6; M: 10.2
  setTime("2013-06-20 19:45", tm, ts);
  rainGauge.update(ts, 10.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   10.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 20:00 -> 0.4; H: 1.6; D: 2; W: 2; M: 10.6
  setTime("2013-06-20 20:00", tm, ts);
  rainGauge.update(ts, 10.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   10.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 20:15 -> 0.4; H: 1.8; D: 2.4; W: 2.4; M: 11
  setTime("2013-06-20 20:15", tm, ts);
  rainGauge.update(ts, 11);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   11.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 20:30 -> 0.2; H: 1.6; D: 2.6; W: 2.6; M: 11.2
  setTime("2013-06-20 20:30", tm, ts);
  rainGauge.update(ts, 11.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   11.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 20:45 -> 0.4; H: 1.4; D: 3; W: 3; M: 11.6
  setTime("2013-06-20 20:45", tm, ts);
  rainGauge.update(ts, 11.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   11.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 21:00 -> 0.4; H: 1.4; D: 3.4; W: 3.4; M: 12
  setTime("2013-06-20 21:00", tm, ts);
  rainGauge.update(ts, 12);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   12.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 21:15 -> 0.2; H: 1.2; D: 3.6; W: 3.6; M: 12.2
  setTime("2013-06-20 21:15", tm, ts);
  rainGauge.update(ts, 12.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 21:30 -> 0; H: 1; D: 3.6; W: 3.6; M: 12.2
  setTime("2013-06-20 21:30", tm, ts);
  rainGauge.update(ts, 12.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 21:45 -> 0.2; H: 0.8; D: 3.8; W: 3.8; M: 12.4
  setTime("2013-06-20 21:45", tm, ts);
  rainGauge.update(ts, 12.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   12.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 22:00 -> 1; H: 1.4; D: 4.8; W: 4.8; M: 13.4
  setTime("2013-06-20 22:00", tm, ts);
  rainGauge.update(ts, 13.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 22:15 -> 0; H: 1.2; D: 4.8; W: 4.8; M: 13.4
  setTime("2013-06-20 22:15", tm, ts);
  rainGauge.update(ts, 13.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 22:30 -> 0; H: 1.2; D: 4.8; W: 4.8; M: 13.4
  setTime("2013-06-20 22:30", tm, ts);
  rainGauge.update(ts, 13.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 22:45 -> 0; H: 1; D: 4.8; W: 4.8; M: 13.4
  setTime("2013-06-20 22:45", tm, ts);
  rainGauge.update(ts, 13.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 23:00 -> 0.2; H: 0.2; D: 5; W: 5; M: 13.6
  setTime("2013-06-20 23:00", tm, ts);
  rainGauge.update(ts, 13.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 23:15 -> 0; H: 0.2; D: 5; W: 5; M: 13.6
  setTime("2013-06-20 23:15", tm, ts);
  rainGauge.update(ts, 13.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 23:30 -> 0.2; H: 0.4; D: 5.2; W: 5.2; M: 13.8
  setTime("2013-06-20 23:30", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-20 23:45 -> 0; H: 0.4; D: 5.2; W: 5.2; M: 13.8
  setTime("2013-06-20 23:45", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 00:00 -> 0; H: 0.2; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 00:00", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 00:15 -> 0; H: 0.2; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 00:15", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 00:30 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 00:30", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 00:45 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 00:45", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 01:00 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 01:00", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 01:15 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 01:15", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 01:30 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 01:30", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 01:45 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 01:45", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 02:00 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 02:00", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 02:15 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 02:15", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 02:30 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 02:30", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 02:45 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 02:45", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 03:00 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 03:00", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 03:15 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 03:15", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 03:30 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 03:30", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 03:45 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 03:45", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 04:00 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 04:00", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 04:15 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 04:15", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 04:30 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 04:30", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 04:45 -> 0; H: 0; D: 0; W: 5.2; M: 13.8
  setTime("2013-06-21 04:45", tm, ts);
  rainGauge.update(ts, 13.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   13.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 05:00 -> 0.2; H: 0.2; D: 0.2; W: 5.4; M: 14
  setTime("2013-06-21 05:00", tm, ts);
  rainGauge.update(ts, 14);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 05:15 -> 0; H: 0.2; D: 0.2; W: 5.4; M: 14
  setTime("2013-06-21 05:15", tm, ts);
  rainGauge.update(ts, 14);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 05:30 -> 0.2; H: 0.4; D: 0.4; W: 5.6; M: 14.2
  setTime("2013-06-21 05:30", tm, ts);
  rainGauge.update(ts, 14.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 05:45 -> 0.2; H: 0.6; D: 0.6; W: 5.8; M: 14.4
  setTime("2013-06-21 05:45", tm, ts);
  rainGauge.update(ts, 14.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 06:00 -> 0.2; H: 0.6; D: 0.8; W: 6; M: 14.6
  setTime("2013-06-21 06:00", tm, ts);
  rainGauge.update(ts, 14.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 06:15 -> 0.2; H: 0.8; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 06:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 06:30 -> 0; H: 0.6; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 06:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 06:45 -> 0; H: 0.4; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 06:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 07:00 -> 0; H: 0.2; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 07:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 07:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 07:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 07:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 07:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 07:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 07:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 08:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 08:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 08:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 08:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 08:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 08:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 08:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 08:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 09:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 09:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 09:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 09:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 09:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 09:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 09:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 09:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 10:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 10:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 10:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 10:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 10:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 10:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 10:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 10:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 11:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 11:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 11:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 11:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 11:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 11:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 11:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 11:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 12:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 12:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 12:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 12:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 12:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 12:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 12:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 12:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 13:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 13:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 13:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 13:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 13:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 13:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 13:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 13:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 14:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 14:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 14:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 14:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 14:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 14:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 14:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 14:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 15:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 15:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 15:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 15:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 15:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 15:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 15:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 15:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 16:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 16:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 16:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 16:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 16:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 16:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 16:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 16:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 17:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 17:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 17:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 17:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 17:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 17:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 17:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 17:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 18:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 18:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 18:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 18:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 18:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 18:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 18:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 18:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 19:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 19:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 19:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 19:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 19:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 19:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 19:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 19:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 20:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 20:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 20:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 20:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 20:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 20:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 20:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 20:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 21:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 21:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 21:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 21:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 21:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 21:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 21:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 21:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 22:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 22:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 22:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 22:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 22:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 22:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 22:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 22:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 23:00 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 23:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 23:15 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 23:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 23:30 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 23:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-21 23:45 -> 0; H: 0; D: 1; W: 6.2; M: 14.8
  setTime("2013-06-21 23:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 00:00 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 00:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 00:15 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 00:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 00:30 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 00:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 00:45 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 00:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 01:00 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 01:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 01:15 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 01:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 01:30 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 01:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 01:45 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 01:45", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 02:00 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 02:00", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 02:15 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 02:15", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 02:30 -> 0; H: 0; D: 0; W: 6.2; M: 14.8
  setTime("2013-06-22 02:30", tm, ts);
  rainGauge.update(ts, 14.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   14.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 02:45 -> 0.6; H: 0.6; D: 0.6; W: 6.8; M: 15.4
  setTime("2013-06-22 02:45", tm, ts);
  rainGauge.update(ts, 15.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   15.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 03:00 -> 0.6; H: 1.2; D: 1.2; W: 7.4; M: 16
  setTime("2013-06-22 03:00", tm, ts);
  rainGauge.update(ts, 16);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    7.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   16.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 03:15 -> 1; H: 2.2; D: 2.2; W: 8.4; M: 17
  setTime("2013-06-22 03:15", tm, ts);
  rainGauge.update(ts, 17);
  DEBUG_CB();
  DOUBLES_EQUAL(    2.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    8.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   17.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 03:30 -> 0.8; H: 3; D: 3; W: 9.2; M: 17.8
  setTime("2013-06-22 03:30", tm, ts);
  rainGauge.update(ts, 17.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    3.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   17.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 03:45 -> 0; H: 2.4; D: 3; W: 9.2; M: 17.8
  setTime("2013-06-22 03:45", tm, ts);
  rainGauge.update(ts, 17.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    2.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   17.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 04:00 -> 0.2; H: 2; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 04:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    2.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 04:15 -> 0; H: 1; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 04:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 04:30 -> 0; H: 0.2; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 04:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 04:45 -> 0; H: 0.2; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 04:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 05:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 05:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 05:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 05:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 05:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 05:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 05:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 05:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 06:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 06:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 06:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 06:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 06:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 06:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 06:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 06:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 07:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 07:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 07:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 07:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 07:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 07:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 07:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 07:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 08:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 08:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 08:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 08:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 08:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 08:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 08:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 08:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 09:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 09:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 09:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 09:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 09:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 09:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 09:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 09:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 10:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 10:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 10:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 10:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 10:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 10:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 10:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 10:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 11:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 11:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 11:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 11:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 11:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 11:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 11:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 11:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 12:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 12:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 12:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 12:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 12:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 12:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 12:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 12:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 13:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 13:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 13:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 13:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 13:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 13:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 13:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 13:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 14:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 14:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 14:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 14:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 14:30 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 14:30", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 14:45 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 14:45", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 15:00 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 15:00", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 15:15 -> 0; H: 0; D: 3.2; W: 9.4; M: 18
  setTime("2013-06-22 15:15", tm, ts);
  rainGauge.update(ts, 18);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 15:30 -> 0.2; H: 0.2; D: 3.4; W: 9.6; M: 18.2
  setTime("2013-06-22 15:30", tm, ts);
  rainGauge.update(ts, 18.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 15:45 -> 0; H: 0.2; D: 3.4; W: 9.6; M: 18.2
  setTime("2013-06-22 15:45", tm, ts);
  rainGauge.update(ts, 18.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 16:00 -> 0; H: 0.2; D: 3.4; W: 9.6; M: 18.2
  setTime("2013-06-22 16:00", tm, ts);
  rainGauge.update(ts, 18.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 16:15 -> 0; H: 0.2; D: 3.4; W: 9.6; M: 18.2
  setTime("2013-06-22 16:15", tm, ts);
  rainGauge.update(ts, 18.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 16:30 -> 0; H: 0; D: 3.4; W: 9.6; M: 18.2
  setTime("2013-06-22 16:30", tm, ts);
  rainGauge.update(ts, 18.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 16:45 -> 0; H: 0; D: 3.4; W: 9.6; M: 18.2
  setTime("2013-06-22 16:45", tm, ts);
  rainGauge.update(ts, 18.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 17:00 -> 0.2; H: 0.2; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 17:00", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 17:15 -> 0; H: 0.2; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 17:15", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 17:30 -> 0; H: 0.2; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 17:30", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 17:45 -> 0; H: 0.2; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 17:45", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 18:00 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 18:00", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 18:15 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 18:15", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 18:30 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 18:30", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 18:45 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 18:45", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 19:00 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 19:00", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 19:15 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 19:15", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 19:30 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 19:30", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 19:45 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 19:45", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 20:00 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 20:00", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 20:15 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 20:15", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 20:30 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 20:30", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 20:45 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 20:45", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 21:00 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 21:00", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 21:15 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 21:15", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 21:30 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 21:30", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 21:45 -> 0; H: 0; D: 3.6; W: 9.8; M: 18.4
  setTime("2013-06-22 21:45", tm, ts);
  rainGauge.update(ts, 18.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    9.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 22:00 -> 0.4; H: 0.4; D: 4; W: 10.2; M: 18.8
  setTime("2013-06-22 22:00", tm, ts);
  rainGauge.update(ts, 18.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   10.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   18.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 22:15 -> 0.4; H: 0.8; D: 4.4; W: 10.6; M: 19.2
  setTime("2013-06-22 22:15", tm, ts);
  rainGauge.update(ts, 19.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   10.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   19.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 22:30 -> 0.8; H: 1.6; D: 5.2; W: 11.4; M: 20
  setTime("2013-06-22 22:30", tm, ts);
  rainGauge.update(ts, 20);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   11.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 22:45 -> 0.2; H: 1.8; D: 5.4; W: 11.6; M: 20.2
  setTime("2013-06-22 22:45", tm, ts);
  rainGauge.update(ts, 20.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   11.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 23:00 -> 0; H: 1.4; D: 5.4; W: 11.6; M: 20.2
  setTime("2013-06-22 23:00", tm, ts);
  rainGauge.update(ts, 20.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   11.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 23:15 -> 0; H: 1; D: 5.4; W: 11.6; M: 20.2
  setTime("2013-06-22 23:15", tm, ts);
  rainGauge.update(ts, 20.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   11.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 23:30 -> 0.4; H: 0.6; D: 5.8; W: 12; M: 20.6
  setTime("2013-06-22 23:30", tm, ts);
  rainGauge.update(ts, 20.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-22 23:45 -> 0; H: 0.4; D: 5.8; W: 12; M: 20.6
  setTime("2013-06-22 23:45", tm, ts);
  rainGauge.update(ts, 20.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 00:00 -> 0.2; H: 0.6; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 00:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 00:15 -> 0; H: 0.6; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 00:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 00:30 -> 0; H: 0.2; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 00:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 00:45 -> 0; H: 0.2; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 00:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 01:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 01:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 01:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 01:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 01:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 01:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 01:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 01:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 02:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 02:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 02:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 02:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 02:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 02:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 02:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 02:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 03:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 03:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 03:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 03:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 03:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 03:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 03:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 03:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 04:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 04:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 04:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 04:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 04:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 04:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 04:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 04:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 05:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 05:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 05:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 05:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 05:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 05:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 05:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 05:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 06:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 06:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 06:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 06:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 06:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 06:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 06:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 06:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 07:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 07:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 07:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 07:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 07:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 07:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 07:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 07:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 08:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 08:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 08:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 08:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 08:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 08:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 08:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 08:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 09:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 09:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 09:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 09:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 09:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 09:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 09:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 09:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 10:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 10:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 10:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 10:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 10:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 10:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 10:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 10:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 11:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 11:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 11:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 11:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 11:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 11:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 11:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 11:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 12:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 12:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 12:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 12:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 12:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 12:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 12:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 12:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 13:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 13:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 13:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 13:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 13:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 13:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 13:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 13:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 14:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 14:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 14:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 14:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 14:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 14:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 14:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 14:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 15:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 15:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 15:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 15:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 15:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 15:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 15:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 15:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 16:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 16:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 16:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 16:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 16:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 16:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 16:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 16:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 17:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 17:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 17:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 17:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 17:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 17:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 17:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 17:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 18:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 18:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 18:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 18:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 18:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 18:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 18:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 18:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 19:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 19:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 19:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 19:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 19:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 19:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 19:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 19:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 20:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 20:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 20:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 20:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 20:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 20:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 20:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 20:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 21:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 21:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 21:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 21:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 21:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 21:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 21:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 21:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 22:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 22:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 22:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 22:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 22:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 22:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 22:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 22:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 23:00 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 23:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 23:15 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 23:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 23:30 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 23:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-23 23:45 -> 0; H: 0; D: 0; W: 12.2; M: 20.8
  setTime("2013-06-23 23:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(   12.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 00:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 00:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 00:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 00:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 00:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 00:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 00:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 00:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 01:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 01:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 01:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 01:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 01:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 01:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 01:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 01:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 02:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 02:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 02:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 02:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 02:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 02:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 02:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 02:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 03:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 03:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 03:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 03:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 03:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 03:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 03:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 03:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 04:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 04:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 04:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 04:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 04:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 04:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 04:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 04:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 05:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 05:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 05:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 05:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 05:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 05:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 05:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 05:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 06:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 06:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 06:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 06:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 06:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 06:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 06:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 06:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 07:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 07:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 07:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 07:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 07:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 07:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 07:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 07:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 08:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 08:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 08:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 08:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 08:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 08:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 08:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 08:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 09:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 09:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 09:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 09:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 09:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 09:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 09:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 09:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 10:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 10:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 10:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 10:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 10:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 10:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 10:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 10:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 11:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 11:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 11:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 11:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 11:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 11:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 11:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 11:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 12:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 12:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 12:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 12:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 12:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 12:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 12:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 12:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 13:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 13:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 13:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 13:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 13:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 13:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 13:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 13:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 14:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 14:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 14:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 14:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 14:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 14:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 14:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 14:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 15:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 15:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 15:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 15:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 15:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 15:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 15:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 15:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 16:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 16:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 16:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 16:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 16:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 16:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 16:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 16:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 17:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 17:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 17:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 17:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 17:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 17:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 17:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 17:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 18:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 18:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 18:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 18:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 18:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 18:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 18:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 18:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 19:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 19:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 19:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 19:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 19:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 19:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 19:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 19:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 20:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 20:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 20:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 20:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 20:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 20:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 20:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 20:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 21:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 21:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 21:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 21:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 21:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 21:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 21:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 21:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 22:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 22:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 22:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 22:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 22:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 22:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 22:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 22:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 23:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 23:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 23:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 23:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 23:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 23:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-24 23:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-24 23:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 00:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 00:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 00:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 00:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 00:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 00:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 00:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 00:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 01:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 01:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 01:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 01:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 01:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 01:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 01:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 01:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 02:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 02:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 02:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 02:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 02:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 02:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 02:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 02:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 03:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 03:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 03:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 03:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 03:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 03:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 03:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 03:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 04:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 04:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 04:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 04:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 04:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 04:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 04:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 04:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 05:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 05:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 05:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 05:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 05:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 05:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 05:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 05:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 06:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 06:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 06:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 06:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 06:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 06:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 06:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 06:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 07:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 07:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 07:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 07:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 07:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 07:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 07:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 07:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 08:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 08:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 08:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 08:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 08:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 08:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 08:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 08:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 09:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 09:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 09:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 09:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 09:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 09:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 09:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 09:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 10:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 10:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 10:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 10:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 10:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 10:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 10:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 10:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 11:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 11:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 11:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 11:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 11:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 11:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 11:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 11:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 12:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 12:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 12:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 12:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 12:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 12:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 12:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 12:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 13:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 13:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 13:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 13:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 13:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 13:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 13:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 13:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 14:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 14:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 14:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 14:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 14:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 14:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 14:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 14:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 15:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 15:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 15:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 15:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 15:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 15:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 15:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 15:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 16:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 16:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 16:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 16:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 16:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 16:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 16:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 16:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 17:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 17:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 17:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 17:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 17:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 17:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 17:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 17:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 18:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 18:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 18:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 18:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 18:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 18:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 18:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 18:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 19:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 19:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 19:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 19:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 19:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 19:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 19:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 19:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 20:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 20:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 20:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 20:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 20:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 20:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 20:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 20:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 21:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 21:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 21:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 21:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 21:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 21:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 21:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 21:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 22:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 22:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 22:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 22:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 22:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 22:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 22:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 22:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 23:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 23:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 23:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 23:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 23:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 23:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-25 23:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-25 23:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 00:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 00:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 00:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 00:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 00:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 00:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 00:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 00:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 01:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 01:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 01:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 01:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 01:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 01:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 01:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 01:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 02:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 02:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 02:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 02:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 02:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 02:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 02:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 02:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 03:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 03:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 03:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 03:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 03:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 03:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 03:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 03:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 04:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 04:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 04:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 04:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 04:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 04:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 04:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 04:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 05:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 05:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 05:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 05:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 05:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 05:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 05:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 05:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 06:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 06:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 06:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 06:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 06:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 06:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 06:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 06:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 07:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 07:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 07:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 07:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 07:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 07:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 07:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 07:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 08:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 08:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 08:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 08:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 08:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 08:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 08:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 08:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 09:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 09:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 09:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 09:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 09:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 09:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 09:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 09:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 10:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 10:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 10:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 10:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 10:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 10:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 10:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 10:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 11:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 11:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 11:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 11:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 11:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 11:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 11:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 11:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 12:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 12:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 12:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 12:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 12:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 12:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 12:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 12:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 13:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 13:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 13:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 13:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 13:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 13:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 13:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 13:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 14:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 14:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 14:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 14:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 14:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 14:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 14:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 14:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 15:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 15:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 15:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 15:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 15:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 15:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 15:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 15:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 16:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 16:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 16:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 16:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 16:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 16:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 16:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 16:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 17:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 17:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 17:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 17:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 17:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 17:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 17:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 17:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 18:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 18:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 18:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 18:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 18:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 18:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 18:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 18:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 19:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 19:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 19:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 19:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 19:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 19:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 19:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 19:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 20:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 20:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 20:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 20:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 20:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 20:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 20:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 20:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 21:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 21:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 21:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 21:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 21:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 21:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 21:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 21:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 22:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 22:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 22:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 22:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 22:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 22:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 22:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 22:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 23:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 23:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 23:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 23:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 23:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 23:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-26 23:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-26 23:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 00:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 00:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 00:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 00:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 00:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 00:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 00:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 00:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 01:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 01:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 01:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 01:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 01:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 01:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 01:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 01:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 02:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 02:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 02:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 02:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 02:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 02:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 02:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 02:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 03:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 03:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 03:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 03:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 03:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 03:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 03:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 03:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 04:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 04:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 04:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 04:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 04:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 04:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 04:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 04:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 05:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 05:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 05:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 05:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 05:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 05:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 05:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 05:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 06:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 06:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 06:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 06:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 06:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 06:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 06:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 06:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 07:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 07:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 07:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 07:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 07:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 07:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 07:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 07:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 08:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 08:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 08:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 08:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 08:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 08:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 08:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 08:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 09:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 09:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 09:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 09:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 09:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 09:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 09:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 09:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 10:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 10:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 10:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 10:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 10:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 10:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 10:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 10:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 11:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 11:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 11:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 11:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 11:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 11:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 11:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 11:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 12:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 12:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 12:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 12:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 12:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 12:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 12:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 12:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 13:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 13:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 13:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 13:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 13:30 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 13:30", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 13:45 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 13:45", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 14:00 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 14:00", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 14:15 -> 0; H: 0; D: 0; W: 0; M: 20.8
  setTime("2013-06-27 14:15", tm, ts);
  rainGauge.update(ts, 20.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   20.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 14:30 -> 0.2; H: 0.2; D: 0.2; W: 0.2; M: 21
  setTime("2013-06-27 14:30", tm, ts);
  rainGauge.update(ts, 21);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 14:45 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 21
  setTime("2013-06-27 14:45", tm, ts);
  rainGauge.update(ts, 21);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 15:00 -> 0.2; H: 0.4; D: 0.4; W: 0.4; M: 21.2
  setTime("2013-06-27 15:00", tm, ts);
  rainGauge.update(ts, 21.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 15:15 -> 0.2; H: 0.6; D: 0.6; W: 0.6; M: 21.4
  setTime("2013-06-27 15:15", tm, ts);
  rainGauge.update(ts, 21.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 15:30 -> 0; H: 0.4; D: 0.6; W: 0.6; M: 21.4
  setTime("2013-06-27 15:30", tm, ts);
  rainGauge.update(ts, 21.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 15:45 -> 0.2; H: 0.6; D: 0.8; W: 0.8; M: 21.6
  setTime("2013-06-27 15:45", tm, ts);
  rainGauge.update(ts, 21.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 16:00 -> 0.2; H: 0.6; D: 1; W: 1; M: 21.8
  setTime("2013-06-27 16:00", tm, ts);
  rainGauge.update(ts, 21.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 16:15 -> 0; H: 0.4; D: 1; W: 1; M: 21.8
  setTime("2013-06-27 16:15", tm, ts);
  rainGauge.update(ts, 21.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   21.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 16:30 -> 0.2; H: 0.6; D: 1.2; W: 1.2; M: 22
  setTime("2013-06-27 16:30", tm, ts);
  rainGauge.update(ts, 22);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 16:45 -> 0.2; H: 0.6; D: 1.4; W: 1.4; M: 22.2
  setTime("2013-06-27 16:45", tm, ts);
  rainGauge.update(ts, 22.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 17:00 -> 0; H: 0.4; D: 1.4; W: 1.4; M: 22.2
  setTime("2013-06-27 17:00", tm, ts);
  rainGauge.update(ts, 22.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 17:15 -> 0; H: 0.4; D: 1.4; W: 1.4; M: 22.2
  setTime("2013-06-27 17:15", tm, ts);
  rainGauge.update(ts, 22.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 17:30 -> 0.2; H: 0.4; D: 1.6; W: 1.6; M: 22.4
  setTime("2013-06-27 17:30", tm, ts);
  rainGauge.update(ts, 22.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 17:45 -> 0.2; H: 0.4; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 17:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 18:00 -> 0; H: 0.4; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 18:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 18:15 -> 0; H: 0.4; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 18:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 18:30 -> 0; H: 0.2; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 18:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 18:45 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 18:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 19:00 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 19:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 19:15 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 19:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 19:30 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 19:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 19:45 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 19:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 20:00 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 20:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 20:15 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 20:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 20:30 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 20:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 20:45 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 20:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 21:00 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 21:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 21:15 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 21:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 21:30 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 21:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 21:45 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 21:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 22:00 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 22:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 22:15 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 22:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 22:30 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 22:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 22:45 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 22:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 23:00 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 23:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 23:15 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 23:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 23:30 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 23:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-27 23:45 -> 0; H: 0; D: 1.8; W: 1.8; M: 22.6
  setTime("2013-06-27 23:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 00:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 00:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 00:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 00:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 00:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 00:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 00:45 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 00:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 01:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 01:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 01:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 01:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 01:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 01:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 01:45 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 01:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 02:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 02:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 02:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 02:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 02:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 02:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 02:45 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 02:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 03:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 03:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 03:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 03:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 03:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 03:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 03:45 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 03:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 04:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 04:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 04:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 04:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 04:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 04:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 04:45 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 04:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 05:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 05:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 05:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 05:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 05:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 05:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 05:45 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 05:45", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 06:00 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 06:00", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 06:15 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 06:15", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 06:30 -> 0; H: 0; D: 0; W: 1.8; M: 22.6
  setTime("2013-06-28 06:30", tm, ts);
  rainGauge.update(ts, 22.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 06:45 -> 0.2; H: 0.2; D: 0.2; W: 2; M: 22.8
  setTime("2013-06-28 06:45", tm, ts);
  rainGauge.update(ts, 22.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   22.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 07:00 -> 0.2; H: 0.4; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 07:00", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 07:15 -> 0; H: 0.4; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 07:15", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 07:30 -> 0; H: 0.4; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 07:30", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 07:45 -> 0; H: 0.2; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 07:45", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 08:00 -> 0; H: 0; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 08:00", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 08:15 -> 0; H: 0; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 08:15", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 08:30 -> 0; H: 0; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 08:30", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 08:45 -> 0; H: 0; D: 0.4; W: 2.2; M: 23
  setTime("2013-06-28 08:45", tm, ts);
  rainGauge.update(ts, 23);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 09:00 -> 0.2; H: 0.2; D: 0.6; W: 2.4; M: 23.2
  setTime("2013-06-28 09:00", tm, ts);
  rainGauge.update(ts, 23.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   23.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 09:15 -> 0.8; H: 1; D: 1.4; W: 3.2; M: 24
  setTime("2013-06-28 09:15", tm, ts);
  rainGauge.update(ts, 24);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   24.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 09:30 -> 0.6; H: 1.6; D: 2; W: 3.8; M: 24.6
  setTime("2013-06-28 09:30", tm, ts);
  rainGauge.update(ts, 24.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   24.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 09:45 -> 0.2; H: 1.8; D: 2.2; W: 4; M: 24.8
  setTime("2013-06-28 09:45", tm, ts);
  rainGauge.update(ts, 24.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   24.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 10:00 -> 0.4; H: 2; D: 2.6; W: 4.4; M: 25.2
  setTime("2013-06-28 10:00", tm, ts);
  rainGauge.update(ts, 25.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    2.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   25.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 10:15 -> 0.2; H: 1.4; D: 2.8; W: 4.6; M: 25.4
  setTime("2013-06-28 10:15", tm, ts);
  rainGauge.update(ts, 25.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    2.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   25.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 10:30 -> 0.2; H: 1; D: 3; W: 4.8; M: 25.6
  setTime("2013-06-28 10:30", tm, ts);
  rainGauge.update(ts, 25.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   25.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 10:45 -> 0; H: 0.8; D: 3; W: 4.8; M: 25.6
  setTime("2013-06-28 10:45", tm, ts);
  rainGauge.update(ts, 25.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   25.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 11:00 -> 0.6; H: 1; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 11:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 11:15 -> 0; H: 0.8; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 11:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 11:30 -> 0; H: 0.6; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 11:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 11:45 -> 0; H: 0.6; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 11:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 12:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 12:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 12:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 12:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 12:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 12:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 12:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 12:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 13:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 13:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 13:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 13:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 13:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 13:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 13:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 13:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 14:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 14:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 14:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 14:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 14:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 14:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 14:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 14:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 15:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 15:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 15:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 15:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 15:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 15:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 15:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 15:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 16:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 16:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 16:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 16:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 16:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 16:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 16:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 16:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 17:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 17:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 17:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 17:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 17:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 17:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 17:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 17:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 18:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 18:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 18:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 18:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 18:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 18:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 18:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 18:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 19:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 19:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 19:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 19:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 19:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 19:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 19:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 19:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 20:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 20:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 20:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 20:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 20:30 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 20:30", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 20:45 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 20:45", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 21:00 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 21:00", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 21:15 -> 0; H: 0; D: 3.6; W: 5.4; M: 26.2
  setTime("2013-06-28 21:15", tm, ts);
  rainGauge.update(ts, 26.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 21:30 -> 0.2; H: 0.2; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 21:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 21:45 -> 0; H: 0.2; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 21:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 22:00 -> 0; H: 0.2; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 22:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 22:15 -> 0; H: 0.2; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 22:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 22:30 -> 0; H: 0; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 22:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 22:45 -> 0; H: 0; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 22:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 23:00 -> 0; H: 0; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 23:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 23:15 -> 0; H: 0; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 23:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 23:30 -> 0; H: 0; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 23:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-28 23:45 -> 0; H: 0; D: 3.8; W: 5.6; M: 26.4
  setTime("2013-06-28 23:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 00:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 00:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 00:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 00:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 00:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 00:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 00:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 00:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 01:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 01:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 01:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 01:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 01:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 01:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 01:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 01:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 02:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 02:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 02:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 02:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 02:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 02:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 02:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 02:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 03:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 03:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 03:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 03:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 03:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 03:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 03:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 03:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 04:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 04:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 04:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 04:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 04:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 04:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 04:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 04:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 05:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 05:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 05:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 05:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 05:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 05:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 05:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 05:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 06:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 06:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 06:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 06:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 06:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 06:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 06:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 06:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 07:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 07:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 07:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 07:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 07:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 07:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 07:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 07:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 08:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 08:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 08:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 08:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 08:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 08:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 08:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 08:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 09:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 09:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 09:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 09:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 09:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 09:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 09:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 09:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 10:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 10:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 10:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 10:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 10:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 10:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 10:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 10:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 11:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 11:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 11:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 11:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 11:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 11:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 11:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 11:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 12:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 12:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 12:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 12:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 12:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 12:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 12:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 12:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 13:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 13:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 13:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 13:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 13:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 13:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 13:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 13:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 14:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 14:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 14:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 14:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 14:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 14:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 14:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 14:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 15:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 15:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 15:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 15:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 15:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 15:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 15:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 15:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 16:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 16:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 16:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 16:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 16:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 16:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 16:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 16:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 17:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 17:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 17:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 17:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 17:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 17:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 17:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 17:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 18:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 18:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 18:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 18:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 18:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 18:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 18:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 18:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 19:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 19:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 19:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 19:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 19:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 19:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 19:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 19:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 20:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 20:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 20:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 20:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 20:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 20:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 20:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 20:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 21:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 21:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 21:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 21:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 21:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 21:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 21:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 21:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 22:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 22:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 22:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 22:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 22:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 22:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 22:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 22:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 23:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 23:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 23:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 23:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 23:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 23:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-29 23:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-29 23:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 00:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 00:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 00:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 00:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 00:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 00:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 00:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 00:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 01:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 01:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 01:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 01:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 01:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 01:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 01:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 01:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 02:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 02:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 02:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 02:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 02:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 02:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 02:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 02:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 03:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 03:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 03:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 03:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 03:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 03:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 03:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 03:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 04:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 04:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 04:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 04:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 04:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 04:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 04:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 04:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 05:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 05:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 05:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 05:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 05:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 05:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 05:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 05:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 06:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 06:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 06:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 06:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 06:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 06:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 06:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 06:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 07:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 07:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 07:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 07:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 07:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 07:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 07:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 07:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 08:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 08:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 08:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 08:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 08:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 08:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 08:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 08:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 09:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 09:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 09:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 09:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 09:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 09:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 09:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 09:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 10:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 10:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 10:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 10:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 10:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 10:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 10:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 10:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 11:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 11:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 11:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 11:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 11:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 11:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 11:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 11:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 12:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 12:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 12:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 12:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 12:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 12:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 12:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 12:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 13:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 13:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 13:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 13:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 13:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 13:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 13:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 13:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 14:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 14:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 14:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 14:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 14:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 14:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 14:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 14:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 15:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 15:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 15:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 15:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 15:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 15:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 15:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 15:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 16:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 16:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 16:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 16:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 16:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 16:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 16:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 16:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 17:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 17:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 17:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 17:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 17:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 17:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 17:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 17:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 18:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 18:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 18:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 18:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 18:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 18:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 18:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 18:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 19:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 19:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 19:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 19:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 19:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 19:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 19:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 19:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 20:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 20:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 20:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 20:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 20:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 20:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 20:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 20:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 21:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 21:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 21:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 21:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 21:30 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 21:30", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 21:45 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 21:45", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 22:00 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 22:00", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 22:15 -> 0; H: 0; D: 0; W: 5.6; M: 26.4
  setTime("2013-06-30 22:15", tm, ts);
  rainGauge.update(ts, 26.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 22:30 -> 0.2; H: 0.2; D: 0.2; W: 5.8; M: 26.6
  setTime("2013-06-30 22:30", tm, ts);
  rainGauge.update(ts, 26.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 22:45 -> 0.2; H: 0.4; D: 0.4; W: 6; M: 26.8
  setTime("2013-06-30 22:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 23:00 -> 0; H: 0.4; D: 0.4; W: 6; M: 26.8
  setTime("2013-06-30 23:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 23:15 -> 0; H: 0.4; D: 0.4; W: 6; M: 26.8
  setTime("2013-06-30 23:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 23:30 -> 0; H: 0.2; D: 0.4; W: 6; M: 26.8
  setTime("2013-06-30 23:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-06-30 23:45 -> 0; H: 0; D: 0.4; W: 6; M: 26.8
  setTime("2013-06-30 23:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    6.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(   26.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 00:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 00:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 00:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 00:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 00:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 00:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 00:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 00:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 01:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 01:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 01:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 01:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 01:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 01:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 01:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 01:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 02:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 02:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 02:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 02:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 02:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 02:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 02:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 02:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 03:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 03:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 03:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 03:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 03:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 03:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 03:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 03:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 04:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 04:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 04:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 04:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 04:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 04:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 04:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 04:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 05:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 05:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 05:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 05:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 05:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 05:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 05:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 05:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 06:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 06:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 06:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 06:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 06:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 06:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 06:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 06:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 07:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 07:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 07:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 07:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 07:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 07:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 07:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 07:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 08:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 08:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 08:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 08:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 08:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 08:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 08:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 08:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 09:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 09:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 09:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 09:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 09:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 09:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 09:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 09:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 10:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 10:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 10:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 10:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 10:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 10:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 10:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 10:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 11:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 11:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 11:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 11:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 11:30 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 11:30", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 11:45 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 11:45", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 12:00 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 12:00", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 12:15 -> 0; H: 0; D: 0; W: 0; M: 0
  setTime("2013-07-01 12:15", tm, ts);
  rainGauge.update(ts, 26.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 12:30 -> 0.2; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 12:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 12:45 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 12:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 13:00 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 13:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 13:15 -> 0; H: 0.2; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 13:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 13:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 13:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 13:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 13:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 14:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 14:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 14:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 14:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 14:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 14:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 14:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 14:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 15:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 15:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 15:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 15:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 15:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 15:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 15:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 15:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 16:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 16:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 16:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 16:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 16:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 16:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 16:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 16:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 17:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 17:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 17:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 17:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 17:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 17:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 17:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 17:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 18:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 18:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 18:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 18:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 18:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 18:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 18:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 18:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 19:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 19:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 19:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 19:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 19:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 19:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 19:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 19:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 20:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 20:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 20:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 20:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 20:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 20:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 20:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 20:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 21:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 21:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 21:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 21:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 21:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 21:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 21:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 21:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 22:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 22:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 22:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 22:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 22:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 22:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 22:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 22:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 23:00 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 23:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 23:15 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 23:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 23:30 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 23:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-01 23:45 -> 0; H: 0; D: 0.2; W: 0.2; M: 0.2
  setTime("2013-07-01 23:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 00:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 00:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 00:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 00:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 00:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 00:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 00:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 00:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 01:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 01:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 01:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 01:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 01:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 01:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 01:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 01:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 02:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 02:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 02:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 02:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 02:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 02:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 02:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 02:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 03:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 03:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 03:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 03:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 03:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 03:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 03:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 03:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 04:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 04:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 04:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 04:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 04:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 04:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 04:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 04:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 05:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 05:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 05:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 05:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 05:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 05:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 05:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 05:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 06:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 06:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 06:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 06:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 06:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 06:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 06:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 06:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 07:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 07:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 07:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 07:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 07:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 07:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 07:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 07:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 08:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 08:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 08:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 08:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 08:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 08:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 08:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 08:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 09:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 09:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 09:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 09:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 09:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 09:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 09:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 09:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 10:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 10:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 10:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 10:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 10:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 10:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 10:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 10:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 11:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 11:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 11:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 11:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 11:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 11:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 11:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 11:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 12:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 12:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 12:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 12:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 12:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 12:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 12:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 12:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 13:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 13:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 13:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 13:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 13:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 13:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 13:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 13:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 14:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 14:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 14:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 14:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 14:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 14:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 14:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 14:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 15:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 15:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 15:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 15:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 15:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 15:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 15:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 15:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 16:00 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 16:00", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 16:15 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 16:15", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 16:30 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 16:30", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 16:45 -> 0; H: 0; D: 0; W: 0.2; M: 0.2
  setTime("2013-07-02 16:45", tm, ts);
  rainGauge.update(ts, 27);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 17:00 -> 0.2; H: 0.2; D: 0.2; W: 0.4; M: 0.4
  setTime("2013-07-02 17:00", tm, ts);
  rainGauge.update(ts, 27.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 17:15 -> 0; H: 0.2; D: 0.2; W: 0.4; M: 0.4
  setTime("2013-07-02 17:15", tm, ts);
  rainGauge.update(ts, 27.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 17:30 -> 0.2; H: 0.4; D: 0.4; W: 0.6; M: 0.6
  setTime("2013-07-02 17:30", tm, ts);
  rainGauge.update(ts, 27.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 17:45 -> 0.2; H: 0.6; D: 0.6; W: 0.8; M: 0.8
  setTime("2013-07-02 17:45", tm, ts);
  rainGauge.update(ts, 27.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 18:00 -> 0; H: 0.4; D: 0.6; W: 0.8; M: 0.8
  setTime("2013-07-02 18:00", tm, ts);
  rainGauge.update(ts, 27.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 18:15 -> 0.2; H: 0.6; D: 0.8; W: 1; M: 1
  setTime("2013-07-02 18:15", tm, ts);
  rainGauge.update(ts, 27.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 18:30 -> 0; H: 0.4; D: 0.8; W: 1; M: 1
  setTime("2013-07-02 18:30", tm, ts);
  rainGauge.update(ts, 27.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 18:45 -> 0.2; H: 0.4; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 18:45", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 19:00 -> 0; H: 0.4; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 19:00", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 19:15 -> 0; H: 0.2; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 19:15", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 19:30 -> 0; H: 0.2; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 19:30", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 19:45 -> 0; H: 0; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 19:45", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 20:00 -> 0; H: 0; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 20:00", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 20:15 -> 0; H: 0; D: 1; W: 1.2; M: 1.2
  setTime("2013-07-02 20:15", tm, ts);
  rainGauge.update(ts, 28);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 20:30 -> 0.4; H: 0.4; D: 1.4; W: 1.6; M: 1.6
  setTime("2013-07-02 20:30", tm, ts);
  rainGauge.update(ts, 28.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.4, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 20:45 -> 0.2; H: 0.6; D: 1.6; W: 1.8; M: 1.8
  setTime("2013-07-02 20:45", tm, ts);
  rainGauge.update(ts, 28.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 21:00 -> 0; H: 0.6; D: 1.6; W: 1.8; M: 1.8
  setTime("2013-07-02 21:00", tm, ts);
  rainGauge.update(ts, 28.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 21:15 -> 0.2; H: 0.8; D: 1.8; W: 2; M: 2
  setTime("2013-07-02 21:15", tm, ts);
  rainGauge.update(ts, 28.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 21:30 -> 0; H: 0.4; D: 1.8; W: 2; M: 2
  setTime("2013-07-02 21:30", tm, ts);
  rainGauge.update(ts, 28.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    2.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 21:45 -> 1.2; H: 1.4; D: 3; W: 3.2; M: 3.2
  setTime("2013-07-02 21:45", tm, ts);
  rainGauge.update(ts, 30);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 22:00 -> 0.2; H: 1.6; D: 3.2; W: 3.4; M: 3.4
  setTime("2013-07-02 22:00", tm, ts);
  rainGauge.update(ts, 30.2);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.4, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 22:15 -> 0.4; H: 1.8; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 22:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 22:30 -> 0; H: 1.8; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 22:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 22:45 -> 0; H: 0.6; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 22:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.6, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 23:00 -> 0; H: 0.4; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 23:00", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 23:15 -> 0; H: 0; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 23:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 23:30 -> 0; H: 0; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 23:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-02 23:45 -> 0; H: 0; D: 3.6; W: 3.8; M: 3.8
  setTime("2013-07-02 23:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    3.6, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 00:00 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 00:00", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 00:15 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 00:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 00:30 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 00:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 00:45 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 00:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 01:00 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 01:00", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 01:15 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 01:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 01:30 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 01:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 01:45 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 01:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 02:00 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 02:00", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 02:15 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 02:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 02:30 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 02:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 02:45 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 02:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 03:00 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 03:00", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 03:15 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 03:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 03:30 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 03:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 03:45 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 03:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 04:00 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 04:00", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 04:15 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 04:15", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 04:30 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 04:30", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 04:45 -> 0; H: 0; D: 0; W: 3.8; M: 3.8
  setTime("2013-07-03 04:45", tm, ts);
  rainGauge.update(ts, 30.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    3.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 05:00 -> 0.8; H: 0.8; D: 0.8; W: 4.6; M: 4.6
  setTime("2013-07-03 05:00", tm, ts);
  rainGauge.update(ts, 31.4);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.8, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.8, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.6, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.6, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 05:15 -> 0.2; H: 1; D: 1; W: 4.8; M: 4.8
  setTime("2013-07-03 05:15", tm, ts);
  rainGauge.update(ts, 31.6);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    4.8, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 05:30 -> 0.2; H: 1.2; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 05:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 05:45 -> 0; H: 1.2; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 05:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    1.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 06:00 -> 0; H: 0.4; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 06:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.4, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 06:15 -> 0; H: 0.2; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 06:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.2, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 06:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 06:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 06:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 06:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 07:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 07:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 07:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 07:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 07:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 07:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 07:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 07:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 08:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 08:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 08:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 08:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 08:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 08:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 08:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 08:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 09:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 09:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 09:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 09:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 09:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 09:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 09:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 09:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 10:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 10:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 10:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 10:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 10:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 10:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 10:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 10:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 11:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 11:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 11:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 11:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 11:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 11:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 11:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 11:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 12:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 12:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 12:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 12:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 12:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 12:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 12:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 12:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 13:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 13:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 13:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 13:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 13:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 13:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 13:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 13:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 14:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 14:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 14:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 14:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 14:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 14:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 14:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 14:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 15:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 15:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 15:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 15:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 15:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 15:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 15:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 15:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 16:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 16:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 16:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 16:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 16:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 16:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 16:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 16:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 17:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 17:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 17:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 17:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 17:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 17:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 17:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 17:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 18:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 18:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 18:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 18:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 18:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 18:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 18:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 18:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 19:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 19:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 19:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 19:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 19:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 19:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 19:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 19:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 20:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 20:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 20:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 20:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 20:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 20:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 20:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 20:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 21:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 21:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 21:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 21:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 21:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 21:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 21:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 21:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 22:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 22:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 22:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 22:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 22:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 22:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 22:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 22:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 23:00 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 23:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 23:15 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 23:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 23:30 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 23:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-03 23:45 -> 0; H: 0; D: 1.2; W: 5; M: 5
  setTime("2013-07-03 23:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    1.2, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 00:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 00:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 00:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 00:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 00:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 00:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 00:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 00:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 01:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 01:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 01:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 01:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 01:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 01:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 01:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 01:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 02:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 02:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 02:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 02:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 02:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 02:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 02:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 02:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 03:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 03:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 03:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 03:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 03:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 03:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 03:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 03:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 04:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 04:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 04:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 04:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 04:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 04:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 04:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 04:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 05:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 05:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 05:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 05:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 05:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 05:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 05:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 05:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 06:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 06:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 06:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 06:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 06:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 06:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 06:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 06:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 07:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 07:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 07:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 07:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 07:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 07:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 07:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 07:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 08:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 08:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 08:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 08:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 08:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 08:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 08:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 08:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 09:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 09:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 09:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 09:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 09:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 09:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 09:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 09:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 10:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 10:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 10:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 10:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 10:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 10:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 10:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 10:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 11:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 11:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 11:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 11:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 11:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 11:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 11:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 11:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 12:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 12:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 12:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 12:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 12:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 12:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 12:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 12:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 13:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 13:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 13:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 13:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 13:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 13:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 13:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 13:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 14:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 14:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 14:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 14:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 14:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 14:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 14:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 14:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 15:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 15:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 15:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 15:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 15:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 15:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 15:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 15:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 16:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 16:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 16:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 16:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 16:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 16:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 16:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 16:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 17:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 17:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 17:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 17:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 17:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 17:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 17:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 17:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 18:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 18:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 18:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 18:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 18:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 18:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 18:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 18:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 19:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 19:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 19:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 19:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 19:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 19:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 19:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 19:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 20:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 20:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 20:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 20:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 20:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 20:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 20:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 20:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 21:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 21:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 21:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 21:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 21:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 21:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 21:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 21:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 22:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 22:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 22:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 22:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 22:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 22:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 22:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 22:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 23:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 23:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 23:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 23:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 23:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 23:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-04 23:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-04 23:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 00:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 00:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 00:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 00:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 00:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 00:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 00:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 00:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 01:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 01:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 01:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 01:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 01:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 01:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 01:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 01:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 02:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 02:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 02:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 02:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 02:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 02:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 02:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 02:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 03:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 03:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 03:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 03:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 03:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 03:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 03:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 03:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 04:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 04:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 04:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 04:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 04:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 04:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 04:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 04:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 05:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 05:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 05:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 05:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 05:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 05:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 05:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 05:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 06:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 06:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 06:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 06:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 06:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 06:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 06:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 06:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 07:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 07:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 07:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 07:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 07:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 07:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 07:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 07:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 08:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 08:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 08:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 08:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 08:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 08:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 08:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 08:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 09:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 09:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 09:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 09:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 09:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 09:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 09:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 09:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 10:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 10:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 10:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 10:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 10:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 10:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 10:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 10:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 11:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 11:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 11:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 11:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 11:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 11:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 11:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 11:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 12:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 12:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 12:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 12:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 12:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 12:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 12:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 12:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 13:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 13:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 13:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 13:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 13:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 13:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 13:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 13:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 14:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 14:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 14:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 14:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 14:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 14:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 14:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 14:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 15:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 15:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 15:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 15:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 15:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 15:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 15:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 15:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 16:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 16:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 16:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 16:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 16:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 16:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 16:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 16:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 17:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 17:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 17:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 17:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 17:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 17:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 17:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 17:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 18:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 18:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 18:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 18:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 18:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 18:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 18:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 18:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 19:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 19:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 19:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 19:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 19:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 19:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 19:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 19:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 20:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 20:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 20:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 20:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 20:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 20:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 20:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 20:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 21:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 21:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 21:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 21:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 21:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 21:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 21:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 21:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 22:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 22:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 22:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 22:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 22:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 22:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 22:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 22:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 23:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 23:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 23:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 23:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 23:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 23:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-05 23:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-05 23:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 00:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 00:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 00:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 00:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 00:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 00:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 00:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 00:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 01:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 01:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 01:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 01:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 01:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 01:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 01:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 01:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 02:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 02:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 02:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 02:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 02:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 02:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 02:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 02:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 03:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 03:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 03:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 03:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 03:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 03:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 03:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 03:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 04:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 04:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 04:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 04:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 04:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 04:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 04:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 04:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 05:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 05:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 05:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 05:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 05:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 05:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 05:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 05:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 06:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 06:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 06:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 06:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 06:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 06:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 06:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 06:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 07:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 07:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 07:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 07:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 07:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 07:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 07:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 07:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 08:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 08:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 08:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 08:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 08:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 08:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 08:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 08:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 09:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 09:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 09:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 09:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 09:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 09:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 09:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 09:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 10:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 10:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 10:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 10:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 10:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 10:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 10:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 10:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 11:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 11:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 11:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 11:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 11:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 11:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 11:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 11:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 12:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 12:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 12:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 12:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 12:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 12:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 12:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 12:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 13:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 13:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 13:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 13:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 13:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 13:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 13:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 13:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 14:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 14:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 14:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 14:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 14:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 14:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 14:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 14:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 15:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 15:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 15:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 15:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 15:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 15:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 15:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 15:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 16:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 16:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 16:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 16:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 16:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 16:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 16:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 16:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 17:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 17:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 17:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 17:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 17:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 17:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 17:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 17:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 18:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 18:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 18:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 18:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 18:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 18:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 18:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 18:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 19:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 19:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 19:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 19:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 19:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 19:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 19:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 19:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 20:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 20:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 20:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 20:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 20:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 20:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 20:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 20:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 21:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 21:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 21:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 21:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 21:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 21:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 21:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 21:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 22:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 22:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 22:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 22:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 22:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 22:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 22:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 22:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 23:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 23:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 23:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 23:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 23:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 23:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-06 23:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-06 23:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 00:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 00:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 00:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 00:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 00:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 00:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 00:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 00:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 01:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 01:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 01:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 01:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 01:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 01:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 01:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 01:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 02:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 02:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 02:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 02:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 02:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 02:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 02:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 02:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 03:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 03:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 03:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 03:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 03:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 03:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 03:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 03:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 04:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 04:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 04:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 04:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 04:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 04:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 04:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 04:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 05:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 05:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 05:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 05:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 05:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 05:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 05:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 05:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 06:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 06:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 06:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 06:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 06:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 06:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 06:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 06:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 07:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 07:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 07:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 07:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 07:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 07:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 07:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 07:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 08:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 08:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 08:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 08:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 08:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 08:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 08:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 08:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 09:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 09:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 09:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 09:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 09:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 09:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 09:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 09:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 10:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 10:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 10:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 10:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 10:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 10:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 10:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 10:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 11:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 11:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 11:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 11:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 11:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 11:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 11:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 11:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 12:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 12:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 12:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 12:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 12:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 12:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 12:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 12:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 13:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 13:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 13:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 13:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 13:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 13:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 13:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 13:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 14:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 14:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 14:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 14:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 14:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 14:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 14:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 14:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 15:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 15:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 15:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 15:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 15:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 15:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 15:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 15:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 16:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 16:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 16:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 16:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 16:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 16:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 16:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 16:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 17:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 17:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 17:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 17:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 17:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 17:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 17:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 17:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 18:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 18:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 18:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 18:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 18:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 18:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 18:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 18:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 19:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 19:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 19:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 19:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 19:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 19:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 19:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 19:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 20:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 20:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 20:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 20:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 20:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 20:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 20:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 20:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 21:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 21:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 21:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 21:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 21:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 21:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 21:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 21:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 22:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 22:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 22:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 22:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 22:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 22:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 22:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 22:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 23:00 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 23:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 23:15 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 23:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 23:30 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 23:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-07 23:45 -> 0; H: 0; D: 0; W: 5; M: 5
  setTime("2013-07-07 23:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 00:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 00:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 00:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 00:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 00:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 00:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 00:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 00:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 01:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 01:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 01:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 01:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 01:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 01:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 01:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 01:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 02:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 02:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 02:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 02:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 02:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 02:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 02:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 02:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 03:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 03:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 03:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 03:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 03:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 03:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 03:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 03:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 04:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 04:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 04:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 04:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 04:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 04:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 04:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 04:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 05:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 05:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 05:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 05:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 05:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 05:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 05:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 05:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 06:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 06:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 06:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 06:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 06:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 06:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 06:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 06:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 07:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 07:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 07:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 07:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 07:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 07:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 07:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 07:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 08:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 08:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 08:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 08:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 08:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 08:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 08:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 08:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 09:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 09:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 09:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 09:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 09:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 09:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 09:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 09:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 10:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 10:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 10:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 10:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 10:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 10:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 10:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 10:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 11:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 11:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 11:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 11:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 11:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 11:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 11:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 11:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 12:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 12:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 12:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 12:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 12:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 12:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 12:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 12:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 13:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 13:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 13:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 13:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 13:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 13:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 13:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 13:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 14:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 14:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 14:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 14:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 14:30 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 14:30", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 14:45 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 14:45", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 15:00 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 15:00", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQUAL(    5.0, rainGauge.currentMonth(), TOLERANCE);

  // 2013-07-08 15:15 -> 0; H: 0; D: 0; W: 0; M: 5
  setTime("2013-07-08 15:15", tm, ts);
  rainGauge.update(ts, 31.8);
  DEBUG_CB();
  DOUBLES_EQUAL(    0.0, rainGauge.pastHour(),     TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentDay(),   TOLERANCE);
  DOUBLES_EQUAL(    0.0, rainGauge.currentWeek(),  TOLERANCE);
  DOUBLES_EQU