#if !defined(WEATHER_UTILS_H)
#define WEATHER_UTILS_H

#ifdef ARDUINO_ARCH_AVR
  #include <stdint.h>
#endif

/*!
 * \brief Calculate dew point
 * 
 * \param celsius air temperature in °C
 * \param humidity relative humidity in %
 * 
 * \returns dew point temperature in °C
 */
float calcdewpoint(float celsius, float humidity);

/*!
 * \brief Calculate windchill temperature
 * 
 * Results are valid for temperatures <= 10°C and windspeeds >4.8 km/h only!
 * 
 * \param celsius air temperature in °C
 * \param windspeed wind speed in km/h
 * 
 * \returns windchill temperature in °C
 */
float calcwindchill(float celsius, float windspeed);

/*!
 * \brief Calculate heat index
 * 
 * Results are valid for temperatures >= 16.7°C and humidity >40% only!
 * 
 * \param celsius air temperature in °C
 * \param humidity relative humidity in %
 * 
 * \returns heat index in °C
 */
float calcheatindex(float celsius, float humidity);

/*!
 * \brief Calculate Humidex
 * 
 * \param temperature air temperature in °C
 * \param humidity relative humidity in %
 * 
 * \returns Humidex
 */
float calchumidex(float temperature, float humidity);

/*!
 * \brief Calculate natural wet bulb temperature
 *
 * \param temperature Dry-bulb temperature (air temperature) in °C
 * \param humidity relative humidity in %
 * 
 * \returns natural wet bulb temperature in °C
 */
float calcnaturalwetbulb(float temperature, float humidity);

/*!
 * \brief Calculate wet bulb globe temperature (WBGT)
 *
 * \param t_wet Natural wet-bulb temperature in °C
 * \param t_globe Globe thermometer temperature (black globe thermometer) in °C
 * \param t_dry Dry-bulb temperature (actual air temperature) in °C
 * 
 * \returns WBGT in °C
 */
float calcwbgt(float t_wet, float t_globe, float t_dry);


/*!
 * \brief Calculate perceived temperature (feels-like temperature)
 * 
 * Apply windchill or heat index depending on current data or
 * just return the real temperature.
 * 
 * \param celsius air temperature in °C
 * \param windspeed wind speed in km/h
 * \param humidity relative humidity in %
 * 
 * \returns perceived temperature in °C
 */
float perceived_temperature(float celsius, float windspeed, float humidity);

/*!
 * \brief Convert wind direction from Degrees to text (N, NNE, NE, ...)
 *
 * \param dir Wind direction in degrees
 * \param buf buffer for result (4 characters required)
 * 
 * \returns pointer to buffer
 */
char * winddir_flt_to_str(float dir, char * buf);

/*!
 * \brief Converts wind speed from Meters per Second to Beaufort.
 * 
 * \param ms Wind speed in m/s.
 * 
 * \returns Wind speed in bft.
*/        
uint8_t windspeed_ms_to_bft(float ms);

#endif
