///////////////////////////////////////////////////////////////////////////////////////////////////
// ATC_MiThermometer_Client.cpp
//
// Bluetooth low energy thermometer/hygrometer sensor client for MCUs supported by NimBLE-Arduino.
// For sensors running ATC_MiThermometer firmware (see https://github.com/pvvx/ATC_MiThermometer)
//
// https://github.com/matthias-bs/ATC_MiThermometer
//
// Based on:
// ---------
// NimBLE-Arduino by h2zero (https://github.com/h2zero/NimBLE-Arduino)
// LYWSD03MMC.py by JsBergbau (https://github.com/JsBergbau/MiTemperature2)
//
// created: 11/2022
//
//
// MIT License
//
// Copyright (c) 2022 Matthias Prinke
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//
// History:
//
// 20221123 Created
// 20240403 Added reedSwitchState, gpioTrgOutput, controlParameters,
//          tempTriggerEvent &humiTriggerEvent
// 20240425 Added device name
//
// To Do: 
// -
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include "ATC_MiThermometer.h"

const int scanTime = 5; // BLE scan time in seconds

// List of known sensors' BLE addresses
std::vector<std::string> knownBLEAddresses = {"a4:c1:38:b8:1f:7f", "a4:c1:38:bf:e1:bc"};

ATC_MiThermometer miThermometer(knownBLEAddresses);

// Iteration counter
int iteration = 0;

void setup() {
    Serial.begin(115200);
    
    // Print free heap before initialization
    Serial.println("Starting:    Free heap is " + String(ESP.getFreeHeap()));
    
    // Initialization
    miThermometer.begin();
    
    // Print free heap after initialization
    Serial.println("Initialized: Free heap is " + String(ESP.getFreeHeap()));
}

void loop() {
    // Set sensor data invalid
    miThermometer.resetData();
    
    // Get sensor data - run BLE scan for <scanTime>
    unsigned found = miThermometer.getData(scanTime);

    for (int i=0; i < miThermometer.data.size(); i++) {  
        if (miThermometer.data[i].valid) {
            Serial.println();
            Serial.printf("Sensor %d: %s\n", i, knownBLEAddresses[i].c_str());
            Serial.printf("Name: %s\n", miThermometer.data[i].name.c_str());
            Serial.printf("%.2f°C\n", miThermometer.data[i].temperature/100.0);
            Serial.printf("%.2f%%\n", miThermometer.data[i].humidity/100.0);
            Serial.printf("%.3fV\n",  miThermometer.data[i].batt_voltage/1000.0);
            Serial.printf("%d%%\n",   miThermometer.data[i].batt_level);
            Serial.printf("%ddBm\n",  miThermometer.data[i].rssi);
            Serial.printf("Measurement Count: %d\n", miThermometer.data[i].count);
            // Print boolean flags
            Serial.printf("Reed Switch State: %s\n", miThermometer.data[i].reedSwitchState ? "Open" : "Closed");
            Serial.printf("GPIO TRG Output: %s\n", miThermometer.data[i].gpioTrgOutput ? "High" : "Low");
            Serial.printf("Control Parameters: %s\n", miThermometer.data[i].controlParameters ? "Set" : "Not Set");
            Serial.printf("Temperature Trigger Event: %s\n", miThermometer.data[i].tempTriggerEvent ? "Triggered" : "Not Triggered");
            Serial.printf("Humidity Trigger Event: %s\n", miThermometer.data[i].humiTriggerEvent ? "Triggered" : "Not Triggered");
            Serial.println();
         }
    }
    Serial.println("BLE Devices found (total): " + String(found));

    // Delete results from BLEScan buffer to release memory
    miThermometer.clearScanResults();
    
    // Print iteration counter and free heap
    Serial.println("Iteration " + String(iteration++) + " - Free heap is " + String(ESP.getFreeHeap()));
    Serial.println("---");

    delay(5000);
}
