/*
 * SPDX-FileCopyrightText: 2025 M5Stack Technology CO LTD
 *
 * SPDX-License-Identifier: MIT
 */
/*
  UnitTest for utility functions
*/
#include <gtest/gtest.h>
#include <M5Unified.h>
#include <M5Utility.h>

using namespace m5::utility;

namespace {
// SHA1
struct Sha1Test {
    const char* name;
    const uint8_t* input;
    uint32_t ilen;
    const uint8_t* expected;
};

constexpr Sha1Test sha1_test_table[] = {
    {"empty", (const uint8_t[]){}, 0, (const uint8_t[]){0xDA, 0x39, 0xA3, 0xEE, 0x5E, 0x6B, 0x4B, 0x0D, 0x32, 0x55,
                                                        0xBF, 0xEF, 0x95, 0x60, 0x18, 0x90, 0xAF, 0xD8, 0x07, 0x09}},
    {"1-byte", (const uint8_t[]){0x00}, 1,
     (const uint8_t[]){0x5B, 0xA9, 0x3C, 0x9D, 0xB0, 0xCF, 0xF9, 0x3F, 0x52, 0xB5,
                       0x21, 0xD7, 0x42, 0x0E, 0x43, 0xF6, 0xED, 0xA2, 0x78, 0x4F}},
    {"55-bytes", (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     55, (const uint8_t[]){0xC1, 0xC8, 0xBB, 0xDC, 0x22, 0x79, 0x6E, 0x28, 0xC0, 0xE1,
                           0x51, 0x63, 0xD2, 0x08, 0x99, 0xB6, 0x56, 0x21, 0xD6, 0x5A}},
    {"56-bytes", (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     56, (const uint8_t[]){0xC2, 0xDB, 0x33, 0x0F, 0x60, 0x83, 0x85, 0x4C, 0x99, 0xD4,
                           0xB5, 0xBF, 0xB6, 0xE8, 0xF2, 0x9F, 0x20, 0x1B, 0xE6, 0x99}},
    {"57-bytes",
     (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     57, (const uint8_t[]){0xF0, 0x8F, 0x24, 0x90, 0x8D, 0x68, 0x25, 0x55, 0x11, 0x1B,
                           0xE7, 0xFF, 0x6F, 0x00, 0x4E, 0x78, 0x28, 0x3D, 0x98, 0x9A}},
    {"63-bytes",
     (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     63, (const uint8_t[]){0x03, 0xF0, 0x9F, 0x5B, 0x15, 0x8A, 0x7A, 0x8C, 0xDA, 0xD9,
                           0x20, 0xBD, 0xDC, 0x29, 0xB8, 0x1C, 0x18, 0xA5, 0x51, 0xF5}},
    {"64-bytes",
     (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     64, (const uint8_t[]){0x00, 0x98, 0xBA, 0x82, 0x4B, 0x5C, 0x16, 0x42, 0x7B, 0xD7,
                           0xA1, 0x12, 0x2A, 0x5A, 0x44, 0x2A, 0x25, 0xEC, 0x64, 0x4D}},
    {"65-bytes", (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                                   0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     65, (const uint8_t[]){0x11, 0x65, 0x53, 0x26, 0xC7, 0x08, 0xD7, 0x03, 0x19, 0xBE,
                           0x26, 0x10, 0xE8, 0xA5, 0x7D, 0x9A, 0x5B, 0x95, 0x9D, 0x3B}},
    {"127-bytes",
     (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     127, (const uint8_t[]){0x89, 0xD9, 0x5F, 0xA3, 0x2E, 0xD4, 0x4A, 0x7C, 0x61, 0x0B,
                            0x7E, 0xE3, 0x85, 0x17, 0xDD, 0xF5, 0x7E, 0x0B, 0xB9, 0x75}},
    {"128-bytes",
     (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     128, (const uint8_t[]){0xAD, 0x5B, 0x3F, 0xDB, 0xCB, 0x52, 0x67, 0x78, 0xC2, 0x83,
                            0x9D, 0x2F, 0x15, 0x1E, 0xA7, 0x53, 0x99, 0x5E, 0x26, 0xA0}},
    {"129-bytes",
     (const uint8_t[]){0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61,
                       0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61},
     129, (const uint8_t[]){0xD9, 0x6D, 0xEB, 0xF1, 0xBD, 0xCB, 0xC8, 0x96, 0xE6, 0xC1,
                            0x34, 0xEA, 0x76, 0xE8, 0x14, 0x1F, 0x40, 0xD7, 0x85, 0x36}},
    {"all-zero-32",
     (const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     32, (const uint8_t[]){0xDE, 0x8A, 0x84, 0x7B, 0xFF, 0x8C, 0x34, 0x3D, 0x69, 0xB8,
                           0x53, 0xA2, 0x15, 0xE6, 0xEE, 0x77, 0x5E, 0xF2, 0xEF, 0x96}},
    {"all-ff-32",
     (const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     32, (const uint8_t[]){0x9E, 0x51, 0x75, 0x00, 0x87, 0x51, 0xD0, 0x8F, 0x36, 0x14,
                           0x88, 0xC9, 0x92, 0x70, 0x86, 0xB2, 0x76, 0xB9, 0x65, 0xFA}},
    {"incrementing-32",
     (const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F},
     32, (const uint8_t[]){0xAE, 0x5B, 0xD8, 0xEF, 0xEA, 0x53, 0x22, 0xC4, 0xD9, 0x98,
                           0x6D, 0x06, 0x68, 0x0A, 0x78, 0x13, 0x92, 0xF9, 0xA6, 0x42}},
    {"pattern-AB-32",
     (const uint8_t[]){0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42,
                       0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42, 0x41, 0x42},
     32, (const uint8_t[]){0xEA, 0xEF, 0x96, 0x66, 0xF6, 0x60, 0x9A, 0x75, 0x6B, 0x49,
                           0x34, 0x01, 0x98, 0xC0, 0x12, 0x74, 0x07, 0xB5, 0xAC, 0x88}},
    {"abc", (const uint8_t[]){0x61, 0x62, 0x63}, 3,
     (const uint8_t[]){0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A, 0xBA, 0x3E,
                       0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C, 0x9C, 0xD0, 0xD8, 0x9D}},
    {"abc-long", (const uint8_t[]){0x61, 0x62, 0x63, 0x64, 0x62, 0x63, 0x64, 0x65, 0x63, 0x64, 0x65, 0x66, 0x64, 0x65,
                                   0x66, 0x67, 0x65, 0x66, 0x67, 0x68, 0x66, 0x67, 0x68, 0x69, 0x67, 0x68, 0x69, 0x6A,
                                   0x68, 0x69, 0x6A, 0x6B, 0x69, 0x6A, 0x6B, 0x6C, 0x6A, 0x6B, 0x6C, 0x6D, 0x6B, 0x6C,
                                   0x6D, 0x6E, 0x6C, 0x6D, 0x6E, 0x6F, 0x6D, 0x6E, 0x6F, 0x70, 0x6E, 0x6F, 0x70, 0x71},
     56, (const uint8_t[]){0x84, 0x98, 0x3E, 0x44, 0x1C, 0x3B, 0xD2, 0x6E, 0xBA, 0xAE,
                           0x4A, 0xA1, 0xF9, 0x51, 0x29, 0xE5, 0xE5, 0x46, 0x70, 0xF1}},
    {"quick-brown-dog",
     (const uint8_t[]){0x54, 0x68, 0x65, 0x20, 0x71, 0x75, 0x69, 0x63, 0x6B, 0x20, 0x62, 0x72, 0x6F, 0x77, 0x6E,
                       0x20, 0x66, 0x6F, 0x78, 0x20, 0x6A, 0x75, 0x6D, 0x70, 0x73, 0x20, 0x6F, 0x76, 0x65, 0x72,
                       0x20, 0x74, 0x68, 0x65, 0x20, 0x6C, 0x61, 0x7A, 0x79, 0x20, 0x64, 0x6F, 0x67},
     43, (const uint8_t[]){0x2F, 0xD4, 0xE1, 0xC6, 0x7A, 0x2D, 0x28, 0xFC, 0xED, 0x84,
                           0x9E, 0xE1, 0xBB, 0x76, 0xE7, 0x39, 0x1B, 0x93, 0xEB, 0x12}},
    {"quick-brown-cog",
     (const uint8_t[]){0x54, 0x68, 0x65, 0x20, 0x71, 0x75, 0x69, 0x63, 0x6B, 0x20, 0x62, 0x72, 0x6F, 0x77, 0x6E,
                       0x20, 0x66, 0x6F, 0x78, 0x20, 0x6A, 0x75, 0x6D, 0x70, 0x73, 0x20, 0x6F, 0x76, 0x65, 0x72,
                       0x20, 0x74, 0x68, 0x65, 0x20, 0x6C, 0x61, 0x7A, 0x79, 0x20, 0x63, 0x6F, 0x67},
     43, (const uint8_t[]){0xDE, 0x9F, 0x2C, 0x7F, 0xD2, 0x5E, 0x1B, 0x3A, 0xFA, 0xD3,
                           0xE8, 0x5A, 0x0B, 0xD1, 0x7D, 0x9B, 0x10, 0x0D, 0xB4, 0xB3}},
};

}  // namespace

TEST(Crypto_SHA1, Basic)
{
    SCOPED_TRACE("SHA1");

    for (uint32_t i = 0; i < m5::stl::size(sha1_test_table); ++i) {
        auto& e = sha1_test_table[i];
        SCOPED_TRACE(e.name);
        uint8_t digest[20]{};
        SHA1::sha1(digest, e.input, e.ilen);
        EXPECT_TRUE(memcmp(digest, e.expected, 20) == 0);
    }

    // 'a' String repeated 1000000 times
    constexpr uint32_t ilen{1000000};
    constexpr uint8_t a1000000[20] = {0x34, 0xaa, 0x97, 0x3c, 0xd4, 0xc4, 0xda, 0xa4, 0xf6, 0x1e,
                                      0xeb, 0x2b, 0xdb, 0xad, 0x27, 0x31, 0x65, 0x34, 0x01, 0x6f};
    uint8_t* in                    = (uint8_t*)malloc(1000000);
    if (in) {
        memset(in, (uint8_t)('a'), ilen);
        uint8_t digest[20]{};
        SHA1::sha1(digest, in, ilen);
        EXPECT_TRUE(memcmp(digest, a1000000, 20) == 0);

        free(in);
    } else {
        M5_LOGW("Cannot allocate %u", ilen);
    }
}
