/*
 * SPDX-FileCopyrightText: 2025 M5Stack Technology CO LTD
 *
 * SPDX-License-Identifier: MIT
 */
/*
  UnitTest for utility functions
*/
#include <gtest/gtest.h>
#include <M5Unified.h>
#include <M5Utility.h>

using namespace m5::utility;

namespace {

// Base64
struct Base64TestData {
    const uint8_t* input;
    const uint32_t ilen;
    const char* expected;
    const uint32_t expected_len;
};

constexpr Base64TestData nolinebreak_std_padding_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA==", 4},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA=", 4},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "/w==", 4},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "//8=", 4},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "////", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2+7w==", 8},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsbG8=", 8},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzdGVy", 8},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECAwQ=", 8},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM4s=", 8},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63, "QUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFB", 84},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", 64},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64, "/////////////////////////////////////////////////////////////////////////////////////w==", 88},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64, "AAECAwQFBgcICQoLDA0ODxAREhMUFRYXGBkaGxwdHh8gISIjJCUmJygpKissLS4vMDEyMzQ1Njc4OTo7PD0+Pw==", 88},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66, "QUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJD", 88},
};

constexpr Base64TestData nolinebreak_std_nopad_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA", 2},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA", 3},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "/w", 2},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "//8", 3},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "////", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2+7w", 6},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsbG8", 7},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzdGVy", 8},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECAwQ", 7},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM4s", 7},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63, "QUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFB", 84},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", 64},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64, "/////////////////////////////////////////////////////////////////////////////////////w", 86},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64, "AAECAwQFBgcICQoLDA0ODxAREhMUFRYXGBkaGxwdHh8gISIjJCUmJygpKissLS4vMDEyMzQ1Njc4OTo7PD0+Pw", 86},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66, "QUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJD", 88},
};

constexpr Base64TestData nolinebreak_url_padding_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA==", 4},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA=", 4},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "_w==", 4},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "__8=", 4},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "____", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2-7w==", 8},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsbG8=", 8},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzdGVy", 8},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECAwQ=", 8},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM4s=", 8},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63, "QUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFB", 84},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", 64},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64, "_____________________________________________________________________________________w==", 88},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64, "AAECAwQFBgcICQoLDA0ODxAREhMUFRYXGBkaGxwdHh8gISIjJCUmJygpKissLS4vMDEyMzQ1Njc4OTo7PD0-Pw==", 88},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66, "QUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJD", 88},
};

constexpr Base64TestData nolinebreak_url_nopad_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA", 2},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA", 3},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "_w", 2},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "__8", 3},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "____", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2-7w", 6},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsbG8", 7},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzdGVy", 8},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECAwQ", 7},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM4s", 7},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63, "QUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFBQUFB", 84},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA", 64},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64, "_____________________________________________________________________________________w", 86},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64, "AAECAwQFBgcICQoLDA0ODxAREhMUFRYXGBkaGxwdHh8gISIjJCUmJygpKissLS4vMDEyMzQ1Njc4OTo7PD0-Pw", 86},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66, "QUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJDQUJD", 88},
};

//
constexpr Base64TestData linebreak_std_padding_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA==", 4},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA=", 4},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "/w==", 4},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "//8=", 4},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "////", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2+7\nw==", 9},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsb\nG8=", 9},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzd\nGVy", 9},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECA\nwQ=", 9},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM\n4s=", 9},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63,
     "QUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQ"
     "UFB",
     100},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAA", 76},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64,
     "/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/"
     "////\nw==",
     105},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64,
     "AAECA\nwQFBg\ncICQo\nLDA0O\nDxARE\nhMUFR\nYXGBk\naGxwd\nHh8gI\nSIjJC\nUmJyg\npKiss\nLS4vM\nDEyMz\nQ1Njc\n4OTo7\nP"
     "D0+P\nw==",
     105},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66,
     "QUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQ"
     "UJDQ\nUJD",
     105},
};

constexpr Base64TestData linebreak_std_nopad_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA", 2},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA", 3},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "/w", 2},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "//8", 3},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "////", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2+7\nw", 7},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsb\nG8", 8},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzd\nGVy", 9},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECA\nwQ", 8},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM\n4s", 8},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63,
     "QUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQ"
     "UFB",
     100},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAA", 76},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64,
     "/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/////\n/"
     "////\nw",
     103},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64,
     "AAECA\nwQFBg\ncICQo\nLDA0O\nDxARE\nhMUFR\nYXGBk\naGxwd\nHh8gI\nSIjJC\nUmJyg\npKiss\nLS4vM\nDEyMz\nQ1Njc\n4OTo7\nP"
     "D0+P\nw",
     103},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66,
     "QUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQ"
     "UJDQ\nUJD",
     105},
};

constexpr Base64TestData linebreak_url_padding_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA==", 4},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA=", 4},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "_w==", 4},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "__8=", 4},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "____", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2-7\nw==", 9},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsb\nG8=", 9},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzd\nGVy", 9},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECA\nwQ=", 9},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM\n4s=", 9},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63,
     "QUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQ"
     "UFB",
     100},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAA", 76},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64,
     "_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_"
     "____\nw==",
     105},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64,
     "AAECA\nwQFBg\ncICQo\nLDA0O\nDxARE\nhMUFR\nYXGBk\naGxwd\nHh8gI\nSIjJC\nUmJyg\npKiss\nLS4vM\nDEyMz\nQ1Njc\n4OTo7\nP"
     "D0-P\nw==",
     105},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66,
     "QUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQ"
     "UJDQ\nUJD",
     105},
};

constexpr Base64TestData linebreak_url_nopad_table[] = {
    {(const uint8_t[]){}, 0, "", 0},
    {(const uint8_t[]){0x00}, 1, "AA", 2},
    {(const uint8_t[]){0x00, 0x00}, 2, "AAA", 3},
    {(const uint8_t[]){0x00, 0x00, 0x00}, 3, "AAAA", 4},
    {(const uint8_t[]){0xFF}, 1, "_w", 2},
    {(const uint8_t[]){0xFF, 0xFF}, 2, "__8", 3},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF}, 3, "____", 4},
    {(const uint8_t[]){0xDE, 0xAD, 0xBE, 0xEF}, 4, "3q2-7\nw", 7},
    {(const uint8_t[]){0x48, 0x65, 0x6C, 0x6C, 0x6F}, 5, "SGVsb\nG8", 8},
    {(const uint8_t[]){0x4D, 0x61, 0x73, 0x74, 0x65, 0x72}, 6, "TWFzd\nGVy", 9},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04}, 5, "AAECA\nwQ", 8},
    {(const uint8_t[]){0x7E, 0xBA, 0x90, 0x33, 0x8B}, 5, "frqQM\n4s", 8},
    {(const uint8_t[]){0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41,
                       0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41, 0x41},
     63,
     "QUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQUFBQ\nUFBQU\nFBQUF\nBQUFB\nQ"
     "UFB",
     100},
    {(const uint8_t[]){0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
     48, "AAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAAA\nAAAA", 76},
    {(const uint8_t[]){0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
                       0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
     64,
     "_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_____\n_"
     "____\nw",
     103},
    {(const uint8_t[]){0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                       0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F},
     64,
     "AAECA\nwQFBg\ncICQo\nLDA0O\nDxARE\nhMUFR\nYXGBk\naGxwd\nHh8gI\nSIjJC\nUmJyg\npKiss\nLS4vM\nDEyMz\nQ1Njc\n4OTo7\nP"
     "D0-P\nw",
     103},
    {(const uint8_t[]){0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41,
                       0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43,
                       0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42,
                       0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43, 0x41, 0x42, 0x43},
     66,
     "QUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQUJDQ\nUJDQU\nJDQUJ\nDQUJD\nQ"
     "UJDQ\nUJD",
     105},
};

}  // namespace

TEST(Utility, Base64)
{
    SCOPED_TRACE("Base64");

    char output[128]{};
    uint32_t olen{};

    {  // no linebraek, std, padding
        SCOPED_TRACE("nolinebreak, std, padding");

        for (uint32_t i = 0; i < m5::stl::size(nolinebreak_std_padding_table); ++i) {
            auto& e = nolinebreak_std_padding_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 0, false, true);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    {  // no linebreak, std, no padding
        for (uint32_t i = 0; i < m5::stl::size(nolinebreak_std_nopad_table); ++i) {
            auto& e = nolinebreak_std_nopad_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 0, false, false);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    {  // no linebreak, url, padding
        for (uint32_t i = 0; i < m5::stl::size(nolinebreak_url_padding_table); ++i) {
            auto& e = nolinebreak_url_padding_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 0, true, true);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    {  // no linebreak, url, no padding
        for (uint32_t i = 0; i < m5::stl::size(nolinebreak_url_nopad_table); ++i) {
            auto& e = nolinebreak_url_nopad_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 0, true, false);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    //
    {  // linebraek, std, padding
        SCOPED_TRACE("linebreak, std, padding");

        for (uint32_t i = 0; i < m5::stl::size(linebreak_std_padding_table); ++i) {
            auto& e = linebreak_std_padding_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 5, false, true);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    {  // linebreak, std, no padding
        for (uint32_t i = 0; i < m5::stl::size(linebreak_std_nopad_table); ++i) {
            auto& e = linebreak_std_nopad_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 5, false, false);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    {  // linebreak, url, padding
        for (uint32_t i = 0; i < m5::stl::size(linebreak_url_padding_table); ++i) {
            auto& e = linebreak_url_padding_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 5, true, true);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }

    {  // linebreak, url, no padding
        for (uint32_t i = 0; i < m5::stl::size(linebreak_url_nopad_table); ++i) {
            auto& e = linebreak_url_nopad_table[i];
            olen    = sizeof(output);

            auto written = encode_base64(output, olen, e.input, e.ilen, 5, true, false);
            EXPECT_EQ(written, e.expected_len) << "idx:" << i;
            EXPECT_TRUE(memcmp(output, e.expected, written) == 0)
                << "idx:" << i << "\nO:[" << output << "]\nE:[" << e.expected << "]";
        }
    }
}
