/**
 * @file 06_RTOS_Callback.ino
 *
 * @brief SnapshotRTOS (callback API).
 *
 * What this shows:
 *  → No Reader struct at all — just three callbacks: update, read, ok.
 *  → snapshot::rtos::start_publisher_cb(...) creates & runs the FreeRTOS task.
 *  → A consumer polls the bus, converts float→bit, and prints ONLY changes.
 */

#include <Arduino.h>
#include <SnapshotBus.h>
#include <SnapshotRTOS.h>
#include <InputModel.h>

// ---- Config ---- //

#define BTN_PIN 6 ///< Wire a button from GPIO to GND (active-LOW).
#define NCH 1     ///< One logical input.

// ---- Frame + Bus ---- //

/**
 * @brief Frame published by SnapshotRTOS (1 channel).
 */
struct InputFrame
{
    std::array<float, NCH> out{}; ///< Channel 0: 0.0f or 1.0f for button.
    uint64_t stamp_us{0};         ///< µs since boot.
    bool failsafe{false};         ///< true if callbacks report unhealthy state.
};

snapshot::SnapshotBus<InputFrame> g_bus; ///< Publisher writes; consumer reads.

// ---- Callback context ---- //

struct ButtonCtx
{
    int pin;
};

// ---- Callbacks (update/read/ok) ---- //

// Use for debouncing, caching, or polling peripherals. GPIO needs none here.
static void cb_update(void *ctx_void)
{
    (void)ctx_void; ///< No periodic work.
}

// Read one active-LOW GPIO into dst[0] as 0.0f/1.0f. No work if n==0 or dst==nullptr.
static void cb_read(void *ctx_void, float *dst, size_t n)
{
    if (!dst || n == 0)
        return; ///< Guard: nothing to write.
    const ButtonCtx *ctx = static_cast<const ButtonCtx *>(ctx_void);
    const bool pressed = (digitalRead(ctx->pin) == LOW); ///< Active-LOW read.
    dst[0] = pressed ? 1.0f : 0.0f;                      ///< Map to float channel.
}

// Health check callback; return true when the link/driver is OK. GPIO is always OK here.
static bool cb_ok(void *ctx_void)
{
    (void)ctx_void; ///< No health state to query in this demo.
    return true;
}

// ---- Consumer side: bitset + edge detection ---- //

using InputState = snapshot::input::State<NCH>;

/**
 * @brief Consumer: peek latest frame → float→bit → print only changes.
 */
void consumerTask(void *)
{
    InputState prev{};

    for (;;)
    {
        const InputFrame f = g_bus.peek();

        InputState cur{};
        cur.stamp_ms = static_cast<uint32_t>(f.stamp_us / 1000ULL); // µs → ms
        cur.set_button(0, f.out[0] > 0.5f);

        snapshot::input::for_each_edge(prev, cur,
                                       [](size_t, bool pressed, uint32_t t_ms)
                                       {
                                           Serial.printf("[t=%lu ms] Button -> %s\n",
                                                         (unsigned long)t_ms,
                                                         pressed ? "Pressed" : "Released");
                                       });

        prev = cur;
        vTaskDelay(pdMS_TO_TICKS(20)); ///< Poll every 10 ms.
    }
}

// ---- Setup ---- //

void setup()
{
    Serial.begin(115200);
    delay(200);

    pinMode(BTN_PIN, INPUT_PULLUP); ///< Active-LOW.

    snapshot::rtos::Policy pol{}; ///< Configure publish policy.
    pol.epsilon = 0.5f;           ///< Publish only on clear 0↔1 edges.
    pol.min_interval_us = 0;      ///< No heartbeat needed here.

    // Start the publisher task (callback API).
    static ButtonCtx ctx{BTN_PIN};
    snapshot::rtos::start_publisher_cb<InputFrame>(
        g_bus, &ctx, cb_update, cb_read, cb_ok, pol,
        "BtnPubCB", ///< Task name.
        2048,       ///< Stack.
        1,          ///< Priority.
        5           ///< Period_ms: poll every 5 ms.
    );

    // Start the consumer task (prints edges).
    xTaskCreate(consumerTask, "Consumer", 2048, nullptr, 1, nullptr);

    Serial.println("SnapshotRTOS callback (1-button) demo. Press the button!");
}

// ---- Loop (unused; RTOS tasks run instead) ---- //

void loop()
{
    vTaskDelete(nullptr);
}