/**
 * @file Basic_Polling.ino
 *
 * @brief Minimal two-task demo using SnapshotBus.
 */

#include <Arduino.h>
#include <SnapshotBus.h>

#define BUTTON_PIN 6

// Simple snapshot payload.
struct InputState
{
    bool button_pressed{false}; ///< True if button is pressed.
    uint32_t stamp_ms{0};       ///< Timestamp in milliseconds.
};

// Global snapshot channel.
snapshot::SnapshotBus<InputState> g_bus; ///< Global bus.

/**
 * @brief Task that reads a button and publishes snapshots.
 */
void publisherTask(void *)
{
    pinMode(BUTTON_PIN, INPUT_PULLUP); ///< Button on pin 6 (LOW = pressed).

    for (;;)
    {
        InputState s{};                                      ///< New snapshot.
        s.stamp_ms = millis();                               ///< Timestamp now.
        s.button_pressed = (digitalRead(BUTTON_PIN) == LOW); ///< Pressed if pin is LOW.
        g_bus.publish(s);                                    ///< Publish to consumers.
        vTaskDelay(pdMS_TO_TICKS(10));                       ///< Sample every 10 ms.
    }
}

/**
 * @brief Task that polls the snapshot bus and prints updates.
 */
void consumerTask(void *)
{
    Serial.begin(115200);
    vTaskDelay(pdMS_TO_TICKS(200)); ///< Allow Serial to start.

    for (;;)
    {
        InputState s = g_bus.peek(); ///< Get latest stable snapshot.
        Serial.printf("[t=%lu] Button = %s\n",
                      (unsigned long)s.stamp_ms,
                      s.button_pressed ? "Pressed." : "Released.");
        vTaskDelay(pdMS_TO_TICKS(250)); ///< Poll every 250 ms.
    }
}

void setup()
{
    // Start publisher on Core 0.
    xTaskCreatePinnedToCore(publisherTask, "Publisher", 2048, nullptr, 1, nullptr, 0);

    // Start consumer on Core 1.
    xTaskCreatePinnedToCore(consumerTask, "Consumer", 2048, nullptr, 1, nullptr, 1);
}

void loop()
{
    vTaskDelete(nullptr); ///< Not used (RTOS tasks run instead).
}