# SnapshotBus

**Single-writer, multi-reader _latest-state_ channel for RTOS/Arduino (ESP32‑tested).**  
Publish a small POD struct from one task (or ISR), and let any number of tasks read a **consistent** snapshot with minimal overhead.

> Queues are for **streams**. **SnapshotBus** is for the **latest state**.

---

## ✨ What’s in v1.1.0

- **SnapshotBus.h** — lock‑free (seqlock‑style) latest-state transport.
- **InputModel.h** — bitset‑based input model + edge detection helpers.  
  `snapshot::input` (renamed from `snapshot::model`).
- **SnapshotRTOS.h** — tiny FreeRTOS publishers so you don’t write loop code:  
  `snapshot::rtos::start_publisher(...)` and `start_publisher_cb(...)` with `Policy{epsilon, min_interval_us}`.
- **C++14 fallback** for `InputModel.h` (no `if constexpr` required).  
  C++17 is still recommended.

This release unifies naming, tightens docs, and adds minimal examples for **Reader** and **callback** styles.

---

## 🚀 Installation

**Arduino IDE**  
1) Search **SnapshotBus** in Library Manager and install.  
2) Or download a release ZIP → *Sketch → Include Library → Add .ZIP Library…*

**PlatformIO**  
Add to your environment:
```ini
lib_deps = littlemanbuilds/SnapshotBus@^1.1.0
```

**Manual**  
Copy `src/` (headers only) into your project and include the headers you need.

---

## 🧭 When to use SnapshotBus

- You have **one writer** (task or ISR) publishing a small struct.
- Many readers need the **latest consistent** value, not every intermediate value.
- You want **zero allocation**, **header-only**, and **portable** code.

Think: sensor fusion inputs, RC channel states, debounced switches, powertrain state, telemetry snapshots.

---

## ⚡ Quick Start (Core Bus)

```cpp
#include <Arduino.h>
#include <SnapshotBus.h>

// Payload must be trivially copyable (POD)
struct InputState {
  uint32_t buttons_mask;
  uint32_t stamp_ms;
};

snapshot::SnapshotBus<InputState> g_bus;

void publisherTask(void*) {
  pinMode(7, INPUT_PULLUP);
  for (;;) {
    InputState s{0, millis()};
    if (digitalRead(7) == LOW) s.buttons_mask |= (1u << 0);
    g_bus.publish(s);
    vTaskDelay(pdMS_TO_TICKS(10));
  }
}

void consumerTask(void*) {
  for (;;) {
    InputState s = g_bus.peek();   // stable snapshot
    if (s.buttons_mask & (1u << 0)) {
      // react to button press
    }
    vTaskDelay(pdMS_TO_TICKS(10));
  }
}

void setup() {
  xTaskCreate(publisherTask, "pub", 2048, nullptr, 1, nullptr);
  xTaskCreate(consumerTask,  "con", 2048, nullptr, 1, nullptr);
}
void loop() { vTaskDelete(nullptr); }
```

**Why this works:** SnapshotBus uses an even/odd sequence counter (seqlock‑style). Reads retry if a write is in flight, returning a **consistent** frame without locking.

---

## 🧩 InputModel: clean edges & masks (bitset helpers)

`InputModel.h` provides a simple `State<N>` to represent button levels and detect edges without writing bit‑twiddly code.

```cpp
#include <InputModel.h>

using InputState = snapshot::input::State<8>; // 8 logical inputs

// Fill a snapshot (true == pressed)
InputState s{};
s.stamp_ms = millis();
s.set_button(0, true);

// Edges between two snapshots
InputState prev{}, cur{};
// ... fill cur ...
auto rising  = snapshot::input::rising_edges(prev, cur);
auto falling = snapshot::input::falling_edges(prev, cur);

// Iterate only changes
snapshot::input::for_each_edge(prev, cur,
  [](size_t i, bool pressed, uint32_t t_ms){
    // handle edge i at time t_ms
  });
```

**SwitchBank interop:** convert to/from masks.
```cpp
uint32_t mask = cur.mask32<8>();          // N<=32
uint64_t mask64 = cur.mask64<8>();        // N<=64
cur.from_mask32<8>(mask);
cur.from_mask64<8>(mask64);
```

> 🔧 C++14 users: the header includes a no‑`if constexpr` fallback; no flags needed. C++17 still recommended.

---

## 🧵 SnapshotRTOS: stop writing publisher loops

`SnapshotRTOS.h` contains two helpers that build a FreeRTOS task for you and publish to a bus with **epsilon** / **time‑gated** logic.

### Reader API (tiny class with `update/read/ok`)

```cpp
#include <SnapshotBus.h>
#include <SnapshotRTOS.h>
#include <InputModel.h>

#define NCH 1
struct InputFrame {
  std::array<float, NCH> out{}; // 0.0f..1.0f
  uint64_t stamp_us{0};
  bool     failsafe{false};
};

snapshot::SnapshotBus<InputFrame> g_bus;

struct ButtonReader {
  void update() {}
  void read(float* dst, size_t n) {
    if (!dst || n==0) return;
    dst[0] = (digitalRead(4)==LOW) ? 1.0f : 0.0f; // active-LOW
  }
  bool ok() const { return true; }
};

void setup() {
  pinMode(4, INPUT_PULLUP);
  snapshot::rtos::Policy pol{};
  pol.epsilon = 0.5f;        // publish on clear 0↔1 edges
  pol.min_interval_us = 0;   // optional heartbeat

  ButtonReader reader{};
  snapshot::rtos::start_publisher<NCH, InputFrame>(
    g_bus, reader, pol, "BtnPub", 1536, 1, 5);
}
```

### Callback API (no class, just functions)

```cpp
struct ButtonCtx { int pin; };
static void cb_update(void* /*ctx*/) { /* optional */ }
static void cb_read(void* ctx, float* dst, size_t n) {
  if (!dst || n==0) return;
  auto* c = static_cast<ButtonCtx*>(ctx);
  dst[0] = (digitalRead(c->pin)==LOW) ? 1.0f : 0.0f;
}
static bool cb_ok(void* /*ctx*/) { return true; }

void setup() {
  static ButtonCtx ctx{4};
  pinMode(ctx.pin, INPUT_PULLUP);
  snapshot::rtos::Policy pol{0.5f, 0};
  snapshot::rtos::start_publisher_cb<InputFrame>(
    g_bus, &ctx, cb_update, cb_read, cb_ok, pol, "BtnPubCB", 1536, 1, 5);
}
```

### Consuming frames (with InputModel edges)

```cpp
using Bits = snapshot::input::State<NCH>;

void consumerTask(void*) {
  Bits prev{};
  for (;;) {
    const InputFrame f = g_bus.peek();
    Bits cur{};
    cur.stamp_ms = static_cast<uint32_t>(f.stamp_us / 1000ULL);
    for (size_t i=0;i<NCH;++i) cur.set_button(i, f.out[i] > 0.5f);
    snapshot::input::for_each_edge(prev, cur, [](size_t i, bool pressed, uint32_t t_ms){
      Serial.printf("[t=%lu ms] ch%u -> %s\n",
        (unsigned long)t_ms, (unsigned)i, pressed ? "Pressed" : "Released");
    });
    prev = cur;
    vTaskDelay(pdMS_TO_TICKS(20));
  }
}
```

---

## 🔍 API Reference (high‑level)

### `snapshot::SnapshotBus<T>`

- `void publish(const T& frame) noexcept` — write new frame (single writer).
- `template<class F> void publish_inplace(F&& fill) noexcept` — fill `T&` in place.
- `T peek() const noexcept` — stable copy of latest frame.
- `bool try_peek(T& out) const noexcept` — bounded‑spin stable copy (false if not stable in limit).
- `T peek_latest() const noexcept` — zero‑spin best‑effort (may be mid‑write).
- `uint32_t sequence() const noexcept` — current seq (even=stable, odd=writing).
- `bool was_updated_since(uint32_t seq) const noexcept` — check if new stable frame exists.

**Constraints**  
`T` must be **trivially copyable**. One writer per bus. Readers are multi‑core safe.

### `snapshot::input`

- `template<size_t N> struct State` with:
  - `buttons` (bitset), `stamp_ms` (ms since boot).
  - `set_button(id,bool)`, `is_pressed(id)`, `is_released(id)`, `count()`, `mask32/64()`, `from_mask32/64()`.
- Free functions: `rising_edges`, `falling_edges`, `changed_edges`, `for_each_edge`, `to_mask32/64`, `to_bitset`.

### `snapshot::rtos`

- `struct Policy { float epsilon; uint32_t min_interval_us; }`
- `start_publisher<N, Frame>(bus, reader, pol, name, stack_words, prio, period_ms)`
- `start_publisher_cb<Frame>(bus, ctx, update, read, ok, pol, name, stack_words, prio, period_ms)`

**Frame contract**  
`Frame` must have:  
`std::array<float, N> out; uint64_t stamp_us; bool failsafe;`

---

## ⚙️ Configuration (macros)

Define these **before** including `SnapshotBus.h` to customize behavior:

- `SNAPSHOTBUS_SPIN_LIMIT` — bound reader spin count (default 0 = unlimited, typical stable quickly).
- `SNAPSHOTBUS_ASSERT(x)` — assert hook (defaults to `configASSERT` if present, else `assert`).
- `SNAPSHOTBUS_PAUSE()` — hint in spin loops (defaults to Xtensa `nop` on ESP).
- `SNAPSHOTBUS_YIELD()` — reader spin yield hook (defaults to no‑op; uses `taskYIELD()` on FreeRTOS when available).

---

## 🧱 Design & Concurrency Model

- **Latest‑value channel** (depth 1). Writers overwrite the previous frame.
- **Seqlock‑style**: writer flips seq to odd, writes payload, release‑fences, flips seq to even.  
  Readers load seq, copy payload, confirm seq unchanged & even.
- **Readers are fast**; if a write collides, they retry.  
  `peek_latest()` trades strict consistency for zero‑spin reads (best‑effort).

---

## 🧯 Pitfalls & Best Practices

- Don’t put dynamic containers in `T`. Prefer plain arrays/fixed structs.
- Keep frames small (a handful of scalars/arrays). Publish less often than you consume.
- For **ISRs**, prefer `publish_inplace` to minimize copies (and do no blocking work).
- Gate noisy inputs in the **publisher** (epsilon/time in `SnapshotRTOS` or your own logic).

---

## 🔄 Versioning & Migration

- **1.1.0** (this release): introduces `snapshot::input` (renamed from `snapshot::model`), SnapshotRTOS helpers, C++14 fallback in `InputModel.h`. Examples updated.
- **Breaking**: If you previously used `snapshot::model`, rename to `snapshot::input`, and replace `setSnapshotButton(...)` with `set_button(...)` (method on `State`).

---

## 🧰 Toolchain Notes

- **C++17 recommended**. For Arduino/ESP32 + PlatformIO:
  ```ini
  build_unflags = -std=gnu++11 -std=gnu++14
  build_flags   = -std=gnu++17
  ```
- If you must stay on **C++14**, `InputModel.h` includes a tag‑dispatch fallback (no `if constexpr`).

---

## 📜 License

MIT © Little Man Builds

Contributions welcome — please keep Doxygen comments aligned with the interfaces and follow the naming conventions (`snapshot::input`, `snapshot::rtos`).

