/**
 * MIT License
 *
 * @brief FreeRTOS publisher helpers for SnapshotBus (epsilon/time-gated).
 *
 * @file SnapshotRTOS.h
 * @author Little Man Builds (Darren Osborne)
 * @date 2025-10-29
 * @copyrightCopyright (c) 2025 Little Man Builds
 */

#pragma once

#include <array>
#include <cstdint>
#include <tuple>
#include <freertos/FreeRTOS.h>
#include <freertos/task.h>

namespace snapshot::rtos
{
    /**
     * @brief Publish policy for RTOS publishers (epsilon/time gating).
     *
     * @details
     * A frame is published when either:
     *   - Any absolute delta across the N-channel vector exceeds @ref epsilon, or
     *   - @ref min_interval_us time has elapsed since the last publish (if non-zero).
     *
     * Keep gating logic *outside* the bus to preserve SnapshotBus simplicity and reuse.
     */
    struct Policy
    {
        float epsilon{0.0f};         ///< Publish if any |delta| > epsilon (0 → always on change).
        uint32_t min_interval_us{0}; ///< Also publish if elapsed >= this (0 → disabled).
    };

    /**
     * @brief Start a FreeRTOS publisher task using a small Reader object.
     * @tparam N Logical channel count (Frame::out must be float[N]).
     * @tparam Frame Frame type with fields: { std::array<float,N> out; uint64_t stamp_us; bool failsafe; }.
     * @tparam BusT SnapshotBus-like type exposing: void publish(const Frame&).
     * @tparam Reader Reader concept:
     *                  - void update();
     *                  - void read(float* dst, std::size_t n);
     *                  - bool ok() const; → true when link/data is valid (not failsafe)
     * @param bus Destination snapshot bus.
     * @param reader Reader instance (copied into task-local state).
     * @param pol Epsilon/time publish policy.
     * @param task_name FreeRTOS task name.
     * @param stack_words Task stack in 32-bit words.
     * @param prio Task priority.
     * @param period_ms Poll period (ms) for update/read loop.
     *
     * @details
     * Creates a FreeRTOS task that:
     *   1) Calls reader.update()
     *   2) Fills Frame::out via reader.read(out, N)
     *   3) Sets Frame::stamp_us = micros() and Frame::failsafe = !reader.ok()
     *   4) Publishes according to Policy (epsilon/time gate)
     *   5) Sleeps via vTaskDelayUntil(period_ms)
     *
     * The first iteration seeds the bus with an initial frame (so consumers never read “empty”).
     */
    template <std::size_t N, class Frame, class BusT, class Reader>
    inline void start_publisher(BusT &bus,
                                Reader reader,
                                Policy pol,
                                const char *task_name,
                                uint16_t stack_words = 2048,
                                UBaseType_t prio = 1,
                                uint32_t period_ms = 5) noexcept
    {
        struct State
        {
            BusT *b{};
            Reader r{};
            Policy p{};
            std::array<float, N> prev{};
            bool seeded{false};
            uint64_t last_pub{0};
            uint32_t period_ms{5};
        };
        static State S;
        S.b = &bus;
        S.r = reader;
        S.p = pol;
        S.prev = std::array<float, N>{};
        S.seeded = false;
        S.last_pub = 0;
        S.period_ms = period_ms;

        auto task_fn = +[](void *arg)
        {
            auto *s = static_cast<State *>(arg);
            TickType_t last = xTaskGetTickCount();

            // Seed (guarantee initial data on the bus).
            {
                s->r.update();
                Frame f{};
                s->r.read(f.out.data(), N);
                f.stamp_us = static_cast<uint64_t>(micros());
                f.failsafe = !s->r.ok();
                s->b->publish(f);
                s->prev = f.out;
                s->last_pub = f.stamp_us;
                s->seeded = true;
            }

            for (;;)
            {
                s->r.update();

                Frame f{};
                s->r.read(f.out.data(), N);
                const uint64_t now = static_cast<uint64_t>(micros());

                bool changed = (s->p.epsilon <= 0.0f);
                if (!changed)
                {
                    for (std::size_t i = 0; i < N; ++i)
                    {
                        const float d = f.out[i] - s->prev[i];
                        if (d > s->p.epsilon || d < -s->p.epsilon)
                        {
                            changed = true;
                            break;
                        }
                    }
                }

                const bool time_gate = s->p.min_interval_us &&
                                       (now - s->last_pub >= s->p.min_interval_us);

                f.stamp_us = now;
                f.failsafe = !s->r.ok();

                if (changed || time_gate)
                {
                    s->b->publish(f);
                    s->prev = f.out;
                    s->last_pub = now;
                    s->seeded = true;
                }

                vTaskDelayUntil(&last, pdMS_TO_TICKS(s->period_ms));
            }
        };

        xTaskCreate(task_fn, task_name, stack_words, &S, prio, nullptr);
    }

    /**
     * @brief Start a FreeRTOS publisher task using callbacks instead of a Reader type.
     *
     * @tparam Frame Has: .out (std::array<float,N>), .stamp_us (uint64_t), .failsafe (bool).
     * @tparam BusT SnapshotBus-like with publish(const Frame&).
     * @param bus Destination snapshot bus.
     * @param ctx Opaque user pointer passed to callbacks.
     * @param update void(*)(void*) → poll hardware/driver.
     * @param read void(*)(void*,float*,std::size_t) → fill out[N].
     * @param ok bool(*)(void*) → link OK (not failsafe).
     * @param pol Epsilon/time publish policy.
     * @param task_name FreeRTOS task name.
     * @param stack_words Task stack in 32-bit words.
     * @param prio Task priority.
     * @param period_ms Poll period (ms) for update/read loop.
     *
     * @details
     * N is inferred from Frame::out element count at compile-time. Behavior is identical
     * to start_publisher for seeding, epsilon/time gating, and publishing.
     */
    template <class Frame, class BusT>
    inline void start_publisher_cb(BusT &bus,
                                   void *ctx,
                                   void (*update)(void *),
                                   void (*read)(void *, float *, std::size_t),
                                   bool (*ok)(void *),
                                   Policy pol,
                                   const char *task_name,
                                   uint16_t stack_words = 2048,
                                   UBaseType_t prio = 1,
                                   uint32_t period_ms = 5) noexcept
    {
        constexpr std::size_t N = std::tuple_size<decltype(Frame{}.out)>::value;

        struct State
        {
            BusT *b{};
            void *ctx{};
            void (*update)(void *){};
            void (*read)(void *, float *, std::size_t){};
            bool (*ok)(void *){};
            Policy p{};
            std::array<float, N> prev{};
            bool seeded{false};
            uint64_t last_pub{0};
            uint32_t period_ms{5};
        };
        static State S;
        S.b = &bus;
        S.ctx = ctx;
        S.update = update;
        S.read = read;
        S.ok = ok;
        S.p = pol;
        S.prev = std::array<float, N>{};
        S.seeded = false;
        S.last_pub = 0;
        S.period_ms = period_ms;

        auto task_fn = +[](void *arg)
        {
            auto *s = static_cast<State *>(arg);
            TickType_t last = xTaskGetTickCount();

            // Seed (guarantee initial data on the bus).
            {
                Frame f{};
                s->update(s->ctx);
                s->read(s->ctx, f.out.data(), N);
                f.stamp_us = static_cast<uint64_t>(micros());
                f.failsafe = !s->ok(s->ctx);
                s->b->publish(f);
                s->prev = f.out;
                s->last_pub = f.stamp_us;
                s->seeded = true;
            }

            for (;;)
            {
                Frame f{};
                s->update(s->ctx);
                s->read(s->ctx, f.out.data(), N);
                const uint64_t now = static_cast<uint64_t>(micros());

                bool changed = (s->p.epsilon <= 0.0f);
                if (!changed)
                {
                    for (std::size_t i = 0; i < N; ++i)
                    {
                        const float d = f.out[i] - s->prev[i];
                        if (d > s->p.epsilon || d < -s->p.epsilon)
                        {
                            changed = true;
                            break;
                        }
                    }
                }

                const bool time_gate = s->p.min_interval_us &&
                                       (now - s->last_pub >= s->p.min_interval_us);

                f.stamp_us = now;
                f.failsafe = !s->ok(s->ctx);

                if (changed || time_gate)
                {
                    s->b->publish(f);
                    s->prev = f.out;
                    s->last_pub = now;
                    s->seeded = true;
                }

                vTaskDelayUntil(&last, pdMS_TO_TICKS(s->period_ms));
            }
        };

        xTaskCreate(task_fn, task_name, stack_words, &S, prio, nullptr);
    }

} ///< Namespace snapshot::rtos.