/**
 * MIT License
 *
 * @brief Single-writer, multi-reader seqlock-style snapshot channel (header-only).
 *
 * @file SnapshotBus.h
 * @author Little Man Builds (Darren Osborne)
 * @date 2025-09-12
 * @copyright Copyright (c) 2025 Little Man Builds
 */

#pragma once

#include <atomic>
#include <cstdint>
#include <type_traits>
#include <cassert>

// ---- Tunables ---- //

/**
 * @brief Overrideable spin limit for bounded peek loops (0 = unlimited).
 *
 * Define before including this header to change default behaviour.
 */
#ifndef SNAPSHOTBUS_SPIN_LIMIT
#define SNAPSHOTBUS_SPIN_LIMIT 0 ///< 0 = unlimited spins (wait until writer finishes).
#endif

/**
 * @brief Assertion macro that prefers FreeRTOS configASSERT if available.
 */
#ifndef SNAPSHOTBUS_ASSERT
#if defined(configASSERT)
#define SNAPSHOTBUS_ASSERT(x) configASSERT(x)
#else
#define SNAPSHOTBUS_ASSERT(x) assert(x)
#endif
#endif

/**
 * @brief Tiny pause hint inside tight spin loops. Overridable per-arch.
 */
#ifndef SNAPSHOTBUS_PAUSE
#if defined(__xtensa__) || defined(ESP_PLATFORM)
#define SNAPSHOTBUS_PAUSE() asm volatile("nop")
#else
#define SNAPSHOTBUS_PAUSE() \
    do                      \
    {                       \
    } while (0)
#endif
#endif

/**
 * @brief Yield hint for RTOS targets (defaults to no-op if unavailable).
 */
#ifndef SNAPSHOTBUS_YIELD
#if defined(ESP_PLATFORM) && defined(FREERTOS_CONFIG_H)
#include <freertos/FreeRTOS.h>
#include <freertos/task.h>
#define SNAPSHOTBUS_YIELD() taskYIELD()
#else
#define SNAPSHOTBUS_YIELD() \
    do                      \
    {                       \
    } while (0)
#endif
#endif

// ------------------ //

namespace snapshot
{
    /**
     * @brief Lock-free, wait-free-for-readers snapshot channel using an even/odd sequence.
     * @tparam T Trivially copyable payload type (POD).
     */
    template <class T>
    class SnapshotBus
    {
        static_assert(std::is_trivially_copyable<T>::value,
                      "SnapshotBus<T>: T must be trivially copyable (POD).");

    public:
        /// @brief Compile-time spin limit mirror of SNAPSHOTBUS_SPIN_LIMIT.
        static constexpr std::uint32_t kSpinLimit =
            static_cast<std::uint32_t>(SNAPSHOTBUS_SPIN_LIMIT);

        /// @brief Default constructor.
        constexpr SnapshotBus() noexcept = default;

        /// @brief Destructor.
        ~SnapshotBus() = default;

        SnapshotBus(const SnapshotBus &) = delete;
        SnapshotBus &operator=(const SnapshotBus &) = delete;
        SnapshotBus(SnapshotBus &&) = delete;
        SnapshotBus &operator=(SnapshotBus &&) = delete;

        // ---- Writer API ---- //

        /**
         * @brief Publish a new snapshot by copying v into the channel.
         * @param v New payload to publish.
         */
        inline void publish(const T &v) noexcept
        {
            // Begin write: flip to odd (writer in-flight).
            seq_.fetch_add(1u, std::memory_order_relaxed);

            // Copy payload (trivially copyable → simple assignment is fine).
            value_ = v;

            // Ensure payload writes are visible before marking stable.
            std::atomic_thread_fence(std::memory_order_release);

            // End write: flip to even (stable).
            seq_.fetch_add(1u, std::memory_order_release);
        }

        /**
         * @brief In-place publish to avoid an extra copy.
         * @tparam F Callable: void(T& out).
         * @param fill Fills value_ directly.
         */
        template <class F>
        inline void publish_inplace(F &&fill) noexcept
        {
            seq_.fetch_add(1u, std::memory_order_relaxed);
            fill(value_);
            std::atomic_thread_fence(std::memory_order_release);
            seq_.fetch_add(1u, std::memory_order_release);
        }

        // ---- Reader API ---- //

        /**
         * @brief Consistent snapshot; spins until a stable frame is observed (or returns best-effort if bounded).
         * @return A consistent snapshot of T when successful; if a spin limit is configured and exceeded,
         *         returns the best-effort latest value.
         */
        [[nodiscard]] inline T peek() const noexcept
        {
            for (std::uint32_t spins = 0;; ++spins)
            {
                const std::uint32_t a = seq_.load(std::memory_order_acquire);

                // Writer in-flight → wait/retry.
                if ((a & 1u) != 0u)
                {
                    SNAPSHOTBUS_PAUSE();
                    SNAPSHOTBUS_YIELD();
#if SNAPSHOTBUS_SPIN_LIMIT > 0
                    if (spins >= kSpinLimit)
                        return value_; ///< Best-effort.
#endif
                    continue;
                }

                // Provisional read.
                T v = value_;

                // Re-check sequence for stability.
                std::atomic_thread_fence(std::memory_order_acquire);
                const std::uint32_t b = seq_.load(std::memory_order_acquire);

                if (a == b && (b & 1u) == 0u)
                    return v; ///< Stable.

                // Not stable → retry or bail (bounded).
                SNAPSHOTBUS_PAUSE();
                SNAPSHOTBUS_YIELD();
#if SNAPSHOTBUS_SPIN_LIMIT > 0
                if (spins >= kSpinLimit)
                    return v; ///< Best-effort.
#endif
            }
        }

        /**
         * @brief Bounded-spin snapshot into out.
         * @param[out] out Destination for the stable snapshot.
         * @return true If a stable snapshot was captured.
         * @return false If spin limit was reached before a stable read.
         */
        inline bool try_peek(T &out) const noexcept
        {
            std::uint32_t spins = 0;
            for (;;)
            {
                const std::uint32_t a = seq_.load(std::memory_order_acquire);

                if ((a & 1u) != 0u)
                {
                    SNAPSHOTBUS_PAUSE();
                    SNAPSHOTBUS_YIELD();
#if SNAPSHOTBUS_SPIN_LIMIT > 0
                    if (++spins >= kSpinLimit)
                        return false;
#endif
                    continue;
                }

                const T v = value_;
                std::atomic_thread_fence(std::memory_order_acquire);
                const std::uint32_t b = seq_.load(std::memory_order_acquire);

                if (a == b && (b & 1u) == 0u)
                {
                    out = v;
                    return true;
                }

                SNAPSHOTBUS_PAUSE();
                SNAPSHOTBUS_YIELD();
#if SNAPSHOTBUS_SPIN_LIMIT > 0
                if (++spins >= kSpinLimit)
                    return false;
#endif
            }
        }

        /**
         * @brief Zero-spin: returns whatever is currently in value_.
         * @return Latest payload without stability guarantees.
         */
        [[nodiscard]] inline T peek_latest() const noexcept
        {
            return value_;
        }

        // ---- Convenience ---- //

        /**
         * @brief Convenience: capture a snapshot directly into out.
         * @param[out] out Destination for the snapshot (stable or best-effort).
         * @return true if a stable snapshot was read; false if best-effort.
         */
        inline bool peek_into(T &out) const noexcept
        {
#if SNAPSHOTBUS_SPIN_LIMIT > 0
            if (try_peek(out))
                return true;
            // Best-effort if spin limit reached.
            out = value_;
            return false;
#else
            // Unlimited spin → always stable.
            out = peek();
            return true;
#endif
        }

        // ---- Observability / helpers ---- //

        /**
         * @brief Current raw sequence value (even = stable, odd = writer in-flight).
         * @return Current sequence counter.
         */
        [[nodiscard]] inline std::uint32_t sequence() const noexcept
        {
            return seq_.load(std::memory_order_relaxed);
        }

        /**
         * @brief Alias: last observed raw sequence value.
         */
        [[nodiscard]] inline std::uint32_t last_seq() const noexcept
        {
            return sequence();
        }

        /**
         * @brief Whether the underlying atomic is lock-free on this platform.
         */
        [[nodiscard]] static constexpr bool is_lock_free() noexcept
        {
            return std::atomic<std::uint32_t>{}.is_lock_free();
        }

        /**
         * @brief True if a new stable frame has been published since seq.
         * @param seq A previously observed sequence value.
         * @return true If a new stable frame exists.
         * @return false Otherwise.
         */
        [[nodiscard]] inline bool was_updated_since(std::uint32_t seq) const noexcept
        {
            const auto s = seq_.load(std::memory_order_acquire);
            return (s != seq) && ((s & 1u) == 0u);
        }

    private:
#if defined(SNAPSHOTBUS_PAD_CACHELINE)
        alignas(32) T value_{};                                 ///< Payload storage (optionally cacheline-aligned).
        alignas(32) mutable std::atomic<std::uint32_t> seq_{0}; ///< Even/odd sequence counter.
#else
        alignas(alignof(T)) T value_{};             ///< Payload storage.
        mutable std::atomic<std::uint32_t> seq_{0}; ///< Even/odd sequence counter.
#endif
    };

    // ---- Helper: fetch only when new (header-only) ---- //

    /**
     * @brief Fetch a snapshot only when the bus has advanced past seen_seq.
     * @tparam T Payload type of the bus.
     * @param bus Snapshot bus instance.
     * @param seen_seq In/out previously seen sequence (call-site owns this).
     * @param[out] out Destination for the snapshot.
     * @return true if a stable snapshot was captured; false if best-effort or unchanged.
     */
    template <class T>
    inline bool try_peek_new(SnapshotBus<T> &bus, std::uint32_t &seen_seq, T &out) noexcept
    {
        const auto s = bus.sequence();
        if (s == seen_seq)
            return false; ///< Nothing new.

        const bool stable = bus.peek_into(out);
        // Update caller's notion of "last seen" regardless; matches observed progression.
        seen_seq = bus.sequence();
        return stable;
    }

} ///< Namespace snapshot.