/**
 * @file 04_RTOS_Reader.ino
 * @brief SnapshotRTOS (reader API).
 *
 * What this shows:
 *  → A tiny Reader reads ONE button as a float (0.0 / 1.0).
 *  → snapshot::rtos::start_publisher(...) creates & runs the FreeRTOS task for you.
 *  → A consumer polls the bus, turns the float into a bit, and prints ONLY changes.
 */

#include <Arduino.h>
#include <SnapshotBus.h>
#include <SnapshotRTOS.h>
#include <InputModel.h>

// ---- Config ---- //

#define BTN_PIN 4 ///< Wire a button from GPIO to GND (active-LOW).
#define NCH 1     ///< One logical input.

// ---- Frame + Bus ---- //

/**
 * @brief Frame published by SnapshotRTOS (1 channel).
 */
struct InputFrame
{
    std::array<float, NCH> out{}; ///< Channel 0: 0.0f or 1.0f for button.
    uint64_t stamp_us{0};         ///< µs since boot.
    bool failsafe{false};         ///< true if reader reports unhealthy state.
};

snapshot::SnapshotBus<InputFrame> g_bus; ///< Publisher writes; consumer reads.

// ---- Reader ---- //

/**
 * @brief ButtonReader: active-LOW GPIO → float(0/1).
 */
struct ButtonReader
{
    void update() { /* no-op for GPIO */ }

    void read(float *dst, size_t n)
    {
        if (n == 0)
            return;
        const bool pressed = (digitalRead(BTN_PIN) == LOW); ///< Active-LOW.
        dst[0] = pressed ? 1.0f : 0.0f;
    }

    bool ok() const { return true; }
};

// ---- Consumer: bitset + edge detection ---- //

using InputState = snapshot::input::State<NCH>;

/**
 * @brief Consumer: peek latest frame → floats→bit → print only changes.
 */
void consumerTask(void *)
{
    InputState prev{};

    for (;;)
    {
        const InputFrame f = g_bus.peek();

        InputState cur{};
        cur.stamp_ms = static_cast<uint32_t>(f.stamp_us / 1000ULL); ///< µs → ms.
        cur.set_button(0, f.out[0] > 0.5f);

        snapshot::input::for_each_edge(prev, cur,
                                       [](size_t, bool pressed, uint32_t t_ms)
                                       {
                                           Serial.printf("[t=%lu ms] Button -> %s\n",
                                                         (unsigned long)t_ms,
                                                         pressed ? "Pressed" : "Released");
                                       });

        prev = cur;
        vTaskDelay(pdMS_TO_TICKS(20)); ///< Poll every 20 ms.
    }
}

// ---- Setup ---- //

void setup()
{
    Serial.begin(115200);
    delay(200);

    pinMode(BTN_PIN, INPUT_PULLUP); ///< Active-LOW.

    snapshot::rtos::Policy pol{};
    pol.epsilon = 0.5f;      ///< Publish only on clear 0↔1 changes.
    pol.min_interval_us = 0; ///< No heartbeat needed here.

    ButtonReader reader{};
    snapshot::rtos::start_publisher<NCH, InputFrame>(
        g_bus, reader, pol,
        "BtnPub", ///< Task name.
        2048,     ///< Stack.
        1,        ///< Priority.
        5         ///< Period_ms: poll every 5 ms.
    );

    xTaskCreate(consumerTask, "Consumer", 2048, nullptr, 1, nullptr);

    Serial.println("SnapshotRTOS ButtonReader (1-button) demo. Press the button!");
}

// ---- Loop (unused; RTOS tasks run instead) ---- //

void loop()
{
    vTaskDelete(nullptr);
}