/**
 * @file 03_Model_Polling.ino
 * @brief InputModel demo: multi-button snapshot + edge detection.
 *
 * Publisher task samples N buttons into a bitset snapshot (pressed = 1) and publishes it.
 * Consumer task polls the bus and prints only the changes (rising/falling edges).
 */

#include <Arduino.h>
#include <SnapshotBus.h>
#include <InputModel.h>

// ---- User config ---- //
#define NUM_BUTTONS 3 ///< Number of buttons in this demo.
static const int BUTTON_PINS[NUM_BUTTONS] = {
    4, 5, 6 ///< Wire each GPIO to a button that goes to GND (active-LOW).
};

// ---- Types ---- //
using InputState = snapshot::input::State<NUM_BUTTONS>; ///< Bitset snapshot: true = pressed.

// ---- Global bus ---- //
snapshot::SnapshotBus<InputState> g_bus;

/**
 * @brief Task that samples buttons and publishes a bitset snapshot.
 */
void publisherTask(void *)
{
    for (;;)
    {
        InputState s{};        ///< New snapshot (all bits default false).
        s.stamp_ms = millis(); ///< Timestamp now.

        // Read each button (active-LOW) and set its bit.
        for (size_t i = 0; i < NUM_BUTTONS; ++i)
        {
            const bool pressed = (digitalRead(BUTTON_PINS[i]) == LOW);
            s.set_button(i, pressed); // <-- new API (was snapshot::model::setSnapshotButton)
        }

        g_bus.publish(s);              ///< Publish to consumers.
        vTaskDelay(pdMS_TO_TICKS(10)); ///< Sample every 10 ms.
    }
}

/**
 * @brief Task that polls the bus and prints only edges (changes).
 */
void consumerTask(void *)
{
    InputState prev{}; ///< Previous snapshot to compare against.

    for (;;)
    {
        const InputState cur = g_bus.peek(); ///< Latest stable snapshot.

        // Visit each bit that changed and print a friendly message.
        snapshot::input::for_each_edge(prev, cur,
                                       [](size_t index, bool pressed, uint32_t stamp_ms)
                                       {
                                           Serial.printf("[t=%lu] Button %u -> %s\n",
                                                         (unsigned long)stamp_ms,
                                                         (unsigned)index,
                                                         pressed ? "Pressed." : "Released.");
                                       });

        prev = cur;                    ///< Remember for next comparison.
        vTaskDelay(pdMS_TO_TICKS(25)); ///< Poll every 25 ms.
    }
}

void setup()
{
    Serial.begin(115200);
    delay(200); ///< Allow Serial to start.

    // Configure all button pins with internal pull-ups (active-LOW).
    for (size_t i = 0; i < NUM_BUTTONS; ++i)
        pinMode(BUTTON_PINS[i], INPUT_PULLUP);

    // Start tasks (no core pinning to keep it simple).
    xTaskCreate(publisherTask, "Publisher", 2048, nullptr, 1, nullptr);
    xTaskCreate(consumerTask, "Consumer", 2048, nullptr, 1, nullptr);

    Serial.println("InputModel edge-detection demo started.");
    Serial.println("Press/release buttons to see edges.");
}

void loop()
{
    vTaskDelete(nullptr); ///< Not used (RTOS tasks run instead).
}