/**
 * MIT License
 *
 * @brief Input-state model utilities and edge-detection helpers (header-only).
 *
 * @file SnapshotModel.h
 * @author Little Man Builds (Darren Osborne)
 * @date 2025-09-12
 * @copyright Copyright (c) 2025 Little Man Builds
 */

#pragma once

#include <cstddef>
#include <cstdint>
#include <bitset>
#include <type_traits>

namespace snapshot::model
{
    /**
     * @brief Snapshot of level-based inputs at a moment in time.
     *
     * @tparam N Number of logical inputs.
     */
    template <std::size_t N>
    struct State
    {
        std::bitset<N> buttons{};  ///< true = pressed.
        std::uint32_t stamp_ms{0}; ///< Sample time in milliseconds.
    };

    /**
     * @brief Index helper that works with enums or integrals.
     *
     * @tparam T Enum or integral type.
     * @param v Value to convert to index.
     * @return std::size_t Zero-based index.
     */
    template <class T>
    [[nodiscard]] constexpr std::size_t idx(T v) noexcept { return static_cast<std::size_t>(v); }

    // ---- Snapshot helpers (pure, no I/O) ---- //

    /**
     * @brief Check if a button is pressed in the snapshot.
     *
     * @tparam N Number of logical inputs.
     * @tparam Id Enum or integral button id.
     * @param s Snapshot state.
     * @param id Button id.
     * @return true If pressed.
     * @return false Otherwise.
     */
    template <std::size_t N, class Id>
    [[nodiscard]] inline bool snapshotPressed(const State<N> &s, Id id) noexcept
    {
        return s.buttons.test(idx(id));
    }

    /**
     * @brief Check if a button is released in the snapshot.
     *
     * @tparam N Number of logical inputs.
     * @tparam Id Enum or integral button id.
     * @param s Snapshot state.
     * @param id Button id.
     * @return true If released.
     * @return false Otherwise.
     */
    template <std::size_t N, class Id>
    [[nodiscard]] inline bool snapshotReleased(const State<N> &s, Id id) noexcept
    {
        return !s.buttons.test(idx(id));
    }

    /**
     * @brief Set a button bit in the snapshot.
     *
     * @tparam N Number of logical inputs.
     * @tparam Id Enum or integral button id.
     * @param s Snapshot state (mutable).
     * @param id Button id.
     * @param is_pressed New level to set.
     */
    template <std::size_t N, class Id>
    inline void setSnapshotButton(State<N> &s, Id id, bool is_pressed) noexcept
    {
        s.buttons.set(idx(id), is_pressed);
    }

    // ---- Edge/mask utilities (pure, no I/O) ---- //

    /**
     * @brief 0->1 transitions between two snapshots.
     */
    template <std::size_t N>
    [[nodiscard]] inline std::bitset<N> rising_edges(const State<N> &prev,
                                                     const State<N> &cur) noexcept
    {
        return (~prev.buttons) & cur.buttons;
    }

    /**
     * @brief 1->0 transitions between two snapshots.
     */
    template <std::size_t N>
    [[nodiscard]] inline std::bitset<N> falling_edges(const State<N> &prev,
                                                      const State<N> &cur) noexcept
    {
        return prev.buttons & (~cur.buttons);
    }

    /**
     * @brief Any bit that changed between two snapshots.
     */
    template <std::size_t N>
    [[nodiscard]] inline std::bitset<N> changed_edges(const State<N> &prev,
                                                      const State<N> &cur) noexcept
    {
        return prev.buttons ^ cur.buttons;
    }

    // ---- Compact mask variants when N <= 32 (handy for telemetry) ---- //

    /**
     * @brief Rising edge mask as 32-bit value (N <= 32).
     */
    template <std::size_t N>
    [[nodiscard]] inline typename std::enable_if<(N <= 32), std::uint32_t>::type
    rising_mask32(const State<N> &p, const State<N> &c) noexcept
    {
        return ((~p.buttons) & c.buttons).to_ulong();
    }

    /**
     * @brief Falling edge mask as 32-bit value (N <= 32).
     */
    template <std::size_t N>
    [[nodiscard]] inline typename std::enable_if<(N <= 32), std::uint32_t>::type
    falling_mask32(const State<N> &p, const State<N> &c) noexcept
    {
        return (p.buttons & (~c.buttons)).to_ulong();
    }

    /**
     * @brief Changed bit mask as 32-bit value (N <= 32).
     */
    template <std::size_t N>
    [[nodiscard]] inline typename std::enable_if<(N <= 32), std::uint32_t>::type
    changed_mask32(const State<N> &p, const State<N> &c) noexcept
    {
        return (p.buttons ^ c.buttons).to_ulong();
    }

    /**
     * @brief Visit each bit that changed between @p prev and @p cur.
     *
     * @tparam N Number of logical inputs.
     * @tparam F Callable: void(std::size_t index, bool pressed, std::uint32_t stamp_ms).
     * @param prev Previous snapshot.
     * @param cur Current snapshot.
     * @param on_edge Invoked for each changed bit.
     */
    template <std::size_t N, class F>
    inline void for_each_edge(const State<N> &prev,
                              const State<N> &cur,
                              F &&on_edge) noexcept
    {
        const auto changed = changed_edges(prev, cur);
        if (!changed.any())
            return;

        for (std::size_t i = 0; i < N; ++i)
        {
            if (changed.test(i))
            {
                const bool pressed = cur.buttons.test(i);
                on_edge(i, pressed, cur.stamp_ms);
            }
        }
    }

} ///< Namespace snapshot::model.