# SnapshotBus

**Single-writer, multi-reader _latest-state_ channel for RTOS/Arduino (ESP32-tested).**  
Publish a small POD struct from one task (or ISR), and let any number of tasks read a **consistent** snapshot with near-zero overhead.

> Queues are for **streams**. **SnapshotBus** is for the **latest state**.

---

## ✨ Features

- Single-writer / multi-reader snapshot (seqlock-style).
- Zero allocations, header-only (drop `src/` in your project).
- Works in tasks or ISRs (tested on ESP32).
- Portable: Arduino & ESP-IDF (C++17 or later).
- Explicit memory-ordering; no hidden locks.
- Tiny footprint: one header + optional model helpers.

---

## 🚀 Installation

**Arduino IDE:** search **SnapshotBus** in Library Manager.  
**Manual:** download the release ZIP → *Sketch → Include Library → Add .ZIP Library…*  
**PlatformIO:** Search libraries and add to `lib_deps` or install from ZIP.

---

## ⚡ Quick Start

```cpp
#include <Arduino.h>
#include <SnapshotBus.h>

// Define your payload (must be trivially copyable)
struct InputState {
  uint32_t buttons_mask;
  uint32_t stamp_ms;
};

// Create a global bus
snapshot::SnapshotBus<InputState> g_bus;

void publisherTask(void*) {
  for (;;) {
    InputState s{0, millis()};
    if (digitalRead(7) == LOW) {
      s.buttons_mask |= (1u << 0);
    }
    g_bus.publish(s);
    vTaskDelay(pdMS_TO_TICKS(10));
  }
}

void consumerTask(void*) {
  for (;;) {
    InputState s = g_bus.peek();
    if (s.buttons_mask & (1u << 0)) {
      // react to button press
    }
    vTaskDelay(pdMS_TO_TICKS(10));
  }
}

void setup() {
  pinMode(7, INPUT_PULLUP);
  xTaskCreatePinnedToCore(publisherTask, "pub", 2048, nullptr, 1, nullptr, 0);
  xTaskCreatePinnedToCore(consumerTask,  "con", 2048, nullptr, 1, nullptr, 0);
}

void loop() { vTaskDelete(nullptr); }
```

---

## 🛠️ Design Notes

- **Single publisher** per bus. Readers may run on any core.
- Payload `T` **must be trivially copyable** (POD). Avoid `std::string`, `std::vector`, etc.
- Uses release/acquire fences (seqlock pattern).  
  Readers never block — they retry if a write is in-flight.

---

## 📂 Examples

- `examples/Basic_Polling` — Minimal two-task demo.
- `examples/ISR_Publisher` — ESP32 timer ISR publishes; task consumes.
- `examples/Model_Polling` — Multi-button snapshot + edge detection using `SnapshotModel`.

---

## ⚙️ Configuration

Optional macros (define before including `SnapshotBus.h`):

- `SNAPSHOTBUS_SPIN_LIMIT` — Max spins for stabilization in `peek/try_peek` (0 = unlimited, default).
- `SNAPSHOTBUS_ASSERT(x)` — Route asserts (defaults to `configASSERT` if available, else `assert`).
- `SNAPSHOTBUS_PAUSE()` — Architecture hint inside spin loops (defaults to `nop` on Xtensa).
- `SNAPSHOTBUS_YIELD()` — Yield hook (defaults to no-op; maps to `taskYIELD()` on FreeRTOS).

---

## 📜 License

MIT © Little Man Builds

Contributions welcome! Please keep Doxygen comments consistent with the interface and follow the established naming conventions.
