/**
 * @file ISR_Publisher.ino
 * @brief ESP32 hardware timer publishes snapshots from an ISR.
 *
 * A hardware timer (every 100 ms) triggers an ISR that increments a tick counter
 * and publishes it to a SnapshotBus. A FreeRTOS task polls the bus and prints the
 * latest tick + timestamp.
 */

#if !defined(ESP32)
#error "This example requires ESP32."
#endif

#include <Arduino.h>
#include <SnapshotBus.h>

// ---- Config ---- //
#define TIMER_PERIOD_US 100000 ///< Timer period in microseconds (100,000 µs = 100 ms).

// ---- Snapshot payload ---- //
struct TickState
{
  uint32_t tick{0};     ///< Incrementing tick count.
  uint32_t stamp_ms{0}; ///< Timestamp in milliseconds when published.
};

// ---- Global bus and timer handle ---- //
snapshot::SnapshotBus<TickState> g_bus; ///< Snapshot channel shared by ISR and task.
hw_timer_t *g_timer = nullptr;          ///< Hardware timer instance.

// ---- ISR: publishes a new snapshot each time the timer fires ---- //
void IRAM_ATTR onTimer()
{
  static uint32_t t = 0; ///< Local counter that only the ISR updates.
  TickState s{};
  s.tick = ++t;          ///< Next tick value.
  s.stamp_ms = millis(); ///< Stamp when the ISR ran (ms).

  // Publish directly from ISR. For small trivially-copyable payloads.
  g_bus.publish(s);
}

// ---- Consumer task: reads and prints snapshots ---- //
/**
 * @brief Task that polls the snapshot bus and prints updates.
 */
void consumerTask(void *)
{
  for (;;)
  {
    TickState s = g_bus.peek(); ///< Get latest stable snapshot.
    Serial.printf("tick=%lu @ %lums\n",
                  (unsigned long)s.tick,
                  (unsigned long)s.stamp_ms);
    vTaskDelay(pdMS_TO_TICKS(200)); ///< Print every 200 ms.
  }
}

// ---- Arduino setup/loop ---- //
void setup()
{
  Serial.begin(115200);
  delay(200); ///< Allow Serial to start.

  // Configure and start hardware timer:
  // divider = 80 → 1 µs per timer tick (80 MHz APB / 80).
  g_timer = timerBegin(/*timer*/ 0, /*divider*/ 80, /*countUp*/ true);
  timerAttachInterrupt(g_timer, &onTimer, /*edge*/ true);
  timerAlarmWrite(g_timer, TIMER_PERIOD_US, /*autoReload*/ true);
  timerAlarmEnable(g_timer);

  // Start the consumer task (no core pinning to keep it simple).
  xTaskCreate(consumerTask, "Consumer", 2048, nullptr, 1, nullptr);

  Serial.println("ISR publisher demo started (100 ms timer).");
}

void loop()
{
  vTaskDelete(nullptr); ///< Not used (RTOS task runs instead).
}