/**
 * MIT License
 *
 * @brief Tiny JSON loader for RcConfig using ArduinoJson.
 *
 * @file Json.hpp
 * @author Little Man Builds (Darren Osborne)
 * @date 2025-10-03
 * @copyright © 2025 Little Man Builds
 */

#pragma once

#include <ArduinoJson.h>
#include <cstdint>
#include <cstddef>
#include <cstring>
#include <utility>
#include <Types.hpp>
#include <Config.hpp>

namespace rc
{
    // --------------------------------------------------------------------
    // Include order requirement:
    //
    // This header relies on helpers generated by RC_DECLARE_ROLES(Enum, ...):
    //   - rc::to_string(Enum)  -> const char*
    //   - rc::role_count(Enum) -> std::size_t
    //
    // Roles must be declared BEFORE including this file, e.g.:
    //
    //   #define MY_ROLES(_)...; RC_DECLARE_ROLES(MyEnum, MY_ROLES)
    //   #include <Json.hpp>
    //
    // The guard below produces a compile error if that order is violated.
    // --------------------------------------------------------------------

    template <typename E>
    inline void require_role_helpers()
    {
        using ToStringFn = const char *(*)(E);
        // Check that an overload with signature `const char*(E)` exists:
        (void)sizeof((ToStringFn)&rc::to_string);
        // Check that role_count(E) is callable and forms a complete type:
        (void)sizeof(rc::role_count(std::declval<E>()));
    }

    /// @brief Look up enum value by role name via RC_DECLARE_ROLES helpers.
    template <typename E>
    inline bool role_from_name(const char *name, E &out)
    {
        require_role_helpers<E>();

        if (!name)
            return false;

        const std::size_t n = rc::role_count(E{});
        for (std::size_t i = 0; i < n; ++i)
        {
            const E e = static_cast<E>(i);
            const char *rn = rc::to_string(e);
            if (rn && std::strcmp(rn, name) == 0)
            {
                out = e;
                return true;
            }
        }
        return false;
    }

    /**
     * @brief Load RcConfig from a JSON string.
     * @code
     * // Schema:
     * {
     *   "map": { "RoleName": 0, ... },
     *   "axes": {
     *     "RoleName": {
     *       "raw": [lo, hi, center],
     *       "deadband_us": 0,
     *       "out": [lo, hi],
     *       "expo": 0.0,
     *       "invert": true
     *     }
     *   },
     *   "switches": {
     *     "RoleName": {
     *       "values": [0,1,2],
     *       "raw_levels": [1000,1500,2000],
     *       "auto_levels": true,
     *       "hyst_us": 60,
     *       "learn_alpha": 0.2
     *     }
     *   }
     * }
     * @endcode
     */
    template <typename E>
    bool load_json(E /*tag*/, RcConfig<E> &cfg, const char *json)
    {
        require_role_helpers<E>();

        if (!json)
            return false;

        JsonDocument doc;
        DeserializationError err = deserializeJson(doc, json);
        if (err)
            return false;

        // ---- Map ---- //
        if (doc["map"].is<JsonObject>())
        {
            JsonObject map = doc["map"].as<JsonObject>();
            for (JsonPair kvp : map)
            {
                const char *roleName = kvp.key().c_str();
                int ch = kvp.value().as<int>();
                if (ch < 0)
                    ch = 0;
                if (ch > 255)
                    ch = 255;

                E role{};
                if (role_from_name(roleName, role))
                    cfg.map(role, static_cast<std::uint8_t>(ch));
            }
        }

        // ---- Axes ---- //
        if (doc["axes"].is<JsonObject>())
        {
            JsonObject axes = doc["axes"].as<JsonObject>();
            for (JsonPair kvp : axes)
            {
                const char *roleName = kvp.key().c_str();
                E role{};
                if (!role_from_name(roleName, role))
                    continue;

                auto b = cfg.axis(role);
                JsonObject a = kvp.value().as<JsonObject>();

                if (a["raw"].is<JsonArray>())
                {
                    JsonArray ra = a["raw"].as<JsonArray>();
                    if (ra.size() >= 2)
                    {
                        const int lo = ra[0].as<int>();
                        const int hi = ra[1].as<int>();
                        const int ct = (ra.size() >= 3) ? ra[2].as<int>() : ((lo + hi) / 2);
                        b.raw(static_cast<std::int16_t>(lo),
                              static_cast<std::int16_t>(hi),
                              static_cast<std::int16_t>(ct));
                    }
                }

                if (a["deadband_us"].is<int>())
                    b.deadband_us(static_cast<std::int16_t>(a["deadband_us"].as<int>()));

                if (a["out"].is<JsonArray>())
                {
                    JsonArray out = a["out"].as<JsonArray>();
                    if (out.size() >= 2)
                        b.out(out[0].as<float>(), out[1].as<float>());
                }

                if (a["expo"].is<float>())
                    b.expo(a["expo"].as<float>());

                if (a["invert"].is<bool>())
                    b.invert(a["invert"].as<bool>());

                (void)b.done();
            }
        }

        // ---- Switches ---- //
        if (doc["switches"].is<JsonObject>())
        {
            JsonObject sws = doc["switches"].as<JsonObject>();
            for (JsonPair kvp : sws)
            {
                const char *roleName = kvp.key().c_str();
                E role{};
                if (!role_from_name(roleName, role))
                    continue;

                JsonObject s = kvp.value().as<JsonObject>();
                auto b = cfg.sw(role);

                if (s["values"].is<JsonArray>())
                {
                    float v[kRcMaxSwitchVals]{};
                    std::uint8_t n = 0;
                    for (JsonVariant it : s["values"].as<JsonArray>())
                    {
                        if (n >= kRcMaxSwitchVals)
                            break;
                        v[n++] = it.as<float>();
                    }
                    switch (n)
                    {
                    default:
                    case 0:
                        b.values({});
                        break;
                    case 1:
                        b.values({v[0]});
                        break;
                    case 2:
                        b.values({v[0], v[1]});
                        break;
                    case 3:
                        b.values({v[0], v[1], v[2]});
                        break;
                    case 4:
                        b.values({v[0], v[1], v[2], v[3]});
                        break;
                    case 5:
                        b.values({v[0], v[1], v[2], v[3], v[4]});
                        break;
                    case 6:
                        b.values({v[0], v[1], v[2], v[3], v[4], v[5]});
                        break;
                    case 7:
                        b.values({v[0], v[1], v[2], v[3], v[4], v[5], v[6]});
                        break;
                    case 8:
                        b.values({v[0], v[1], v[2], v[3], v[4], v[5], v[6], v[7]});
                        break;
                    }
                }

                if (s["raw_levels"].is<JsonArray>())
                {
                    std::int16_t lv[kRcMaxSwitchVals]{};
                    std::uint8_t n = 0;
                    for (JsonVariant it : s["raw_levels"].as<JsonArray>())
                    {
                        if (n >= kRcMaxSwitchVals)
                            break;
                        lv[n++] = static_cast<std::int16_t>(it.as<int>());
                    }
                    switch (n)
                    {
                    default:
                    case 0:
                        break;
                    case 1:
                        b.raw_levels({lv[0]});
                        break;
                    case 2:
                        b.raw_levels({lv[0], lv[1]});
                        break;
                    case 3:
                        b.raw_levels({lv[0], lv[1], lv[2]});
                        break;
                    case 4:
                        b.raw_levels({lv[0], lv[1], lv[2], lv[3]});
                        break;
                    case 5:
                        b.raw_levels({lv[0], lv[1], lv[2], lv[3], lv[4]});
                        break;
                    case 6:
                        b.raw_levels({lv[0], lv[1], lv[2], lv[3], lv[4], lv[5]});
                        break;
                    case 7:
                        b.raw_levels({lv[0], lv[1], lv[2], lv[3], lv[4], lv[5], lv[6]});
                        break;
                    case 8:
                        b.raw_levels({lv[0], lv[1], lv[2], lv[3], lv[4], lv[5], lv[6], lv[7]});
                        break;
                    }
                }

                if (s["auto_levels"].is<bool>() || s["hyst_us"].is<int>() || s["learn_alpha"].is<float>())
                {
                    RcChannelSpec &spec = cfg.specs[static_cast<std::size_t>(role)];
                    if (s["auto_levels"].is<bool>())
                        spec.sw.auto_levels = s["auto_levels"].as<bool>();
                    if (s["hyst_us"].is<int>())
                        spec.sw.hyst_us = static_cast<std::uint16_t>(s["hyst_us"].as<int>());
                    if (s["learn_alpha"].is<float>())
                    {
                        float la = s["learn_alpha"].as<float>();
                        if (la < 0.f)
                            la = 0.f;
                        if (la > 1.f)
                            la = 1.f;
                        spec.sw.learn_alpha = la;
                    }
                }

                (void)b.done();
            }
        }

        return true;
    }

} ///< namespace rc.