#include <docFileObj.h>
#include <resizeBuff.h>
#include <timeObj.h>
#include <filePath.h>
#include <strTools.h>

//#include <debug.h>


// A docFile object MUST have a path to exist. 	
docFileObj::docFileObj(const char* filePath) {
		
	docFilePath		= NULL;									// Dynamic things start at NULL.
	editFilePath	= NULL;									// Dynamic things start at NULL.
	returnBuffer	= NULL;									// Someone wants..  Say.. a path? Store it here. Pass this back.
	autoGenFile		= false;									// We assume this is NOT an autoGenerated temp file.
	mode 				= fClosed;								// We are NOT editing at the moment.
	heapStr(&docFilePath,filePath);						// Grab our local copy of the file path.		
}


docFileObj::~docFileObj(void) {

	closeDocFile();					// Close files and Recycle the editFile stuff.
	if (autoGenFile) {				// If this was originally an auto generated file..
		SD.remove(docFilePath);		// Delete the auto file. (This and changFile())
		autoGenFile = false;			// And we are no longer carrying an auto generated file.
	}
	freeStr(&docFilePath);			// Recycle the docFilePath.
	freeStr(&editFilePath);			// Recycle the editFilePath.
	freeStr(&returnBuffer);			// And the return buffer.
}


// Called by user to open a document.
//
// If opening for reading, its left open for reading. User MUST call close on it when
// finished.
// 
// If open for editing, this creates a clone of the doc file for editing and leaves that
// open to be used by the user for editing. And of course the user must call closed on it
// when finished. But, before that, saveDocFile() needs to be called for saving changes to
// the original file.
bool docFileObj::openDocFile(fOpenMode openMode) {

	File	tempFile;
	bool	success;

	success = false;																// Not a success yet..
	if (docFilePath) {															// If we have a non-NULL path..
		if (openMode==fWrite) {											// If they want to edit this file..
			if (mode==fOpenToEdit || mode==fEdited) {						// If already open for editing.
				ourFile.seek(ourFile.size());									// Opening a file to WRITE assumes you start at the end.
				return true;														// Actually nothing else to do, return success!
			}
			if (mode==fOpenToRead) {											// If currently open to read..
				closeDocFile();													// We close the file to open later.
			}
			tempFile = SD.open(docFilePath,FILE_READ);						// Have a go at opening the doc file path.
			if (tempFile) {														// If we were able to open the file..
				if (checkDoc(tempFile)) {										// If the file passed the acid test..
					if (createEditPath()) {										// If we were able to create an edit path..
						ourFile = SD.open(editFilePath,FILE_WRITE);		// Have a go at opening the edit file path.	
						if (ourFile) {												// If we were able to open the edit file..
							ourFile.close();										// Close the file. We know we can open it.
							ourFile = TRUNCATE_FILE(editFilePath);
							fcpy(ourFile,tempFile);								// Copy the original to the new editing file.
							mode = fOpenToEdit;									// We are open for editing.
							success = true;										// Success!! The edit file is open and ready to use.
						}
					}
				}
				tempFile.close();													// Close the doc file.
			}
		} else if (openMode==fRead) {											// If they just want to read this file..
			if (mode==fOpenToRead) {											// If we're already open to read..
				ourFile.seek(0);													// Opening to READ assumes you start at the beginning of the file.
				return true;														// We're all set, lets go.
			} else {
				closeDocFile();													// We close the file.
				ourFile = SD.open(docFilePath,FILE_READ);					// Have a go at opening the doc file path.
				if (ourFile) {														// If we were able to open the file..
					if (checkDoc(ourFile)) {									// If the file passed the acid test..
						ourFile.seek(0);											// Point at first byte of the file.
						mode = fOpenToRead;										// We are open for reading.
						success = true;											// Success! The file is open and ready to read.
					}
				}
			}
		}
	}
	return success;
}


// If we are editing a file, we will save the edited version either to a new file path. If
// one has been passed in. Or if not, to the original doc file. If we are NOT editing AND
// we are passed in a new file path, we'll clone the doc file to this new path. If, in
// this case, no path is passed in? Its kind of.. "Shrug your shoulders and return false."
// The user is obviously just not getting it.
//
// In any case, if the new path is different than the original path, we leave here
// pointing at the new path.
bool docFileObj::saveDocFile(const char* newFilePath) {

	File			tempFile;
	char*			savedPath;
	fileModes	savedMode;
	bool			success;
	bool			changePath;
	
	success = false;														// Not a success yet.
	savedPath = NULL;														// Damn! forgot this last time.
	if (!newFilePath) {													// If we don't have a new file path..
		changePath = false;												// We don't change paths.
	} else if (!strcmp(newFilePath,docFilePath)) {				// If we DO have a new file path, but its the same as ours..
		changePath = false;												// We don't change paths.
	} else {																	// Else, only other choice is different paths..
		changePath = true;												// We DO change paths.
	}																			// Ok that's settled.
	if (!changePath) {													// If we're NOT going to change paths..
		if (mode==fEdited || mode==fOpenToEdit) {					// If we are using an editFile.
			tempFile = SD.open(docFilePath,FILE_WRITE);			// Have a go at opening our (real) file path.
			if (tempFile) {												// If we got the (real) file..
				tempFile.close();											// Close the file. We know we can open it.
				tempFile = TRUNCATE_FILE(docFilePath);				// Now we clear out the contents of the (real) file..
				fcpy(tempFile,ourFile);									// Copy the original editfile to the (real) file.
				tempFile.close();											// Close the (real) file.
				mode=fOpenToEdit;											// Edits have been saved so, mode is just open for edit.
				success = true;											// We are a success!
			}																	//
		} else {																// Else, mode is either closed or read only?
		}																		// We can not save to a closed or read only file.
	} else {																	// Else, we're doing a "saveAs" command.
		if (heapStr(&savedPath,docFilePath)) {						// If we can save off the document's path..
			if (mode==fEdited || mode==fOpenToEdit) {				// If we are using an editFile.
				tempFile = SD.open(newFilePath,FILE_WRITE);		// Have a go at opening our (new) file path.
				if (tempFile) {											// If we got the (new) file..
					tempFile.close();										// Close the file. We know we can open it.
					tempFile = TRUNCATE_FILE(newFilePath);			// Now we clear out the contents of the (new) file..
					fcpy(tempFile,ourFile);								// Copy the original editfile to the (new) file.
					tempFile.close();										// Close the (new) file.
					closeDocFile();										// Close ourselves..
					if (changeDocFile(newFilePath)) {				// Switch to the (new) file we saved.
						if (openDocFile(fWrite)) {						// If we can open the new doc file for writing..
							if (autoGenFile) {							// If this was originally an auto generated file..
								SD.remove(savedPath);					// Delete the auto file.
								autoGenFile = false;						// And we are no longer carrying an auto generated file.
							}													//
							success = true;								// Success!
						}														//
					}															//
				}																//
			} else {															// Else, mode is either closed or read only..
				savedMode = mode;											// Save off the mode we started with.
				if (mode==fClosed) {										// If the file's closed..
					openDocFile(fRead);								// Give it a shot to open it for reading.
				}																//
				if (mode==fOpenToRead) {								// If the file is open to read..
					tempFile = SD.open(newFilePath,FILE_WRITE);	// Have a go at opening our (new) file path.
					if (tempFile) {										// If we got the (new) file..
						tempFile.close();									// Close the file. We know we can open it.
						tempFile = TRUNCATE_FILE(newFilePath);		// Now we clear out the contents of the (new) file..
						fcpy(tempFile,ourFile);							// Copy the original editfile to the (new) file.
						tempFile.close();									// Close the (new) file.
						closeDocFile();									// Close ourselves..
						if (changeDocFile(newFilePath)) {			// If we can switch to the (new) file we saved.
							if (savedMode==fOpenToRead) {				// If the original was open for reading..
								if (openDocFile(fRead)) {			// If we can Open the new doc file for reading..
									success = true;						// We'll call that a success!
								}												//
							} else {											// Else it was originally closed..
								success = true;							// We'll call that a success as well!
							}													//
						}														//
					}															//
				}																//
			}																	//
			freeStr(&savedPath);											// Free the allocated string buffer.
		}																		//
	}																			//
	return success;														// Return the result, success or failure.
}


// This closes the docFile and if there is a temp file associated with it, that is
// deleted. There is no checking if this is ok to do, or if data will be lost. All that is
// up to the user of this object.
void docFileObj::closeDocFile(void) {


	switch(mode) {									// Mode decides our action..
		case fClosed		: return;			// If already closed, bolt.
		case fOpenToRead	:						// If open to read..
			ourFile.close();						// Close the file.
		break;										// And we're done.
		case fOpenToEdit	:						// Open for editing..
		case fEdited		:						// Or open for editing and have changes..
			ourFile.close();						// We close our document file.
			SD.remove(editFilePath);			// We remove our document editing file.
			freeStr(&editFilePath);				// Recycle the edit file path.
		break;										//
	}
	mode = fClosed;								// Mode is now, closed.
}


bool docFileObj::changeDocFile(const char* newPath) {
	
	if (newPath) {									// If we have a new file path. (Sanity, we'd better!)
		closeDocFile();							// Close our current document file.
		if (autoGenFile) {						// If this was originally an auto generated file..
			SD.remove(docFilePath);				// Delete the auto file.
			autoGenFile = false;					// And we are no longer carrying an auto generated file.
		}												//
		heapStr(&docFilePath,newPath);		// Save the new path.
		if (docFilePath) {						// If we got the RAM to store it..
			return true;							// Success!
		}												//					
	}													//
	return false;									// Either no name or not enough RAM to store it.
}



// Save off whether our document has an auto generated name or not.
void docFileObj::setAsAutoGen(bool trueFalse) { autoGenFile = trueFalse; }


// Return if we have unsaved changes or not.
bool docFileObj::fileEdited(void) {  return mode==fEdited; }


// This should 

char* docFileObj::getName(void) {
	
	filePath aPath;
	
	aPath.setPath(docFilePath);
	heapStr(&returnBuffer,aPath.getCurrItemName());
	return returnBuffer;
}


// This should give us the folder path that our file is saved in.
char* docFileObj::getFolder(void) {

	filePath aPath;
	
	aPath.setPath(docFilePath);					// Setup the path obj for this path string.
	aPath.popItem();									// Pop the last file/folder off the string.
	heapStr(&returnBuffer,aPath.getPath());	// Allocate and save off the resultant path string.
	return returnBuffer;								// Return a pinter to the resultant path string.
}


byte docFileObj::peek(void) {

	if (mode != fClosed) {
		return ourFile.peek();
	} else {
		return 0;
	}
}


uint32_t docFileObj::position(void) {
	
	if (mode != fClosed) {
		return ourFile.position();
	} else {
		return 0;
	}
}


bool docFileObj::seek(uint32_t index) {
	
	if (mode != fClosed) {
		return ourFile.seek(index);
	} else {
		return false;
	}
}


uint32_t docFileObj::size(void) {
	
	if (mode != fClosed) {
		return ourFile.size(); 
	} else {
		return 0;
	}
}


int docFileObj::read(void) {

	if (mode != fClosed) {
		return ourFile.read();
	} else {
		return -1;
	}
}


uint16_t docFileObj::read(byte* buff,uint16_t numBytes) {

	if (mode != fClosed) {
		return ourFile.read(buff,numBytes);
	} else {
		return 0;
	}
}


size_t docFileObj::write(uint8_t aByte) {
	
	size_t	result;
	
	result = 0;
	if (mode==fOpenToEdit || mode==fEdited) {
		result = ourFile.write(&aByte,1);
		if (result>0) {
			mode = fEdited;
		}
	}
	return result;
}


size_t docFileObj::write(byte* buff,size_t numBytes) {
		
	size_t result;
	
	result = 0;
	if (mode==fOpenToEdit || mode==fEdited) {
		result = ourFile.write(buff,numBytes);
		if (result>0) {
			mode = fEdited;
		}
	}
	return result;
}	

		
// When the inherited opens up a document, this will be called to see if its parsable as
// the kind of file the class is looking for.
bool docFileObj::checkDoc(File inFile) { return false; }


// Find an unused temp file and assign it to the edit file path.
bool docFileObj::createEditPath(void) {
		
	heapStr(&editFilePath,numberedFilePath(TEMP_FOLDER,"efp",".tmp"));	// Find an unused valid temp file path.
	if (editFilePath) {																	// If we succeeded in this task..
		return true;																		// If we got here, we've succeeded!
	}																							//
	return false;																			// Otherwise, we return false.
}

										
					
