/*
  The MIT License (MIT)

  Copyright (c) 2025 Kay Kasper

  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the “Software”),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included
  in all copies or substantial portions of the Software.

  The Software is provided “as is”, without warranty of any kind, express
  or implied, including but not limited to the warranties of merchantability,
  fitness for a particular purpose and noninfringement. In no event shall
  the authors or copyright holders be liable for any claim, damages or other
  liability, whether in an action of contract, tort or otherwise, arising
  from, out of or in connection with the software or the use or other dealings
  in the Software.
*/

#ifndef GENERAL_BUFFER_EXTENSIONS
#define GENERAL_BUFFER_EXTENSIONS

#include "GeneralBuffer.h"

/*
  Specific FIFO subclass of GeneralBuffer for easier handling
  of first in and first out buffers or queues. The FIFO class
  is defined to work only with predefined external storage,
  which makes the memory handling more predictable and obvious.

  Several functions of the base class GeneralBuffer are declared
  to be privat in the FIFO context, so that they are not allowed
  to be used. Positions are in the same order as in GeneralBuffer.

  The two template types are:
  1. type VAL defines the type of the handled values 
  2. type IND defines the type of the internal pointers and calculations

  @seealso GeneralBuffer
*/
template <typename VAL, typename IND = uint8_t>
class FIFO : public GeneralBuffer<VAL, IND> {
  private:
    VAL getFirstValue(){};
    VAL getLastValue(){};
    bool putFirstValue(VAL value, bool allowOverwrite){};
    bool putLastValue(VAL value, bool allowOverwrite){};
    bool overwriteValueAtPos(VAL value, IND pos){};
    VAL getValueAtPos(IND pos){};
    bool putValueAtPos(VAL value, IND pos, bool overwrite){};
    IND findValueAtPos(VAL refValue, IND startPos, bool forward, uint8_t condition){};

  public:

    /*
      Creates a new FIFO buffer or queue based on an already defined
      and handed over storage.

      The storage must be located somewhere, where it is never (re)moved
      while the FIFO instance still exists. It must be ensured, that no
      reading or writing is done in the storage by others than the here
      defind FIFO instance.
      
      The storage type must be the same as the 1. type VAL of the
      FIFO instance. The storage size is always defining the
      available capacity for storing values (size - 1 = capacity).
      The 2. type IND of the FIFO instance is optional and
      defines the max bufferLen of the buffer.

      @param bufferLen  storage size, values from 2 to 251 (type IND 8 Bit) or 16001 (type IND > 8 Bit) allowed
      @param bufferAddr pointer to the first value of the storage
      @seealso GeneralBuffer(IND bufferLen, VAL* bufferAddr)
    */
    FIFO(IND bufferLen, VAL* bufferAddr) : GeneralBuffer<VAL, IND>(bufferLen, bufferAddr){
    };

    /*
      Returns the first value of the internal sequence (position 1) of stored
      values and removes it from the buffer. Positions, used capacity and
      free capacity will be changed by the call.
      
      If no value is available in the buffer, the result will be 0. Before a call
      of this function, it must be checked, that at least one value is available.
    
      @returns    first value of the sequence, if at least one value was available before, otherwise 0
    */
    VAL getValue(){
      return GeneralBuffer<VAL, IND>::getFirstValue();
    };

    /*
      Adds the value at the end of the internal sequence of stored values.
      Positions, used capacity and free capacity may be changed by the call.
      
      If no free capacity exists for an additional value, there are two options.
      The currently last value can be overwritten and will be lost, but the
      buffer will not overflow, or the given value will not be stored.

      @param value          the value that shall be stored in the buffer
      @param allowOverwrite the information, if overwriting is allowed
      @returns              true, if value could be stored, even with overwriting, otherwise false
    */
    bool putValue(VAL value, bool allowOverwrite){
      return GeneralBuffer<VAL, IND>::putLastValue(value, allowOverwrite);
    };
};


/*
  Specific LIFO subclass of GeneralBuffer for easier handling
  of last in and first out buffers or stacks. The LIFO class
  is defined to work only with predefined external storage,
  which makes the memory handling more predictable and obvious.

  Several functions of the base class GenerelBuffer are declared
  to be privat in the LIFO context, so that they are not allowed
  to be used.
  
  In contrast to the positions of GeneralBuffer and FIFO, the
  positions are reversed: position 1 is the last added value
  and position [usedcapacity] is the oldest value in the stack.

  The two template types are:
  1. type VAL defines the type of the handled values 
  2. type IND defines the type of the internal pointers and calculations

  @seealso GeneralBuffer
  @seealso FIFO
*/
template <typename VAL, typename IND = uint8_t>
class LIFO : public GeneralBuffer<VAL, IND> {
  private:
    VAL getFirstValue(){};
    VAL getLastValue(){};
    bool putFirstValue(VAL value, bool allowOverwrite){};
    bool putLastValue(VAL value, bool allowOverwrite){};
    bool overwriteValueAtPos(VAL value, IND pos){};
    VAL getValueAtPos(IND pos){};
    bool putValueAtPos(VAL value, IND pos, bool overwrite){};
    IND findValueAtPos(VAL refValue, IND startPos, bool forward, uint8_t condition){};

  public:

    /*
      Creates a new LIFO buffer or stack based on an already defined
      and handed over storage.

      The storage must be located somewhere, where it is never (re)moved
      while the LIFO instance still exists. It must be ensured, that no
      reading or writing is done in the storage by others than the here
      defind LIFO instance.
      
      The storage type must be the same as the 1. type VAL of the
      LIFO instance. The storage size is always defining the
      available capacity for storing values (size - 1 = capacity).
      The 2. type IND of the LIFO instance is optional and
      defines the max bufferLen of the buffer.

      @param bufferLen  storage size, values from 2 to 251 (type IND 8 Bit) or 16001 (type IND > 8 Bit) allowed
      @param bufferAddr pointer to the first value of the storage
      @seealso GeneralBuffer(IND bufferLen, VAL* bufferAddr)
    */
    LIFO(IND bufferLen, VAL* bufferAddr) : GeneralBuffer<VAL, IND>(bufferLen, bufferAddr){
    };

    /*
      Returns the first value of the internal sequence (position 1) of stored
      values and removes it from the buffer. Positions, used capacity and
      free capacity will be changed by the call.
      
      If no value is available in the buffer, the result will be 0. Before a call
      of this function, it must be checked, that at least one value is available.
    
      @returns    first value of the sequence, if at least one value was available before, otherwise 0
    */
    VAL getValue(){
      return GeneralBuffer<VAL, IND>::getLastValue();
    };

    /*
      Adds the value at the begining (new position 1) of the internal
      sequence of stored values. Positions, used capacity and free capacity
      may be changed by the call.
      
      If no free capacity exists for an additional value, there are two options.
      The currently first value can be overwritten and will be lost, but the
      buffer will not overflow, or the given value will not be stored.

      @param value          the value that shall be stored in the buffer
      @param allowOverwrite the information, if overwriting is allowed
      @returns              true, if value could be stored, even with overwriting, otherwise false
    */
    bool putValue(VAL value, bool allowOverwrite){
      return GeneralBuffer<VAL, IND>::putLastValue(value, allowOverwrite);
    };

    /*
      Returns the value at the given position of the internal sequence of stored
      values. The value will not be removed from the buffer.
      Positions, used capacity and free capacity will stay the same as before the call.

      In contrast to the positions of GeneralBuffer and FIFO, the
      positions are reversed: position 1 is the last added value
      and position [usedcapacity] is the oldest value in the stack.

      If no value is available at the position (means position is invalid) in the buffer,
      the result will be 0. Before a call of this function, it must be checked, that
      a value is available at the position in the buffer.
    
      @param pos  the position of the requested value, range from 1 to usedCapacity
      @returns    value at the position of the sequence, if a value is available there, otherwise 0
    */
    VAL valueAtPos(IND pos){
      return GeneralBuffer<VAL, IND>::valueAtPos(GeneralBuffer<VAL, IND>::getUsedCapacity() - pos + 1);
    };
};

#endif
