/*
  Copyright (c) 2025 Kay Kasper
  under the MIT License (MIT)
*/

#include <GeneralBuffer.h>
#include <GeneralBufferExtensions.h>

/*
  Example that tests the functionality of the
  FIFO class with byte values by putting and getting
  values in predefined orders to check the results.

  Results will be printed out via Serial.
  If checking doesn't have findings, no output
  will be printed. Only errors are printed.

  The test is based on a fixed capacity of 6 values.

  At the end also some performance testing is
  included.
*/

unsigned long current = millis();
unsigned long lastOutput = 0;
unsigned long _micros = micros();
unsigned long _diff = 0;

// capacity 6
byte buf[6 + 1];
FIFO<byte> bFIFO(6 + 1, &buf[0]);


// do the checking and print hint in case of and error
void check(int value, int ref, int id, int seq){
  if(value != ref){
    Serial.print("ID: ");
    Serial.print(id);
    Serial.print("/");
    Serial.print(seq);
    Serial.print(": '");
    Serial.print(value) ;
    Serial.print("' != reference '");
    Serial.print(ref);
    Serial.println("'");
  }
}


// the setup function is called once for initialization
void setup() {
  Serial.begin(9600);
  Serial.println("\r\nChecking:\r\n");

  bFIFO.reset();

  // Content (begin of buffer): empty 
  check(bFIFO.hasValue(), false, 0, 1);
  check(bFIFO.hasValues(0), true, 0, 2);
  check(bFIFO.hasValues(1), false, 0, 3);
  check(bFIFO.hasValues(2), false, 0, 4);
  check(bFIFO.hasValues(3), false, 0, 5);
  check(bFIFO.hasValues(4), false, 0, 6);
  check(bFIFO.hasValues(5), false, 0, 7);
  check(bFIFO.hasValues(6), false, 0, 8);
  check(bFIFO.hasValues(7), false, 0, 9);
  check(bFIFO.hasFreeCapacity(), true, 0, 10);
  check(bFIFO.hasFreeCapacity(0), true, 0, 11);
  check(bFIFO.hasFreeCapacity(1), true, 0, 12);
  check(bFIFO.hasFreeCapacity(5), true, 0, 13);
  check(bFIFO.hasFreeCapacity(6), true, 0, 14);
  check(bFIFO.getFreeCapacity(), 6, 0, 15);
  check(bFIFO.getUsedCapacity(), 0, 0, 16);

  check(bFIFO.putValue(101, false), true, 1, 0);
  // Content: 101
  check(bFIFO.hasValue(), true, 1, 1);
  check(bFIFO.hasValues(0), true, 1, 2);
  check(bFIFO.hasValues(1), true, 1, 3);
  check(bFIFO.hasValues(2), false, 1, 4);
  check(bFIFO.hasValues(3), false, 1, 5);
  check(bFIFO.hasValues(4), false, 1, 6);
  check(bFIFO.hasValues(5), false, 1, 7);
  check(bFIFO.hasValues(6), false, 1, 8);
  check(bFIFO.hasValues(7), false, 1, 9);
  check(bFIFO.hasFreeCapacity(), true, 1, 10);
  check(bFIFO.hasFreeCapacity(0), true, 1, 11);
  check(bFIFO.hasFreeCapacity(1), true, 1, 12);
  check(bFIFO.hasFreeCapacity(5), true, 1, 13);
  check(bFIFO.hasFreeCapacity(6), false, 1, 14);
  check(bFIFO.getFreeCapacity(), 5, 1, 15);
  check(bFIFO.getUsedCapacity(), 1, 1, 16);
  check(bFIFO.valueAtPos(0), 0, 1, 17);
  check(bFIFO.valueAtPos(1), 101, 1, 18);
  check(bFIFO.valueAtPos(2), 0, 1, 19);

  check(bFIFO.putValue(102, false), true, 2, 0);
  // Content: 101, 102
  check(bFIFO.hasValue(), true, 2, 1);
  check(bFIFO.hasValues(0), true, 2, 2);
  check(bFIFO.hasValues(1), true, 2, 3);
  check(bFIFO.hasValues(2), true, 2, 4);
  check(bFIFO.hasValues(3), false, 2, 5);
  check(bFIFO.hasFreeCapacity(), true, 2, 6);
  check(bFIFO.hasFreeCapacity(0), true, 2, 7);
  check(bFIFO.hasFreeCapacity(1), true, 2, 8);
  check(bFIFO.hasFreeCapacity(5), false, 2, 9);
  check(bFIFO.hasFreeCapacity(6), false, 2, 10);
  check(bFIFO.getFreeCapacity(), 4, 2, 11);
  check(bFIFO.getUsedCapacity(), 2, 2, 12);
  check(bFIFO.valueAtPos(0), 0, 2, 13);
  check(bFIFO.valueAtPos(1), 101, 2, 13);
  check(bFIFO.valueAtPos(2), 102, 2, 14);
  check(bFIFO.valueAtPos(3), 0, 2, 15);

  check(bFIFO.getValue(), 101, 3, 0);
  // Content: 102
  check(bFIFO.hasValue(), true, 3, 1);
  check(bFIFO.hasValues(0), true, 3, 2);
  check(bFIFO.hasValues(1), true, 3, 3);
  check(bFIFO.hasValues(2), false, 3, 4);
  check(bFIFO.hasFreeCapacity(), true, 3, 5);
  check(bFIFO.hasFreeCapacity(0), true, 3, 6);
  check(bFIFO.hasFreeCapacity(1), true, 3, 7);
  check(bFIFO.hasFreeCapacity(5), true, 3, 8);
  check(bFIFO.hasFreeCapacity(6), false, 3, 9);
  check(bFIFO.getFreeCapacity(), 5, 3, 10);
  check(bFIFO.getUsedCapacity(), 1, 3, 11);
  check(bFIFO.valueAtPos(0), 0, 3, 12);
  check(bFIFO.valueAtPos(1), 102, 3, 13);
  check(bFIFO.valueAtPos(2), 0, 3, 14);

  check(bFIFO.putValue(103, false), true, 4, 0);
  check(bFIFO.putValue(104, false), true, 4, 1);
  check(bFIFO.getValue(), 102, 4, 2);
  check(bFIFO.getValue(), 103, 4, 3);
  check(bFIFO.getValue(), 104, 4, 4);

  // Content (mid of buffer): empty
  check(bFIFO.hasValue(), false, 5, 0);
  check(bFIFO.hasValues(0), true, 5, 1);
  check(bFIFO.hasValues(1), false, 5, 2);
  check(bFIFO.hasValues(6), false, 5, 3);
  check(bFIFO.hasFreeCapacity(), true, 5, 4);
  check(bFIFO.hasFreeCapacity(0), true, 5, 5);
  check(bFIFO.hasFreeCapacity(1), true, 5, 6);
  check(bFIFO.hasFreeCapacity(5), true, 5, 7);
  check(bFIFO.hasFreeCapacity(6), true, 5, 8);
  check(bFIFO.getFreeCapacity(), 6, 5, 9);
  check(bFIFO.getUsedCapacity(), 0, 5, 10);

  check(bFIFO.putValue(105, false), true, 6, 0);
  // Content (mid of buffer): 105
  check(bFIFO.hasValue(), true, 6, 1);
  check(bFIFO.hasValues(0), true, 6, 2);
  check(bFIFO.hasValues(1), true, 6, 3);
  check(bFIFO.hasValues(2), false, 6, 4);
  check(bFIFO.hasValues(3), false, 6, 5);
  check(bFIFO.hasValues(4), false, 6, 6);
  check(bFIFO.hasValues(5), false, 6, 7);
  check(bFIFO.hasValues(6), false, 6, 8);
  check(bFIFO.hasValues(7), false, 6, 9);
  check(bFIFO.hasFreeCapacity(), true, 6, 10);
  check(bFIFO.hasFreeCapacity(0), true, 6, 11);
  check(bFIFO.hasFreeCapacity(1), true, 6, 12);
  check(bFIFO.hasFreeCapacity(2), true, 6, 13);
  check(bFIFO.hasFreeCapacity(3), true, 6, 14);
  check(bFIFO.hasFreeCapacity(4), true, 6, 15);
  check(bFIFO.hasFreeCapacity(5), true, 6, 16);
  check(bFIFO.hasFreeCapacity(6), false, 6, 17);
  check(bFIFO.hasFreeCapacity(7), false, 6, 18);
  check(bFIFO.getFreeCapacity(), 5, 6, 19);
  check(bFIFO.getUsedCapacity(), 1, 6, 20);
  check(bFIFO.valueAtPos(0), 0, 6, 21);
  check(bFIFO.valueAtPos(1), 105, 6, 22);
  check(bFIFO.valueAtPos(2), 0, 6, 23);

  check(bFIFO.putValue(106, false), true, 7, 0);
  // Content (mid of buffer): 105, 106
  check(bFIFO.hasValue(), true, 7, 1);
  check(bFIFO.hasValues(0), true, 7, 2);
  check(bFIFO.hasValues(1), true, 7, 3);
  check(bFIFO.hasValues(2), true, 7, 4);
  check(bFIFO.hasValues(3), false, 7, 5);
  check(bFIFO.hasValues(4), false, 7, 6);
  check(bFIFO.hasValues(5), false, 7, 7);
  check(bFIFO.hasValues(6), false, 7, 8);
  check(bFIFO.hasValues(7), false, 7, 9);
  check(bFIFO.hasFreeCapacity(), true, 7, 10);
  check(bFIFO.hasFreeCapacity(0), true, 7, 11);
  check(bFIFO.hasFreeCapacity(1), true, 7, 12);
  check(bFIFO.hasFreeCapacity(2), true, 7, 13);
  check(bFIFO.hasFreeCapacity(3), true, 7, 14);
  check(bFIFO.hasFreeCapacity(4), true, 7, 15);
  check(bFIFO.hasFreeCapacity(5), false, 7, 16);
  check(bFIFO.hasFreeCapacity(6), false, 7, 17);
  check(bFIFO.hasFreeCapacity(7), false, 7, 18);
  check(bFIFO.getFreeCapacity(), 4, 7, 19);
  check(bFIFO.getUsedCapacity(), 2, 7, 20);
  check(bFIFO.valueAtPos(0), 0, 7, 21);
  check(bFIFO.valueAtPos(1), 105, 7, 22);
  check(bFIFO.valueAtPos(2), 106, 7, 23);
  check(bFIFO.valueAtPos(3), 0, 7, 24);

  check(bFIFO.putValue(107, false), true, 8, 0);
  check(bFIFO.putValue(108, false), true, 8, 1);
  check(bFIFO.putValue(109, false), true, 8, 2);
  // Content (mid of buffer): 105, 106, 107, 108, 109
  check(bFIFO.hasValue(), true, 8, 3);
  check(bFIFO.hasValues(0), true, 8, 4);
  check(bFIFO.hasValues(1), true, 8, 5);
  check(bFIFO.hasValues(2), true, 8, 6);
  check(bFIFO.hasValues(3), true, 8, 7);
  check(bFIFO.hasValues(4), true, 8, 8);
  check(bFIFO.hasValues(5), true, 8, 9);
  check(bFIFO.hasValues(6), false, 8, 10);
  check(bFIFO.hasValues(7), false, 8, 11);
  check(bFIFO.hasFreeCapacity(), true, 8, 12);
  check(bFIFO.hasFreeCapacity(0), true, 8, 13);
  check(bFIFO.hasFreeCapacity(1), true, 8, 14);
  check(bFIFO.hasFreeCapacity(2), false, 8, 15);
  check(bFIFO.hasFreeCapacity(3), false, 8, 16);
  check(bFIFO.hasFreeCapacity(4), false, 8, 17);
  check(bFIFO.hasFreeCapacity(5), false, 8, 18);
  check(bFIFO.hasFreeCapacity(6), false, 8, 19);
  check(bFIFO.hasFreeCapacity(7), false, 8, 20);
  check(bFIFO.getFreeCapacity(), 1, 8, 21);
  check(bFIFO.getUsedCapacity(), 5, 8, 22);
  check(bFIFO.valueAtPos(0), 0, 8, 23);
  check(bFIFO.valueAtPos(1), 105, 8, 24);
  check(bFIFO.valueAtPos(2), 106, 8, 25);
  check(bFIFO.valueAtPos(3), 107, 8, 26);
  check(bFIFO.valueAtPos(4), 108, 8, 27);
  check(bFIFO.valueAtPos(5), 109, 8, 28);
  check(bFIFO.valueAtPos(6), 0, 8, 29);

  check(bFIFO.putValue(110, false), true, 9, 0);
  // Content (mid of buffer): 105, 106, 107, 108, 109, 110
  check(bFIFO.hasValue(), true, 9, 1);
  check(bFIFO.hasValues(0), true, 9, 2);
  check(bFIFO.hasValues(1), true, 9, 3);
  check(bFIFO.hasValues(2), true, 9, 4);
  check(bFIFO.hasValues(3), true, 9, 5);
  check(bFIFO.hasValues(4), true, 9, 6);
  check(bFIFO.hasValues(5), true, 9, 7);
  check(bFIFO.hasValues(6), true, 9, 8);
  check(bFIFO.hasValues(7), false, 9, 9);
  check(bFIFO.hasFreeCapacity(), false, 9, 10);
  check(bFIFO.hasFreeCapacity(0), true, 9, 11);
  check(bFIFO.hasFreeCapacity(1), false, 9, 12);
  check(bFIFO.hasFreeCapacity(2), false, 9, 13);
  check(bFIFO.hasFreeCapacity(3), false, 9, 14);
  check(bFIFO.hasFreeCapacity(4), false, 9, 15);
  check(bFIFO.hasFreeCapacity(5), false, 9, 16);
  check(bFIFO.hasFreeCapacity(6), false, 9, 17);
  check(bFIFO.hasFreeCapacity(7), false, 9, 18);
  check(bFIFO.getFreeCapacity(), 0, 9, 19);
  check(bFIFO.getUsedCapacity(), 6, 9, 20);
  check(bFIFO.valueAtPos(0), 0, 9, 21);
  check(bFIFO.valueAtPos(1), 105, 9, 22);
  check(bFIFO.valueAtPos(2), 106, 9, 23);
  check(bFIFO.valueAtPos(3), 107, 9, 24);
  check(bFIFO.valueAtPos(4), 108, 9, 25);
  check(bFIFO.valueAtPos(5), 109, 9, 26);
  check(bFIFO.valueAtPos(6), 110, 9, 27);
  check(bFIFO.valueAtPos(7), 0, 9, 28);

  check(bFIFO.putValue(111, false), false, 10, 0);
  // Content (mid of buffer): 105, 106, 107, 108, 109, 110
  check(bFIFO.getFreeCapacity(), 0, 10, 1);
  check(bFIFO.getUsedCapacity(), 6, 10, 2);
  check(bFIFO.valueAtPos(0), 0, 10, 3);
  check(bFIFO.valueAtPos(1), 105, 10, 4);
  check(bFIFO.valueAtPos(2), 106, 10, 5);
  check(bFIFO.valueAtPos(3), 107, 10, 6);
  check(bFIFO.valueAtPos(4), 108, 10, 7);
  check(bFIFO.valueAtPos(5), 109, 10, 8);
  check(bFIFO.valueAtPos(6), 110, 10, 9);
  check(bFIFO.valueAtPos(7), 0, 10, 10);

  check(bFIFO.putValue(111, true), true, 11, 0);
  // Content (mid of buffer): 105, 106, 107, 108, 109, 111
  check(bFIFO.getFreeCapacity(), 0, 11, 1);
  check(bFIFO.getUsedCapacity(), 6, 11, 2);
  check(bFIFO.valueAtPos(0), 0, 11, 3);
  check(bFIFO.valueAtPos(1), 105, 11, 4);
  check(bFIFO.valueAtPos(2), 106, 11, 5);
  check(bFIFO.valueAtPos(3), 107, 11, 6);
  check(bFIFO.valueAtPos(4), 108, 11, 7);
  check(bFIFO.valueAtPos(5), 109, 11, 8);
  check(bFIFO.valueAtPos(6), 111, 11, 9);
  check(bFIFO.valueAtPos(7), 0, 11, 10);

  // Performance
  Serial.println("\r\nPerformance\r\n");

  // reset
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.reset();
  }
  _diff = micros()-_micros;
  Serial.print("reset 10000: ");
  Serial.println(_diff);

  // put linear
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.reset();
    bFIFO.putValue(100, false);
    bFIFO.putValue(101, false);
    bFIFO.putValue(102, false);
    bFIFO.putValue(103, false);
    bFIFO.putValue(104, false);
    bFIFO.putValue(105, false);
  }
  _diff = micros()-_micros;
  Serial.print("putValue 60000: ");
  Serial.println(_diff);

  // put with override
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.putValue(106, true);
  }
  _diff = micros()-_micros;
  Serial.print("putValue 10000 over: ");
  Serial.println(_diff);

  // put without override
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.putValue(106, false);
  }
  _diff = micros()-_micros;
  Serial.print("putValue 10000 no over: ");
  Serial.println(_diff);

  // valueAtPos (from Start)
  bFIFO.reset();
  bFIFO.putValue(101,true);
  bFIFO.putValue(102,true);
  bFIFO.putValue(103,true);
  bFIFO.putValue(104,true);
  bFIFO.putValue(105,true);
  bFIFO.putValue(106,true);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.valueAtPos(1);
    bFIFO.valueAtPos(2);
    bFIFO.valueAtPos(3);
    bFIFO.valueAtPos(4);
    bFIFO.valueAtPos(5);
    bFIFO.valueAtPos(6);
  }
  _diff = micros()-_micros;
  Serial.print("valueAt Start 60000: ");
  Serial.println(_diff);

  // valueAtPos (from Middle)
  bFIFO.reset();
  bFIFO.putValue(101,true);
  bFIFO.putValue(102,true);
  bFIFO.putValue(103,true);
  bFIFO.putValue(104,true);
  bFIFO.putValue(105,true);
  bFIFO.putValue(106,true);
  bFIFO.getValue();
  bFIFO.getValue();
  bFIFO.getValue();
  bFIFO.putValue(107,true);
  bFIFO.putValue(108,true);
  bFIFO.putValue(109,true);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.valueAtPos(1);
    bFIFO.valueAtPos(2);
    bFIFO.valueAtPos(3);
    bFIFO.valueAtPos(4);
    bFIFO.valueAtPos(5);
    bFIFO.valueAtPos(6);
  }
  _diff = micros()-_micros;
  Serial.print("valueAt Mid 60000: ");
  Serial.println(_diff);

  // put and getLast (from Middle)
  bFIFO.reset();
  bFIFO.putValue(101,true);
  bFIFO.putValue(102,true);
  bFIFO.putValue(103,true);
  bFIFO.putValue(104,true);
  bFIFO.getValue();
  bFIFO.getValue();
  bFIFO.getValue();
  bFIFO.getValue();

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    bFIFO.putValue(101,true);
    bFIFO.putValue(102,true);
    bFIFO.putValue(103,true);
    bFIFO.putValue(104,true);
    bFIFO.putValue(105,true);
    bFIFO.putValue(106,true);
    bFIFO.getValue();
    bFIFO.getValue();
    bFIFO.getValue();
    bFIFO.getValue();
    bFIFO.getValue();
    bFIFO.getValue();
  }
  _diff = micros()-_micros;
  Serial.print("putValue/getValue Mid 120000: ");
  Serial.println(_diff);

  // fill buffer for printing out the content as char
  // values: 'a','b','c','d','e','f'
  bFIFO.putValue(97,true);
  bFIFO.putValue(98,true);
  bFIFO.putValue(99,true);
  bFIFO.putValue(100,true);
  bFIFO.putValue(101,true);
  bFIFO.putValue(102,true);
}


// the loop function runs over and over again forever
void loop() {
  current = micros();

  if(bFIFO.hasValue() && current - lastOutput > 50){
    lastOutput = current;
    Serial.print((char)bFIFO.getValue());
  }

}

