/*
  Copyright (c) 2025 Kay Kasper
  under the MIT License (MIT)
*/

#include <GeneralBuffer.h>
#include <GeneralBufferExtensions.h>

/*
  Example that tests the functionality of the
  GeneralBuffer class with byte values by putting
  and getting values in predefined orders to
  check the results.

  Results will be printed out via Serial.
  If checking doesn't have findings, no output
  will be printed. Only errors are printed.

  The test is based on a fixed capacity of 6 values.

  At the end also some performance testing is
  included.
*/

// fixed capacity 6
byte buf[6 + 1];
GeneralBuffer<byte> ring(6 + 1, &buf[0]);

// alternative GeneralBuffer instance creations with same results
//
// GeneralBuffer<byte, byte> ring(6 + 1, &buf[0]);
// GeneralBuffer<byte, byte> ring = GeneralBuffer<byte, byte>(6 + 1, &buf[0]);
// GeneralBuffer<byte> ring = GeneralBuffer<byte>(6 + 1, &buf[0]);
// GeneralBuffer<byte> ring(6);

unsigned long current = millis();
unsigned long lastOutput = 0;
unsigned long _micros = micros();
unsigned long _diff = 0;


// do the checking and print hint in case of and error
void check(int value, int ref, int id, int seq){
  if(value != ref){
    Serial.print("ID: ");
    Serial.print(id);
    Serial.print("/");
    Serial.print(seq);
    Serial.print(": '");
    Serial.print(value) ;
    Serial.print("' != reference '");
    Serial.print(ref);
    Serial.println("'");
  }
}


// the setup function is called once for initialization
void setup() {
  Serial.begin(9600);
  Serial.println("\r\nChecking:\r\n");

  ring.reset();

  // Content (begin of buffer): empty 
  check(ring.hasValue(), false, 0, 1);
  check(ring.hasValues(0), true, 0, 2);
  check(ring.hasValues(1), false, 0, 3);
  check(ring.hasValues(2), false, 0, 4);
  check(ring.hasValues(3), false, 0, 5);
  check(ring.hasValues(4), false, 0, 6);
  check(ring.hasValues(5), false, 0, 7);
  check(ring.hasValues(6), false, 0, 8);
  check(ring.hasValues(7), false, 0, 9);
  check(ring.hasFreeCapacity(), true, 0, 10);
  check(ring.hasFreeCapacity(0), true, 0, 11);
  check(ring.hasFreeCapacity(1), true, 0, 12);
  check(ring.hasFreeCapacity(5), true, 0, 13);
  check(ring.hasFreeCapacity(6), true, 0, 14);
  check(ring.getFreeCapacity(), 6, 0, 15);
  check(ring.getUsedCapacity(), 0, 0, 16);

  check(ring.putLastValue(101, false), true, 1, 0);
  // Content: 101
  check(ring.hasValue(), true, 1, 1);
  check(ring.hasValues(0), true, 1, 2);
  check(ring.hasValues(1), true, 1, 3);
  check(ring.hasValues(2), false, 1, 4);
  check(ring.hasValues(3), false, 1, 5);
  check(ring.hasValues(4), false, 1, 6);
  check(ring.hasValues(5), false, 1, 7);
  check(ring.hasValues(6), false, 1, 8);
  check(ring.hasValues(7), false, 1, 9);
  check(ring.hasFreeCapacity(), true, 1, 10);
  check(ring.hasFreeCapacity(0), true, 1, 11);
  check(ring.hasFreeCapacity(1), true, 1, 12);
  check(ring.hasFreeCapacity(5), true, 1, 13);
  check(ring.hasFreeCapacity(6), false, 1, 14);
  check(ring.getFreeCapacity(), 5, 1, 15);
  check(ring.getUsedCapacity(), 1, 1, 16);
  check(ring.valueAtPos(0), 0, 1, 17);
  check(ring.valueAtPos(1), 101, 1, 18);
  check(ring.valueAtPos(2), 0, 1, 19);

  check(ring.putLastValue(102, false), true, 2, 0);
  // Content: 101, 102
  check(ring.hasValue(), true, 2, 1);
  check(ring.hasValues(0), true, 2, 2);
  check(ring.hasValues(1), true, 2, 3);
  check(ring.hasValues(2), true, 2, 4);
  check(ring.hasValues(3), false, 2, 5);
  check(ring.hasFreeCapacity(), true, 2, 6);
  check(ring.hasFreeCapacity(0), true, 2, 7);
  check(ring.hasFreeCapacity(1), true, 2, 8);
  check(ring.hasFreeCapacity(5), false, 2, 9);
  check(ring.hasFreeCapacity(6), false, 2, 10);
  check(ring.getFreeCapacity(), 4, 2, 11);
  check(ring.getUsedCapacity(), 2, 2, 12);
  check(ring.valueAtPos(0), 0, 2, 13);
  check(ring.valueAtPos(1), 101, 2, 13);
  check(ring.valueAtPos(2), 102, 2, 14);
  check(ring.valueAtPos(3), 0, 2, 15);

  check(ring.getLastValue(), 102, 3, 0);
  // Content: 101
  check(ring.hasValue(), true, 3, 1);
  check(ring.hasValues(0), true, 3, 2);
  check(ring.hasValues(1), true, 3, 3);
  check(ring.hasValues(2), false, 3, 4);
  check(ring.hasFreeCapacity(), true, 3, 5);
  check(ring.hasFreeCapacity(0), true, 3, 6);
  check(ring.hasFreeCapacity(1), true, 3, 7);
  check(ring.hasFreeCapacity(5), true, 3, 8);
  check(ring.hasFreeCapacity(6), false, 3, 9);
  check(ring.getFreeCapacity(), 5, 3, 10);
  check(ring.getUsedCapacity(), 1, 3, 11);
  check(ring.valueAtPos(0), 0, 3, 12);
  check(ring.valueAtPos(1), 101, 3, 13);
  check(ring.valueAtPos(2), 0, 3, 14);

  check(ring.putLastValue(102, false), true, 4, 0);
  check(ring.putLastValue(103, false), true, 4, 1);
  check(ring.getFirstValue(), 101, 4, 2);
  check(ring.getLastValue(), 103, 4, 3);
  check(ring.getFirstValue(), 102, 4, 4);
  // Content (mid of buffer): empty
  check(ring.hasValue(), false, 5, 0);
  check(ring.hasValues(0), true, 5, 1);
  check(ring.hasValues(1), false, 5, 2);
  check(ring.hasValues(6), false, 5, 3);
  check(ring.hasFreeCapacity(), true, 5, 4);
  check(ring.hasFreeCapacity(0), true, 5, 5);
  check(ring.hasFreeCapacity(1), true, 5, 6);
  check(ring.hasFreeCapacity(5), true, 5, 7);
  check(ring.hasFreeCapacity(6), true, 5, 8);
  check(ring.getFreeCapacity(), 6, 5, 9);
  check(ring.getUsedCapacity(), 0, 5, 10);

  check(ring.putLastValue(104, false), true, 6, 0);
  // Content (mid of buffer): 104
  check(ring.hasValue(), true, 6, 1);
  check(ring.hasValues(0), true, 6, 2);
  check(ring.hasValues(1), true, 6, 3);
  check(ring.hasValues(2), false, 6, 4);
  check(ring.hasValues(3), false, 6, 5);
  check(ring.hasValues(4), false, 6, 6);
  check(ring.hasValues(5), false, 6, 7);
  check(ring.hasValues(6), false, 6, 8);
  check(ring.hasValues(7), false, 6, 9);
  check(ring.hasFreeCapacity(), true, 6, 10);
  check(ring.hasFreeCapacity(0), true, 6, 11);
  check(ring.hasFreeCapacity(1), true, 6, 12);
  check(ring.hasFreeCapacity(2), true, 6, 13);
  check(ring.hasFreeCapacity(3), true, 6, 14);
  check(ring.hasFreeCapacity(4), true, 6, 15);
  check(ring.hasFreeCapacity(5), true, 6, 16);
  check(ring.hasFreeCapacity(6), false, 6, 17);
  check(ring.hasFreeCapacity(7), false, 6, 18);
  check(ring.getFreeCapacity(), 5, 6, 19);
  check(ring.getUsedCapacity(), 1, 6, 20);
  check(ring.valueAtPos(0), 0, 6, 21);
  check(ring.valueAtPos(1), 104, 6, 22);
  check(ring.valueAtPos(2), 0, 6, 23);

  check(ring.putLastValue(105, false), true, 7, 0);
  // Content (mid of buffer): 104, 105
  check(ring.hasValue(), true, 7, 1);
  check(ring.hasValues(0), true, 7, 2);
  check(ring.hasValues(1), true, 7, 3);
  check(ring.hasValues(2), true, 7, 4);
  check(ring.hasValues(3), false, 7, 5);
  check(ring.hasValues(4), false, 7, 6);
  check(ring.hasValues(5), false, 7, 7);
  check(ring.hasValues(6), false, 7, 8);
  check(ring.hasValues(7), false, 7, 9);
  check(ring.hasFreeCapacity(), true, 7, 10);
  check(ring.hasFreeCapacity(0), true, 7, 11);
  check(ring.hasFreeCapacity(1), true, 7, 12);
  check(ring.hasFreeCapacity(2), true, 7, 13);
  check(ring.hasFreeCapacity(3), true, 7, 14);
  check(ring.hasFreeCapacity(4), true, 7, 15);
  check(ring.hasFreeCapacity(5), false, 7, 16);
  check(ring.hasFreeCapacity(6), false, 7, 17);
  check(ring.hasFreeCapacity(7), false, 7, 18);
  check(ring.getFreeCapacity(), 4, 7, 19);
  check(ring.getUsedCapacity(), 2, 7, 20);
  check(ring.valueAtPos(0), 0, 7, 21);
  check(ring.valueAtPos(1), 104, 7, 22);
  check(ring.valueAtPos(2), 105, 7, 23);
  check(ring.valueAtPos(3), 0, 7, 24);

  check(ring.putLastValue(106, false), true, 8, 0);
  check(ring.putLastValue(107, false), true, 8, 1);
  check(ring.putLastValue(108, false), true, 8, 2);
  // Content (mid of buffer): 104, 105, 106, 107, 108
  check(ring.hasValue(), true, 8, 3);
  check(ring.hasValues(0), true, 8, 4);
  check(ring.hasValues(1), true, 8, 5);
  check(ring.hasValues(2), true, 8, 6);
  check(ring.hasValues(3), true, 8, 7);
  check(ring.hasValues(4), true, 8, 8);
  check(ring.hasValues(5), true, 8, 9);
  check(ring.hasValues(6), false, 8, 10);
  check(ring.hasValues(7), false, 8, 11);
  check(ring.hasFreeCapacity(), true, 8, 12);
  check(ring.hasFreeCapacity(0), true, 8, 13);
  check(ring.hasFreeCapacity(1), true, 8, 14);
  check(ring.hasFreeCapacity(2), false, 8, 15);
  check(ring.hasFreeCapacity(3), false, 8, 16);
  check(ring.hasFreeCapacity(4), false, 8, 17);
  check(ring.hasFreeCapacity(5), false, 8, 18);
  check(ring.hasFreeCapacity(6), false, 8, 19);
  check(ring.hasFreeCapacity(7), false, 8, 20);
  check(ring.getFreeCapacity(), 1, 8, 21);
  check(ring.getUsedCapacity(), 5, 8, 22);
  check(ring.valueAtPos(0), 0, 8, 23);
  check(ring.valueAtPos(1), 104, 8, 24);
  check(ring.valueAtPos(2), 105, 8, 25);
  check(ring.valueAtPos(3), 106, 8, 26);
  check(ring.valueAtPos(4), 107, 8, 27);
  check(ring.valueAtPos(5), 108, 8, 28);
  check(ring.valueAtPos(6), 0, 8, 29);

  check(ring.putLastValue(109, false), true, 9, 0);
  // Content (mid of buffer): 104, 105, 106, 107, 108, 109
  check(ring.hasValue(), true, 9, 1);
  check(ring.hasValues(0), true, 9, 2);
  check(ring.hasValues(1), true, 9, 3);
  check(ring.hasValues(2), true, 9, 4);
  check(ring.hasValues(3), true, 9, 5);
  check(ring.hasValues(4), true, 9, 6);
  check(ring.hasValues(5), true, 9, 7);
  check(ring.hasValues(6), true, 9, 8);
  check(ring.hasValues(7), false, 9, 9);
  check(ring.hasFreeCapacity(), false, 9, 10);
  check(ring.hasFreeCapacity(0), true, 9, 11);
  check(ring.hasFreeCapacity(1), false, 9, 12);
  check(ring.hasFreeCapacity(2), false, 9, 13);
  check(ring.hasFreeCapacity(3), false, 9, 14);
  check(ring.hasFreeCapacity(4), false, 9, 15);
  check(ring.hasFreeCapacity(5), false, 9, 16);
  check(ring.hasFreeCapacity(6), false, 9, 17);
  check(ring.hasFreeCapacity(7), false, 9, 18);
  check(ring.getFreeCapacity(), 0, 9, 19);
  check(ring.getUsedCapacity(), 6, 9, 20);
  check(ring.valueAtPos(0), 0, 9, 21);
  check(ring.valueAtPos(1), 104, 9, 22);
  check(ring.valueAtPos(2), 105, 9, 23);
  check(ring.valueAtPos(3), 106, 9, 24);
  check(ring.valueAtPos(4), 107, 9, 25);
  check(ring.valueAtPos(5), 108, 9, 26);
  check(ring.valueAtPos(6), 109, 9, 27);
  check(ring.valueAtPos(7), 0, 9, 28);

  check(ring.putLastValue(110, false), false, 10, 0);
  // Content (mid of buffer): 104, 105, 106, 107, 108, 109
  check(ring.getFreeCapacity(), 0, 10, 1);
  check(ring.getUsedCapacity(), 6, 10, 2);
  check(ring.valueAtPos(0), 0, 10, 3);
  check(ring.valueAtPos(1), 104, 10, 4);
  check(ring.valueAtPos(2), 105, 10, 5);
  check(ring.valueAtPos(3), 106, 10, 6);
  check(ring.valueAtPos(4), 107, 10, 7);
  check(ring.valueAtPos(5), 108, 10, 8);
  check(ring.valueAtPos(6), 109, 10, 9);
  check(ring.valueAtPos(7), 0, 10, 10);

  check(ring.putLastValue(110, true), true, 11, 0);
  // Content (mid of buffer): 104, 105, 106, 107, 108, 110
  check(ring.getFreeCapacity(), 0, 11, 1);
  check(ring.getUsedCapacity(), 6, 11, 2);
  check(ring.valueAtPos(0), 0, 11, 3);
  check(ring.valueAtPos(1), 104, 11, 4);
  check(ring.valueAtPos(2), 105, 11, 5);
  check(ring.valueAtPos(3), 106, 11, 6);
  check(ring.valueAtPos(4), 107, 11, 7);
  check(ring.valueAtPos(5), 108, 11, 8);
  check(ring.valueAtPos(6), 110, 11, 9);
  check(ring.valueAtPos(7), 0, 11, 10);

  ring.reset();
  check(ring.putLastValue(106, true), true, 12, 0);
  // Content (start of buffer): 106
  check(ring.getFreeCapacity(), 5, 12, 1);
  check(ring.getUsedCapacity(), 1, 12, 2);
  check(ring.valueAtPos(0), 0, 12, 3);
  check(ring.valueAtPos(1), 106, 12, 4);
  check(ring.valueAtPos(2), 0, 12, 5);
  check(ring.valueAtPos(3), 0, 12, 6);
  check(ring.valueAtPos(4), 0, 12, 7);
  check(ring.valueAtPos(5), 0, 12, 8);
  check(ring.valueAtPos(6), 0, 12, 9);
  check(ring.valueAtPos(7), 0, 12, 10);

  check(ring.putFirstValue(105, true), true, 13, 0);
  // Content (start of buffer): 105, 106
  check(ring.getFreeCapacity(), 4, 13, 1);
  check(ring.getUsedCapacity(), 2, 13, 2);
  check(ring.valueAtPos(0), 0, 13, 3);
  check(ring.valueAtPos(1), 105, 13, 4);
  check(ring.valueAtPos(2), 106, 13, 5);
  check(ring.valueAtPos(3), 0, 13, 6);

  check(ring.putFirstValue(104, true), true, 14, 0);
  // Content (start of buffer): 104, 105, 106
  check(ring.getFreeCapacity(), 3, 14, 1);
  check(ring.getUsedCapacity(), 3, 14, 2);
  check(ring.valueAtPos(0), 0, 14, 3);
  check(ring.valueAtPos(1), 104, 14, 4);
  check(ring.valueAtPos(2), 105, 14, 5);
  check(ring.valueAtPos(3), 106, 14, 6);
  check(ring.valueAtPos(4), 0, 14, 7);

  check(ring.putFirstValue(103, true), true, 15, 0);
  check(ring.putFirstValue(102, true), true, 15, 1);
  check(ring.putFirstValue(101, true), true, 15, 2);
  // Content (start of buffer): 101, 102, 103, 104, 105, 106
  check(ring.getFreeCapacity(), 0, 15, 3);
  check(ring.getUsedCapacity(), 6, 15, 4);
  check(ring.valueAtPos(0), 0, 15, 5);
  check(ring.valueAtPos(1), 101, 15, 6);
  check(ring.valueAtPos(2), 102, 15, 7);
  check(ring.valueAtPos(3), 103, 15, 8);
  check(ring.valueAtPos(4), 104, 15, 9);
  check(ring.valueAtPos(5), 105, 15, 10);
  check(ring.valueAtPos(6), 106, 15, 11);
  check(ring.valueAtPos(7), 0, 15, 12);

  check(ring.putFirstValue(100, true), true, 16, 0);
  // Content (start of buffer): 100, 102, 103, 104, 105, 106
  check(ring.getFreeCapacity(), 0, 16, 1);
  check(ring.getUsedCapacity(), 6, 16, 2);
  check(ring.valueAtPos(0), 0, 16, 3);
  check(ring.valueAtPos(1), 100, 16, 4);
  check(ring.valueAtPos(2), 102, 16, 5);
  check(ring.valueAtPos(3), 103, 16, 6);
  check(ring.valueAtPos(4), 104, 16, 7);
  check(ring.valueAtPos(5), 105, 16, 8);
  check(ring.valueAtPos(6), 106, 16, 9);
  check(ring.valueAtPos(7), 0, 16, 10);

  check(ring.getFirstValue(), 100, 17, 0);
  check(ring.getFirstValue(), 102, 17, 1);
  check(ring.putLastValue(107, true), true, 17, 2);
  // Content (start of buffer): 103, 104, 105, 106, 107
  check(ring.getFreeCapacity(), 1, 17, 3);
  check(ring.getUsedCapacity(), 5, 17, 4);
  check(ring.valueAtPos(0), 0, 17, 5);
  check(ring.valueAtPos(1), 103, 17, 6);
  check(ring.valueAtPos(2), 104, 17, 7);
  check(ring.valueAtPos(3), 105, 17, 8);
  check(ring.valueAtPos(4), 106, 17, 9);
  check(ring.valueAtPos(5), 107, 17, 10);
  check(ring.valueAtPos(6), 0, 17, 11);
  check(ring.valueAtPos(7), 0, 17, 12);

  check(ring.overwriteValueAtPos(89, 0), false, 18, 0);
  check(ring.overwriteValueAtPos(90, 1), true, 18, 1);
  check(ring.overwriteValueAtPos(91, 3), true, 18, 2);
  check(ring.overwriteValueAtPos(92, 5), true, 18, 3);
  check(ring.overwriteValueAtPos(93, 6), false, 18, 4);
  // Content (start of buffer): 90, 104, 91, 106, 92
  check(ring.getFreeCapacity(), 1, 18, 5);
  check(ring.getUsedCapacity(), 5, 18, 6);
  check(ring.valueAtPos(0), 0, 18, 7);
  check(ring.valueAtPos(1), 90, 18, 8);
  check(ring.valueAtPos(2), 104, 18, 9);
  check(ring.valueAtPos(3), 91, 18, 10);
  check(ring.valueAtPos(4), 106, 18, 11);
  check(ring.valueAtPos(5), 92, 18, 12);
  check(ring.valueAtPos(6), 0, 18, 13);
  check(ring.valueAtPos(7), 0, 18, 14);

  ring.reset();
  check(ring.putFirstValue(100, false), true, 19, 0);
  check(ring.putFirstValue(99, false), true, 19, 1);
  check(ring.putFirstValue(98, false), true, 19, 2);
  check(ring.putFirstValue(97, false), true, 19, 3);
  // Content (start of buffer): 97, 98, 99, 100
  check(ring.getFreeCapacity(), 2, 19, 4);
  check(ring.getUsedCapacity(), 4, 19, 5);
  check(ring.putFirstValue(96, false), true, 19, 6);
  // Content (start of buffer): 96, 97, 98, 99, 100
  check(ring.getFreeCapacity(), 1, 19, 7);
  check(ring.getUsedCapacity(), 5, 19, 8);
  check(ring.putLastValue(101, false), true, 19, 9);
  // Content (start of buffer): 96, 97, 98, 99, 100, 101
  check(ring.getFreeCapacity(), 0, 19, 10);
  check(ring.getUsedCapacity(), 6, 19, 11);
  check(ring.valueAtPos(0), 0, 19, 12);
  check(ring.valueAtPos(1), 96, 19, 13);
  check(ring.valueAtPos(2), 97, 19, 14);
  check(ring.valueAtPos(3), 98, 19, 15);
  check(ring.valueAtPos(4), 99, 19, 16);
  check(ring.valueAtPos(5), 100, 19, 17);
  check(ring.valueAtPos(6), 101, 19, 18);
  check(ring.valueAtPos(7), 0, 19, 19);

  check(ring.getValueAtPos(0), 0, 20, 0);
  check(ring.getValueAtPos(1), 96, 20, 1);
  check(ring.getValueAtPos(2), 98, 20, 2);
  check(ring.getValueAtPos(4), 101, 20, 3);
  // Content (start of buffer): 97, 99, 100
  check(ring.valueAtPos(0), 0, 20, 4);
  check(ring.valueAtPos(1), 97, 20, 5);
  check(ring.valueAtPos(2), 99, 20, 6);
  check(ring.valueAtPos(3), 100, 20, 7);
  check(ring.valueAtPos(4), 0, 20, 8);
  check(ring.valueAtPos(5), 0, 20, 9);
  check(ring.valueAtPos(6), 0, 20, 10);
  check(ring.valueAtPos(7), 0, 20, 11);

  check(ring.getValueAtPos(2), 99, 21, 0);
  check(ring.getValueAtPos(2), 100, 21, 1);
  check(ring.getValueAtPos(1), 97, 21, 2);
  check(ring.getValueAtPos(1), 0, 21, 3);
  // Content (start of buffer): leer
  check(ring.getFreeCapacity(), 6, 21, 4);
  check(ring.getUsedCapacity(), 0, 21, 5);
  check(ring.valueAtPos(0), 0, 21, 6);
  check(ring.valueAtPos(1), 0, 21, 7);

  ring.reset();
  check(ring.putValueAtPos(100, 1, false), true, 22, 0);
  // Content (start of buffer): 100
  check(ring.getFreeCapacity(), 5, 22, 1);
  check(ring.getUsedCapacity(), 1, 22, 2);
  check(ring.valueAtPos(0), 0, 22, 3);
  check(ring.valueAtPos(1), 100, 22, 4);
  check(ring.valueAtPos(2), 0, 22, 5);

  check(ring.putValueAtPos(101, 2, false), true, 23, 0);
  check(ring.putValueAtPos(99, 1, false), true, 23, 1);
  // Content (start of buffer): 99, 100, 101
  check(ring.getFreeCapacity(), 3, 23, 2);
  check(ring.getUsedCapacity(), 3, 23, 3);
  check(ring.valueAtPos(0), 0, 23, 4);
  check(ring.valueAtPos(1), 99, 23, 5);
  check(ring.valueAtPos(2), 100, 23, 6);
  check(ring.valueAtPos(3), 101, 23, 7);
  check(ring.valueAtPos(4), 0, 23, 8);

  check(ring.putValueAtPos(1, 2, false), true, 24, 0);
  check(ring.putValueAtPos(2, 3, false), true, 24, 1);
  check(ring.putValueAtPos(3, 6, false), true, 24, 2);
  // Content (start of buffer): 99, 1, 2, 100, 101, 3
  check(ring.getFreeCapacity(), 0, 24, 3);
  check(ring.getUsedCapacity(), 6, 24, 4);
  check(ring.valueAtPos(0), 0, 24, 5);
  check(ring.valueAtPos(1), 99, 24, 6);
  check(ring.valueAtPos(2), 1, 24, 7);
  check(ring.valueAtPos(3), 2, 24, 8);
  check(ring.valueAtPos(4), 100, 24, 9);
  check(ring.valueAtPos(5), 101, 24, 10);
  check(ring.valueAtPos(6), 3, 24, 11);
  check(ring.valueAtPos(7), 0, 24, 12);

  check(ring.putValueAtPos(1, 2, false), false, 25, 0);
  check(ring.putValueAtPos(10, 1, true), true, 25, 1);
  check(ring.putValueAtPos(11, 3, true), true, 25, 2);
  check(ring.putValueAtPos(12, 6, true), true, 25, 3);
  check(ring.putValueAtPos(13, 7, true), false, 25, 4);
  // Content (start of buffer): 10, 1, 11, 100, 101, 12
  check(ring.getFreeCapacity(), 0, 25, 5);
  check(ring.getUsedCapacity(), 6, 25, 6);
  check(ring.valueAtPos(0), 0, 25, 7);
  check(ring.valueAtPos(1), 10, 25, 8);
  check(ring.valueAtPos(2), 1, 25, 9);
  check(ring.valueAtPos(3), 11, 25, 10);
  check(ring.valueAtPos(4), 100, 25, 11);
  check(ring.valueAtPos(5), 101, 25, 12);
  check(ring.valueAtPos(6), 12, 25, 13);
  check(ring.valueAtPos(7), 0, 25, 14);

  ring.getLastValue();
  // Content (start of buffer): 10, 1, 11, 100, 101
  check(ring.findValueAtPos(0, 0, true, GB_COMP_EQUAL), 0, 26, 0);
  check(ring.findValueAtPos(0, 6, true, GB_COMP_EQUAL), 0, 26, 1);
  check(ring.findValueAtPos(0, 1, true, GB_COMP_EQUAL), 0, 26, 2);
  check(ring.findValueAtPos(1, 1, true, GB_COMP_EQUAL+10), 0, 26, 3);
  check(ring.findValueAtPos(1, 1, true, GB_COMP_EQUAL), 2, 26, 4);
  check(ring.findValueAtPos(10, 1, true, GB_COMP_EQUAL), 1, 26, 5);
  check(ring.findValueAtPos(100, 1, true, GB_COMP_GREATER_EQUAL), 4, 26, 6);
  check(ring.findValueAtPos(100, 5, true, GB_COMP_GREATER_EQUAL), 5, 26, 7);
  check(ring.findValueAtPos(100, 1, true, GB_COMP_GREATER), 5, 26, 8);
  check(ring.findValueAtPos(100, 3, true, GB_COMP_EQUAL), 4, 26, 9);
  check(ring.findValueAtPos(10, 1, true, GB_COMP_LESS), 2, 26, 10);
  check(ring.findValueAtPos(99, 2, true, GB_COMP_LESS_EQUAL), 2, 26, 11);

  check(ring.findValueAtPos(0, 0, false, GB_COMP_EQUAL), 0, 27, 0);
  check(ring.findValueAtPos(0, 6, false, GB_COMP_EQUAL), 0, 27, 1);
  check(ring.findValueAtPos(0, 1, false, GB_COMP_EQUAL), 0, 27, 2);
  check(ring.findValueAtPos(1, 1, false, GB_COMP_EQUAL+10), 0, 27, 3);
  check(ring.findValueAtPos(1, 5, false, GB_COMP_EQUAL), 2, 27, 4);
  check(ring.findValueAtPos(10, 2, false, GB_COMP_EQUAL), 1, 27, 5);
  check(ring.findValueAtPos(100, 4, false, GB_COMP_GREATER_EQUAL), 4, 27, 6);
  check(ring.findValueAtPos(100, 5, false, GB_COMP_GREATER_EQUAL), 5, 27, 7);
  check(ring.findValueAtPos(10, 5, false, GB_COMP_GREATER), 5, 27, 8);
  check(ring.findValueAtPos(10, 3, false, GB_COMP_EQUAL), 1, 27, 9);
  check(ring.findValueAtPos(10, 5, false, GB_COMP_LESS), 2, 27, 10);
  check(ring.findValueAtPos(99, 5, false, GB_COMP_LESS_EQUAL), 3, 27, 11);

  // Performance
  Serial.println("\r\nPerformance:\r\n");

  // reset
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.reset();
  }
  _diff = micros()-_micros;
  Serial.print("reset 10000: ");
  Serial.println(_diff);

  // put linear
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.reset();
    ring.putLastValue(100, false);
    ring.putLastValue(101, false);
    ring.putLastValue(102, false);
    ring.putLastValue(103, false);
    ring.putLastValue(104, false);
    ring.putLastValue(105, false);
  }
  _diff = micros()-_micros;
  Serial.print("putLast 60000: ");
  Serial.println(_diff);

  // put with override
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putLastValue(106, true);
  }
  _diff = micros()-_micros;
  Serial.print("putLast 10000 over: ");
  Serial.println(_diff);

  // put without override
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putLastValue(106, false);
  }
  _diff = micros()-_micros;
  Serial.print("putLast 10000 no over: ");
  Serial.println(_diff);

  // valueAtPos (from Start)
  ring.reset();
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
  ring.putLastValue(103,true);
  ring.putLastValue(104,true);
  ring.putLastValue(105,true);
  ring.putLastValue(106,true);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.valueAtPos(1);
    ring.valueAtPos(2);
    ring.valueAtPos(3);
    ring.valueAtPos(4);
    ring.valueAtPos(5);
    ring.valueAtPos(6);
  }
  _diff = micros()-_micros;
  Serial.print("valueAt Start 60000: ");
  Serial.println(_diff);

  // valueAtPos (from Middle)
  ring.reset();
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
  ring.putLastValue(103,true);
  ring.putLastValue(104,true);
  ring.putLastValue(105,true);
  ring.putLastValue(106,true);
  ring.getFirstValue();
  ring.getFirstValue();
  ring.getFirstValue();
  ring.putLastValue(107,true);
  ring.putLastValue(108,true);
  ring.putLastValue(109,true);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.valueAtPos(1);
    ring.valueAtPos(2);
    ring.valueAtPos(3);
    ring.valueAtPos(4);
    ring.valueAtPos(5);
    ring.valueAtPos(6);
  }
  _diff = micros()-_micros;
  Serial.print("valueAt Mid 60000: ");
  Serial.println(_diff);

  // putLast and getLast (from Middle)
  ring.reset();
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
  ring.putLastValue(103,true);
  ring.putLastValue(104,true);
  ring.getFirstValue();
  ring.getFirstValue();
  ring.getFirstValue();
  ring.getFirstValue();

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putLastValue(101,true);
    ring.putLastValue(102,true);
    ring.putLastValue(103,true);
    ring.putLastValue(104,true);
    ring.putLastValue(105,true);
    ring.putLastValue(106,true);
    ring.getLastValue();
    ring.getLastValue();
    ring.getLastValue();
    ring.getLastValue();
    ring.getLastValue();
    ring.getLastValue();
  }
  _diff = micros()-_micros;
  Serial.print("putLast/getLast Mid 120000: ");
  Serial.println(_diff);

  // putLast and getFirst (from Middle)
  ring.reset();
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
  ring.putLastValue(103,true);
  ring.putLastValue(104,true);
  ring.getFirstValue();
  ring.getFirstValue();
  ring.getFirstValue();
  ring.getFirstValue();

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putLastValue(101,true);
    ring.putLastValue(102,true);
    ring.putLastValue(103,true);
    ring.putLastValue(104,true);
    ring.putLastValue(105,true);
    ring.putLastValue(106,true);
    ring.getFirstValue();
    ring.getFirstValue();
    ring.getFirstValue();
    ring.getFirstValue();
    ring.getFirstValue();
    ring.getFirstValue();
  }
  _diff = micros()-_micros;
  Serial.print("putLast/getFirst Mid 120000: ");
  Serial.println(_diff);

  // put and getValueAtPos1 (from Middle)
  ring.reset();
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
  ring.putLastValue(103,true);
  ring.putLastValue(104,true);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putLastValue(101,true);
    ring.putLastValue(102,true);
    ring.putLastValue(103,true);
    ring.putLastValue(104,true);
    ring.putLastValue(105,true);
    ring.putLastValue(106,true);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
  }
  _diff = micros()-_micros;
  Serial.print("putLast/valueAt 1 120000: ");
  Serial.println(_diff);

  // put and getValueAtPos1 (from Middle)
  ring.reset();
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
  ring.putLastValue(103,true);
  ring.putLastValue(104,true);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putLastValue(101,true);
    ring.putLastValue(102,true);
    ring.putLastValue(103,true);
    ring.putLastValue(104,true);
    ring.putLastValue(105,true);
    ring.putLastValue(106,true);
    ring.getValueAtPos(6);
    ring.getValueAtPos(5);
    ring.getValueAtPos(4);
    ring.getValueAtPos(3);
    ring.getValueAtPos(2);
    ring.getValueAtPos(1);
  }
  _diff = micros()-_micros;
  Serial.print("putLast/valueAt Rev 120000: ");
  Serial.println(_diff);

  // putValueAtPos1 and getValueAtPos1 (from Middle)
  ring.reset();
  ring.putValueAtPos(101,1,true);
  ring.putValueAtPos(102,1,true);
  ring.putValueAtPos(103,1,true);
  ring.putValueAtPos(104,1,true);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putValueAtPos(101,1,true);
    ring.putValueAtPos(102,1,true);
    ring.putValueAtPos(103,1,true);
    ring.putValueAtPos(104,1,true);
    ring.putValueAtPos(105,1,true);
    ring.putValueAtPos(106,1,true);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
    ring.getValueAtPos(1);
  }
  _diff = micros()-_micros;
  Serial.print("putValueAt/getValueAt 1 120000: ");
  Serial.println(_diff);

  // putValueAtPos1 and getValueAtPos1 (from Middle)
  ring.reset();
  ring.putValueAtPos(101,1,true);
  ring.putValueAtPos(102,1,true);
  ring.putValueAtPos(103,1,true);
  ring.putValueAtPos(104,1,true);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);
  ring.getValueAtPos(1);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    ring.putValueAtPos(101,1,true);
    ring.putValueAtPos(102,2,true);
    ring.putValueAtPos(103,3,true);
    ring.putValueAtPos(104,4,true);
    ring.putValueAtPos(105,5,true);
    ring.putValueAtPos(106,6,true);
    ring.getValueAtPos(6);
    ring.getValueAtPos(5);
    ring.getValueAtPos(4);
    ring.getValueAtPos(3);
    ring.getValueAtPos(2);
    ring.getValueAtPos(1);
  }
  _diff = micros()-_micros;
  Serial.print("putValueAt/getValueAt Rev 120000: ");
  Serial.println(_diff);

  // fill buffer for printing out the content as char
  // values: 'a','b','c','d','e','f'
  ring.reset();
  ring.putLastValue(97,true);
  ring.putLastValue(98,true);
  ring.putLastValue(99,true);
  ring.putLastValue(100,true);
  ring.putLastValue(101,true);
  ring.putLastValue(102,true);
}


// the loop function runs over and over again forever
void loop() {
  current = micros();

  if(ring.hasValue() && current - lastOutput > 50){
    lastOutput = current;
    Serial.print((char)ring.getFirstValue());
  }

}
