/*
  Copyright (c) 2025 Kay Kasper
  under the MIT License (MIT)
*/

#include <GeneralBuffer.h>
#include <GeneralBufferExtensions.h>

/*
  Example that uses buffers to sort values with different
  algorithms.

  Random values are created in an array and printed.
  Then they are sorted by 3 algorithms. For comparison
  the needed microseconds are also written as output.
  In the loop() function the processing is repeated
  continously.

  Prerequisites: Serial output
*/

#define CAPACITY 70

char randomValues[CAPACITY];
unsigned long startMicros = 0;
GeneralBuffer<char> sortGB(CAPACITY);


// sorting with bubble sort (only good for very small buffers)
// and output of the results
// count is number of values to be sorted
void sortBubble(byte count, char* values){
  unsigned long microSec = 0;
  byte i, j;
  char tmpChar1, tmpChar2;
  GeneralBuffer<char> sortGB(count);

  // sort1: bubble, lowest at position 1
  microSec = micros();

  for(i = 0 ; i < count ; i++){
    sortGB.putLastValue(values[i], true);
  }

  for(i = 1 ; i <= count - 1 ; i++){
    for(j = 1 ; j <= count - i ; j++){
      tmpChar1 = sortGB.valueAtPos(j);
      tmpChar2 = sortGB.valueAtPos(j + 1);
      if(tmpChar1 > tmpChar2){
        sortGB.overwriteValueAtPos(tmpChar2, j);
        sortGB.overwriteValueAtPos(tmpChar1, j + 1);
      }
    }
  }

  microSec = micros() - microSec;
  Serial.print("Sorted1 (bubble) values in ");
  Serial.print(microSec);
  Serial.println("ns :");
  
  for(i = 0 ; i < count ; i++){
    Serial.print(sortGB.valueAtPos(i + 1));
  }
  Serial.print('\n');
}


// sorting with best performance for buffers < 70 values
// and output of the results
// count is number of values to be sorted
void sortBestAverage(byte count, char* values){
  unsigned long microSec = 0;
  byte i, j;
  char tmpChar1, tmpChar2;
  GeneralBuffer<char> sortGB(count);

  // sort2: insert direct in to buffer at correct position
  // lowest at position 1
  microSec = micros();

  for(i = 0 ; i < count ; i++){
    j = sortGB.findValueAtPos(values[i], 1, true, GB_COMP_GREATER);
    if(j == 0){
      sortGB.putLastValue(values[i], false);
    }
    else if(j == 1){
      sortGB.putFirstValue(values[i], false);
    }
    else{
      sortGB.putValueAtPos(values[i], j, false);
    }
  }

  microSec = micros() - microSec;
  Serial.print("Sorted2 (best for buffer <70) values in ");
  Serial.print(microSec);
  Serial.println("ns :");
  
  for(i = 0 ; i < count ; i++){
    Serial.print(sortGB.valueAtPos(i + 1));
  }
  Serial.print('\n');
}


// sorting with best performance for buffers > 70 values
// and output of the results
// count is number of values to be sorted
void sortOptimizedBig(byte count, char* values){
  unsigned long microSec = 0;
  byte i, j, foundPos;
  char tmpChar1, minChar;
  GeneralBuffer<char> sortGB(count);

  // sort3: find from beginning smallest value + change of values
  // smallest at position 1
  microSec = micros();

  for(i = 0 ; i < count ; i++){
    sortGB.putLastValue(values[i], true);
  }

  for(i = 1 ; i <= count - 1; i++){

    minChar = sortGB.valueAtPos(i);
    foundPos = i;
    while(true){
      j = sortGB.findValueAtPos(minChar, foundPos + 1, true, GB_COMP_LESS);
      if(j > 0){
        foundPos = j;
        minChar = sortGB.valueAtPos(foundPos);
      }
      else{
        break;
      }
    }

    if(foundPos > i){
      tmpChar1 = sortGB.valueAtPos(i);
      sortGB.overwriteValueAtPos(tmpChar1, foundPos);
      sortGB.overwriteValueAtPos(minChar, i);
    }
  }

  microSec = micros() - microSec;
  Serial.print("Sorted3 (best for buffer >70) values in ");
  Serial.print(microSec);
  Serial.println("ns :");
  
  for(i = 0 ; i < count ; i++){
    Serial.print(sortGB.valueAtPos(i + 1));
  }
  Serial.print('\n');
}


// the setup function is called once for initialization
void setup() {
  Serial.begin(9600);
}


// the loop function runs over and over again forever
void loop() {
  byte i ,j;
  char tmpChar1, tmpChar2;

  for(i = 0 ; i < CAPACITY ; i++){
    randomValues[i] = random('A','z' + 1);
  }

  Serial.print("\nOriginal ");
  Serial.print(CAPACITY);
  Serial.println(" random values and sequence:");
  for(i = 0 ; i < CAPACITY ; i++){
    Serial.print(randomValues[i]);
  }
  Serial.print('\n');

  sortBubble(CAPACITY, randomValues);
  sortBestAverage(CAPACITY, randomValues);
  sortOptimizedBig(CAPACITY, randomValues);

  delay(3000);
}

