/*
  Copyright (c) 2025 Kay Kasper
  under the MIT License (MIT)
*/

#include <GeneralBuffer.h>
#include <GeneralBufferExtensions.h>

/*
  Simple example that puts values into the buffer,
  whenever a change of an digital input was identified.
  The value is defined to be positive for changes from
  LOW to HIGH and negative for changes from HIGH to LOW.
  The buffer content is written in the loop one value
  per loop() call.

  Prerequisites:
  A push button that is connected on one side to GND and
  on the other side to INPUT_PIN (inverse or negative logic).
*/

#define INPUT_PIN 2
#define CAPACITY 10

bool printed = false;
int previousPinState = LOW;
byte count = 0;

int fifoStorage[CAPACITY + 1];
FIFO<int> fifoBuffer(CAPACITY + 1, &fifoStorage[0]);


// analyze the input pin and puts in case of state changes
// values into the FIFO:
// value 1 when changes from LOW to HIGH (button pushed)
// value -1 when changes from HIGH to LOW (button released)
void analyzeInput(){
  byte pinState = digitalRead(INPUT_PIN);
  pinState = (pinState == HIGH ? LOW : HIGH); // due to negative logic

  if(previousPinState == LOW and pinState == HIGH){
    fifoBuffer.putValue(1, true);
  }
  else if(previousPinState == HIGH and pinState == LOW){
    fifoBuffer.putValue(-1, true);
  }
  previousPinState = pinState;
}


// the setup function is called once for initialization
void setup() {
  Serial.begin(9600);
  pinMode(INPUT_PIN, INPUT_PULLUP);
}


// the loop function runs over and over again forever
void loop() {
  analyzeInput();

  if(!printed){
    Serial.println("Printing buffer content:");
    printed = true;
  }

  if(fifoBuffer.hasValue()){
    Serial.print(fifoBuffer.getValue());
    count++;
    if(count < 20){
      Serial.print(',');
    }
    else{
      Serial.print('\n');
      count = 0;
    }
  }
}

