/*
  Copyright (c) 2025 Kay Kasper
  under the MIT License (MIT)
*/

#include <GeneralBuffer.h>
#include <GeneralBufferExtensions.h>

/*
  Example that puts values into the buffer, whenever a
  interrupt function is called and a digital input is read.

  The value is defined to be positive for HIGH state (button pushed)
  and negative for LOW state (button released). Due to button bouncing
  with very fast pin state changes, not always is a positive
  number followed by a negative number and vice versa.
  The values height tells the time difference (in micros) between
  previous and current call of the interrupt routine.
  
  The buffer content is written as output in the loop one value
  per loop() call.

  Prerequisites:
  A push button that is connected on one side to GND and
  on the other side to INPUT_PIN (inverse or negative logic).
*/

#define INPUT_PIN 2
#define CAPACITY 10

bool printed = false;
unsigned long previousMicros = 0;
byte count = 0;

long fifoStorage[CAPACITY + 1];
FIFO<long> fifoBuffer(CAPACITY + 1, &fifoStorage[0]);


// handles the negative logic of the electrical setup
// analyze the input pin and puts values into the FIFO:
// value < 0 when button pushed (HIGH state)
// value > 0 when button released (LOW state)
void analyzeInput(){
  byte pinState = digitalRead(INPUT_PIN);
  unsigned long currentMicros = micros();

  pinState = (pinState == HIGH ? LOW : HIGH); // due to negative logic

  if(pinState == HIGH){
    fifoBuffer.putValue(currentMicros - previousMicros, true);
  }
  else{
    fifoBuffer.putValue(previousMicros - currentMicros, true);
  }
  previousMicros = currentMicros;
}


// the setup function is called once for initialization
void setup() {
  Serial.begin(9600);
  pinMode(INPUT_PIN, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(INPUT_PIN),analyzeInput, CHANGE);
}


// the loop function runs over and over again forever
void loop() {
  if(!printed){
    Serial.println("Printing buffer content:");
    printed = true;
  }

  if(fifoBuffer.hasValue()){
    // one value per loop execution with 20 values per line
    Serial.print(fifoBuffer.getValue());
    count++;
    if(count < 20){
      Serial.print(',');
    }
    else{
      Serial.print('\n');
      count = 0;
    }
  }
}

