/*
  The MIT License (MIT)

  Copyright (c) 2025 Kay Kasper

  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the “Software”),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included
  in all copies or substantial portions of the Software.

  The Software is provided “as is”, without warranty of any kind, express
  or implied, including but not limited to the warranties of merchantability,
  fitness for a particular purpose and noninfringement. In no event shall
  the authors or copyright holders be liable for any claim, damages or other
  liability, whether in an action of contract, tort or otherwise, arising
  from, out of or in connection with the software or the use or other dealings
  in the Software.
*/

#ifndef GENERAL_BUFFER
#define GENERAL_BUFFER


#include <inttypes.h>
#include <stdlib.h>

#define GB_COMP_EQUAL         0   // constant for value comparison, == refValue
#define GB_COMP_GREATER_EQUAL 1   // constant for value comparison, >= refValue
#define GB_COMP_GREATER       2   // constant for value comparison, > refValue
#define GB_COMP_LESS          3   // constant for value comparison, < refValue
#define GB_COMP_LESS_EQUAL    4   // constant for value comparison, <= refValue

/*
  General template based buffer for various types and universal use due to
  getting and putting values at any position. FIFO and LIFO subclasses available
  for easy usage. Can help to decouple "parallel" (e.g interrupt and loop)
  running activities on a microcontroller.

  The buffer is implemented as ring buffer. The storage capacity is limited to
  max 250 or max 16000 values, depending on the 2. type IND in the variable definition.
  If 2. type is an 8 Bit type (uint8_t/byte/char) the limit is 250. Other types
  like int or uint16_t limit it to 16000. Due to the best performance on
  microcontrollers for all internal calculations u8int_t (default) is preferred,
  if capacity is sufficient. The priority in the implementation was the
  performance with additionally keeping the code lean and understandable.

  The implementation is not thread-safe when real random access by several parallel
  threads is used. If this is needed, another solution must be chosen.
  Interrupts are not prohibited in the code. On the other hand is it possible,
  with some small restrictions, to handle "parallel" activities (of max 2 threads)
  in microcontrollers with this implementation. If only values are added (by
  using "putLastValue()") in e.g. interrupt routines and only values are removed
  (by using "hasValue()" and "getFirstValue()") within e.g. the loop, practically
  no problems will appear.

  Whenever "parallel" activities are writing and reading the same buffer, it must be
  ensured, that over the time different speeds in the processing will not exceed
  the capacity. The speed in the processing and the defined capacity of the buffer
  must be aligned.

  The storage for the handled values can be dynamically (based on necessary
  capacity) generated internally or can be handed over as (outside the class)
  defined field. In microcontrollers it is better to use outside predefined
  fields of the same type as the buffer itself (1. type VAL) is defined.
  The instantiation of a new buffer object with dynamically allocated memory
  is slower than the other version.
  
  The values are stored as a sequence (without gaps) or queue which has a beginning and
  an end. Internally two pointers are defining in the ring the first value (the
  beginning) and the last value (the end). The first value ist defined to be at
  position 1 and the last value is to be defined at position [usedCapacity].

  The two template types are:
  1. type VAL defines the type of the handled values (mandatory)
  2. type IND defines the type of the internal pointers and calculations (optional, default uint8_t)

  Advantages:
  - general use for temporary storage
  - value manipulation in order or out of order
  - access to all values at any time by direct addressing
  - template based for various storage types
  - template based for 2 different optimized maximum sizes
  - best performance up to 250 values capacity (type IND = uint8_t)
  - up to 16000 values capacity passible (type IND = uint16_t or int)
  - very high performance for most used functions
  - small footprint (RAM usage, minimum 8 bytes with IND = uint8_t) for each buffer instance
  - works with externaly defined or dynamically generated storage
  - supports queues (FIFO) and stacks (LIFO)
  - supports up to 2 parallel threads (one putting and one getting values)
  - inserting, deleting and overwriting of values at any position in the buffer
*/
template <typename VAL, typename IND = uint8_t>
class GeneralBuffer {
  protected:

  /*
    if _nextLastIndex == _headIndex, then buffer is empty
    if _nextLastIndex + 1 == _headIndex, then buffer is full
  */

  // potential number of values in the storage (field), capacity + 1
  IND _realLen;
  // pointer to real memory block where values are addressed like in fields
  volatile VAL* _buffer;
  // index in the field (_buffer[_nextLastIndex]) one position behind current
  // last value -> index where next last value will be stored
  volatile IND _nextLastIndex;
  // index in the field (_buffer[_headIndex]) where first values is stored
  volatile IND _headIndex;
  // is set to true, if during the initialization memory was allocated with malloc()
  bool _dynamicMem;
  
  private:

  /*
    All add and remove operations need the special handling of the
    pointers/indexes to create the mapping between virtual ring and
    sequencially field storage.
  */


  /*
    Calculate new index based on existing index and an offset value.
    Only values in the sequence "after" the index can be addressed.
    
    @param index  base index, typically the first value _headIndex
    @param add    to be addressed value based on index, typically a position
    @returns      new calculated index of the addressed value in the field
  */
  IND recalcIndexAdd(IND index, IND add){
    // most performant variant
    IND tmp = index + add;
    IND correct; // difference between max byte value and _realLen when byte overflows
    
    if(tmp >= index){   // overflow?
      // no overflow
      return (tmp >= _realLen ? tmp - _realLen : tmp);
    }
    // overflow in calculation needs to be taken into account when uint8_t size is used
    // if type IND is only 8 Bit, correction needed (because max _realLen = 251)
    // if type IND >= 16 Bit, max _realLen is limited to 16001 and correction not needed
    // -> following code is only used in case of type IND is 8 Bit
    correct = 0xFF - _realLen + 1;
    return tmp + correct;
  };


  /*
    Calculate index of next value in the sequence after given index.
    
    @param index  base index, whose next value should be addressed
    @returns      new calculated index of the addressed value in the field
  */
  IND recalcIndexInc(IND index){
    return (index == _realLen - 1 ? 0 : index + 1);
  };


  /*
    Calculate index of previous value in the sequence before given index.
    
    @param index  base index, whose previous value should be addressed
    @returns      new calculated index of the addressed value in the field
  */
  IND recalcIndexDec(IND index){
    return (index == 0 ? _realLen - 1 : index - 1);
  };


  /*
    Initialize the internal structures, depending on the given
    capacity (> 0; buffer infos must be == 0) or buffer (len > 0 and addr != 0,
    capacity must be == 0).

    @param capacity   number of values, that shall be handled by the buffer,
                      if number is to big for storage allocation, buffer can not be used
    @param bufferLen  storage size, values from 2 to 251 (type IND 8 Bit) or 16001 (type IND > 8 Bit) allowed
    @param bufferAddr pointer to the first value of the storage
  */
  void init(IND capacity, IND bufferLen, VAL* bufferAddr){
    _nextLastIndex = 0;
    _headIndex = 0;
    _dynamicMem = false;
    _buffer = 0;
    _realLen = 1;

    if(capacity > 0){
      _realLen = capacity + 1;
      _buffer = (VAL*)malloc(sizeof(VAL) * _realLen);
      _dynamicMem = (_buffer == 0 ? false : true);
    }
    else if(bufferLen > 0 and bufferAddr != 0){
      _realLen = bufferLen;
      _buffer = bufferAddr;
    }
    // check size limits for safe implementation
    if(sizeof(IND) > 1){
      if(_realLen > 16001){
        _realLen = 16001;
      }
    }
    else{
      if(_realLen > 251){
        _realLen = 251;
      }
    }
    // for safety, make sure, memory is available
    if(_buffer == 0){
      _realLen = 1;  // -> usedCapacity = freeCapacity = 0 -> no values will be buffered
    }
  }

  /*
    Comparison of two values. The condition is defined as constants
    GB_COMP_EQUAL, GB_COMP_GREATER_EQUAL, GB_COMP_GREATER, GB_COMP_LESS, GB_COMP_LESS_EQUAL
    for the comparison:
    compValue [condition] refValue (e.g. GB_COMP_GREATER_EQUAL: compValue >= refValue)

    @param compValue  to be compared value
    @param refValue   reference value
    @param condition  comparison condition as one of the constants GB_COMP_*
    @returns          true, if condition is met, false otherwise
  */
  bool compare(VAL compValue, VAL refValue, uint8_t condition){
    switch(condition) {
      case GB_COMP_EQUAL:
        return compValue == refValue;
      case GB_COMP_GREATER:
        return compValue > refValue;
      case GB_COMP_GREATER_EQUAL:
        return compValue >= refValue;
      case GB_COMP_LESS:
        return compValue < refValue;
      case GB_COMP_LESS_EQUAL:
        return compValue <= refValue;
    }
    return false;
  };


  public:

  /*
    Creates a new buffer based on an already defined and handed over
    storage.
    
    The storage must be located somewhere, where it is never (re)moved
    while the GeneralBuffer instance still exists. It must be ensured,
    that no reading or writing is done in the storage by others than
    the here defind instance.
    
    The storage type must be the same as the 1. type VAL of the
    GeneralBuffer instance. The storage size is always defining the
    available capacity for storing values (size - 1 = capacity).
    The 2. type IND of the GeneralBuffer instance is optional and
    defines the max bufferLen of the buffer.

    @param bufferLen  storage size, values from 2 to 251 (type IND 8 Bit) or 16001 (type IND > 8 Bit) allowed
    @param bufferAddr pointer to the first value of the storage
  */
  GeneralBuffer(IND bufferLen, VAL* bufferAddr){
    init(0, bufferLen, bufferAddr);
  };


  /*
    Creates a new buffer based on a planned capacity, which is the
    maximum number of values, that shall be handled.
    The storage will be allocated based on the values type VAL internally
    and dynamically. It is expected to be never freed.
    The 2. type IND of the GeneralBuffer instance is optional and
    defines the max bufferLen of the buffer. Buffer capacity and type definition
    IND for the internal calculations and indexes must match.

    @param capacity   number of values, that shall be handled by the buffer
  */
  GeneralBuffer(IND capacity){
    init(capacity, 0, 0);
  };


  /* 
    If object is finalized, the dynamically allocated memory must be freed.
    When GeneralBuffer variable is defined in a function, the memory will be
    freed by this when the programm leaves the function.
  */
  ~GeneralBuffer(){
    if(_dynamicMem){
      free(_buffer);
    }
  };


  /*
    Returns the information, if at least number of values is in the buffer.
    
    @param num  number of requested values in the buffer, should be less or equal to defined capacity
    @returns    true, if at least number of values are available in the buffer, else false
  */
  bool hasValues(IND num){
    return num <= getUsedCapacity();
  };


  /*
    Returns the information, if at least one value is in the buffer.
    
    @returns    true, if at least one values is available in the buffer, else false
  */
  bool hasValue(){
    return (_nextLastIndex != _headIndex);
  };


  /*
    Returns the information, if at least a number of additional values can be stored
    in the buffer without overwriting.
    
    @param num  number of requested additional values in the buffer, should be less or equal to defined capacity
    @returns    true, if at least additional number of values can be stored in the buffer, else false
  */
  bool hasFreeCapacity(IND num){
    return num <= getFreeCapacity();
  };


  /*
    Returns the information, if at least one additional value can be stored
    in the buffer without overwriting.
    
    @returns    true, if at least one additional value can be stored in the buffer, else false
  */
  bool hasFreeCapacity(){
    IND tmpInIndex = _nextLastIndex + 1;
    return _headIndex != (tmpInIndex == _realLen ? 0 : tmpInIndex);
  };


  /*
    Returns the number of additionally storeable values in the buffer
    without overwriting.
    Sum of getFreeCapacity() and getUsedCapacity() will be the defined capacity.
    
    @returns    number of additionally storable values without overwritung, less or equal the defined capacity 
    @seealso    getUsedCapacity()
  */
  IND getFreeCapacity(){
    if(_nextLastIndex >= _headIndex){
      // kein Wert oder Werte linear vorhanden
      return (_realLen - 1) - (_nextLastIndex - _headIndex);
    }
    // Werte nicht linear vorhanden
    return _headIndex - _nextLastIndex - 1;
  };


  /*
    Returns the number of stored values in the buffer.
    Sum of getFreeCapacity() and getUsedCapacity() will be the defined capacity.
    
    @returns    number of stored values in the buffer, less or equal the defined capacity 
    @seealso    getFreeCapacity()
  */
  IND getUsedCapacity(){
    if(_nextLastIndex >= _headIndex){
      // kein Wert oder Werte linear vorhanden
      return _nextLastIndex - _headIndex;
    }
    // Werte nicht linear vorhanden
    return _realLen - _headIndex + _nextLastIndex;
  };


  /*
    Returns the first value of the internal sequence (position 1) of stored
    values and removes it from the buffer. Positions, used capacity and
    free capacity will be changed by the call.
    
    If no value is available in the buffer, the result will be 0. Before a call
    of this function, it must be checked, that at least one value is available.
  
    @returns    first value of the sequence, if at least one value was available before, otherwise 0
  */
  VAL getFirstValue(){
    VAL tmpVal = 0;

    if(_nextLastIndex == _headIndex){
      // kein Wert vorhanden
      return tmpVal;
    }

    tmpVal = _buffer[_headIndex];
    _headIndex = recalcIndexInc(_headIndex);
    return tmpVal;
  };


  /*
    Adds the value at the beginning (position 1) of the internal sequence
    of stored values. Positions, used capacity and free capacity may be
    changed by the call.
    
    If no free capacity exists for an additional value, there are two options.
    The currently first value can be overwritten and will be lost, but the
    buffer will not overflow, or the given value will not be stored.

    @param value          the value that shall be stored in the buffer
    @param allowOverwrite the information, if overwriting is allowed
    @returns              true, if value could be stored, even with overwriting, otherwise false
  */
  bool putFirstValue(VAL value, bool allowOverwrite){
    IND newHeadIndex = recalcIndexDec(_headIndex);
    
    if(newHeadIndex != _nextLastIndex){ // see hasFreeCapacity()
      // capacity available
      _buffer[newHeadIndex] = value;
      _headIndex = newHeadIndex;
      return true;
    }
    else{
      // no capacity available
      if(allowOverwrite){
        // overwrite the first value
        _buffer[_headIndex] = value;
        return true;
      }
    }
    return false;
  };


  /*
    Returns the value at the given position of the internal sequence of stored
    values and removes it from the buffer. Positions, used capacity and
    free capacity will be changed by the call.
    
    If no value is available at the position (means position is invalid) in the buffer,
    the result will be 0. Before a call of this function, it must be checked, that
    a value is available at the position in the buffer.
  
    @param pos  the position of the requested value, range from 1 to usedCapacity
    @returns    value at the position of the sequence, if a value is available there, otherwise 0
  */
  VAL getValueAtPos(IND pos){
    IND offset = pos - 1;
    IND usedCapacity = getUsedCapacity();
    VAL result;
    
    if(pos == 0 || pos > usedCapacity){
      return 0;
    }

    // position and offset are valid frome here on
    result = _buffer[recalcIndexAdd(_headIndex, offset)];
    // i is offset to head: move -1 position from [pos to usedCapacity-1] to [offset to usedCapacity-2]
    if(usedCapacity > 1){
      for(IND i = offset ; i <= usedCapacity - 2 ; i++){
        _buffer[recalcIndexAdd(_headIndex, i)] = _buffer[recalcIndexAdd(_headIndex , i + 1)];
      }
    }
    _nextLastIndex = recalcIndexDec(_nextLastIndex);
    return result;
  };


  /*
    Insert the value at the given position of the internal sequence
    of stored values. Positions, used capacity and free capacity may be
    changed by the call.
    
    If no free capacity exists for an additional value, there are two options.
    The current value at the position can be overwritten and will be lost, but the
    buffer will not overflow, or the given value will not be stored.
    If a position is given, that is usedCapacity+1, then the value will be
    added as new last value of the sequence. If a position is given, that already
    exists, then all existing values (starting with given position) will be
    moved to the next higher position and the value will be stored in the given
    position.

    @param value          the value that shall be stored in the buffer
    @param pos            the requested position for the value, range from 1 to usedCapacity+1
    @param allowOverwrite the information, if overwriting is allowed
    @returns              true, if value could be stored, even with overwriting, otherwise false
  */
  bool putValueAtPos(VAL value, IND pos, bool overwrite){
    IND offset = pos - 1;
    IND usedCapacity = getUsedCapacity();
    IND freeCapacity = _realLen - 1 - usedCapacity;
    
    if(pos == 0 || pos > usedCapacity + 1){
      return 0;
    }
    // position and offset are valid from here on
    if(freeCapacity == 0){
      if(overwrite){
        if(pos <= usedCapacity){
          _buffer[recalcIndexAdd(_headIndex , offset)] = value;
          return true;
        }
      }
      return false;
    }

    // capacity available
    // i is offset to head: move values one position to end from [usedCapacity-1 to offset] to [usedCapacity to pos]
    if(usedCapacity > 0){
      for(IND i = usedCapacity ; i >= pos ; i--){
        _buffer[recalcIndexAdd(_headIndex, i)] = _buffer[recalcIndexAdd(_headIndex , i - 1)];
      }
    }
    _buffer[recalcIndexAdd(_headIndex, offset)] = value;
    _nextLastIndex = recalcIndexInc(_nextLastIndex);
    return true;
  };


  /*
    Returns the value at the end (last postion = getUsedCapacity) of the
    internal sequence of stored values and removes it from the buffer.
    Positions, used capacity and free capacity will be changed by the call.
    
    If no value is available in the buffer, the result will be 0.
    Before a call of this function, it must be checked, that a value is available.
  
    @returns    value from the end of the sequence, if a value is available, otherwise 0
  */
  VAL getLastValue(){
    if(_nextLastIndex == _headIndex){
      // no value available
      return 0;
    }

    _nextLastIndex = recalcIndexDec(_nextLastIndex);
    return _buffer[_nextLastIndex];
  };


  /*
    Adds the value at the end of the internal sequence of stored values.
    Positions, used capacity and free capacity may be changed by the call.
    
    If no free capacity exists for an additional value, there are two options.
    The currently last value can be overwritten and will be lost, but the
    buffer will not overflow, or the given value will not be stored.

    @param value          the value that shall be stored in the buffer
    @param allowOverwrite the information, if overwriting is allowed
    @returns              true, if value could be stored, even with overwriting, otherwise false
  */
  bool putLastValue(VAL value, bool allowOverwrite){
    IND newInIndex = recalcIndexInc(_nextLastIndex);
    
    if(_headIndex != newInIndex){ // see hasFreeCapacity()
      // capacity available
      _buffer[_nextLastIndex] = value;
      _nextLastIndex = newInIndex;
      return true;
    }
    else{
      // no capacity available
      if(allowOverwrite){
        // overwrite the last value
        _buffer[recalcIndexDec(_nextLastIndex)] = value;
        return true;
      }
      else{
        return false;
      }
    }
  };


  /*
    Returns the value at the given position of the internal sequence of stored
    values. The value will not be removed from the buffer.
    Positions, used capacity and free capacity will stay the same as before the call.
    
    If no value is available at the position (means position is invalid) in the buffer,
    the result will be 0. Before a call of this function, it must be checked, that
    a value is available at the position in the buffer.
  
    @param pos  the position of the requested value, range from 1 to usedCapacity
    @returns    value at the position of the sequence, if a value is available there, otherwise 0
  */
  VAL valueAtPos(IND pos){
    IND offset = pos - 1;
    
    if(pos == 0 || pos > getUsedCapacity()){
      return 0;
    }
    // position and offset are valid from here on
    return _buffer[recalcIndexAdd(_headIndex , offset)];
  };


  /*
    Overwrites the value at the given position of the internal sequence of stored
    values. The existing value will be replaced and not removed.
    Positions, used capacity and free capacity will stay the same as before the call.
    
    If no value is available at the position (means position is invalid) in the buffer,
    the result will be false. Before a call of this function, it must be checked, that
    a value is available at the position in the buffer.

    @param value  the value that shall be stored in the buffer
    @param pos    the position, where the exisiting value shall be replaced, range from 1 to usedCapacity
    @returns      true, if position is valid and value was overwritten, otherwise false
  */
  bool overwriteValueAtPos(VAL value, IND pos){
    IND offset = pos - 1;
    
    if(pos == 0 || pos > getUsedCapacity()){
      return false;
    }
    // position and offset are valid from here on
    _buffer[recalcIndexAdd(_headIndex , offset)] = value;
    return true;
  };


  /*
    Reset the buffer to its initial, empty state. Before the call stored values will be lost.
  */
  void reset(){
    _nextLastIndex = 0;
    _headIndex = 0;
  };


  /*
    Trys to find in the buffer a value based on a reference value, a starting position,
    a comparison condition and a direction.

    The condition is defined as constants
    GB_COMP_EQUAL, GB_COMP_GREATER_EQUAL, GB_COMP_GREATER, GB_COMP_LESS, GB_COMP_LESS_EQUAL
    for the comparison:
    compValue [condition] refValue (e.g. GB_COMP_GREATER_EQUAL: compValue >= refValue)

    @param refValue   reference value
    @param startPos   starting position in the buffer, is first compared value, range 1 to usedCapacity
    @param forward    search direction: true means all positions >= startPos, false means all positions <= startPos
    @param condition  comparison condition as one of the constants GB_COMP_*
    @returns          the position where the first suitable value was found, 0 if not found or parameter error
  */
  IND findValueAtPos(VAL refValue, IND startPos, bool forward, uint8_t condition){
    IND startOffset = startPos - 1;
    IND usedCapacity = getUsedCapacity();
    IND endOffset;

    if(startPos == 0 || startPos > usedCapacity){
      return 0;
    }

    if(forward){
      endOffset = usedCapacity - 1;
      for(IND i = startOffset ; i <= endOffset ; i++){
        if(compare(_buffer[recalcIndexAdd(_headIndex , i)], refValue, condition)){
          return i + 1;
        }
      }
    }
    else{
      for(IND i = startPos ; i > 0 ; i--){
        if(compare(_buffer[recalcIndexAdd(_headIndex , i - 1)], refValue, condition)){
          return i;
        }
      }
    }
    return 0;
  };
};

#endif
