/*
  Copyright (c) 2025 Kay Kasper
  under the MIT License (MIT)
*/

#include <GeneralBuffer.h>
#include <GeneralBufferExtensions.h>

/*
  Example that tests the functionality of the
  FIFO class with int values and int internal
  calculations by putting and getting
  values in predefined orders to check the results.

  Results will be printed out via Serial.
  If checking doesn't have findings, no output
  will be printed. Only errors are printed.

  The test is based on a fixed capacity of 6 values.

  At the end also some performance testing is
  included.
*/

unsigned long current = millis();
unsigned long lastOutput = 0;
unsigned long _micros = micros();
unsigned long _diff = 0;

// capacity 6
int buf[6 + 1];
FIFO<int, int> iFIFO(6 + 1, &buf[0]);


// do the checking and print hint in case of and error
void check(int value, int ref, int id, int seq){
  if(value != ref){
    Serial.print("ID: ");
    Serial.print(id);
    Serial.print("/");
    Serial.print(seq);
    Serial.print(": '");
    Serial.print(value) ;
    Serial.print("' != reference '");
    Serial.print(ref);
    Serial.println("'");
  }
}


// the setup function is called once for initialization
void setup() {
  Serial.begin(9600);
  Serial.println("\r\nChecking");

  iFIFO.reset();

  check(iFIFO.hasValue(), false, 0, 1);
  check(iFIFO.hasValues(0), true, 0, 2);
  check(iFIFO.hasValues(1), false, 0, 3);
  check(iFIFO.hasValues(2), false, 0, 4);
  check(iFIFO.hasValues(3), false, 0, 5);
  check(iFIFO.hasValues(4), false, 0, 6);
  check(iFIFO.hasValues(5), false, 0, 7);
  check(iFIFO.hasValues(6), false, 0, 8);
  check(iFIFO.hasValues(7), false, 0, 9);
  check(iFIFO.hasFreeCapacity(), true, 0, 10);
  check(iFIFO.hasFreeCapacity(0), true, 0, 11);
  check(iFIFO.hasFreeCapacity(1), true, 0, 12);
  check(iFIFO.hasFreeCapacity(5), true, 0, 13);
  check(iFIFO.hasFreeCapacity(6), true, 0, 14);
  check(iFIFO.getFreeCapacity(), 6, 0, 15);
  check(iFIFO.getUsedCapacity(), 0, 0, 16);

  check(iFIFO.putValue(101, false), true, 1, 0);
  // Content: 101
  check(iFIFO.hasValue(), true, 1, 1);
  check(iFIFO.hasValues(0), true, 1, 2);
  check(iFIFO.hasValues(1), true, 1, 3);
  check(iFIFO.hasValues(2), false, 1, 4);
  check(iFIFO.hasValues(3), false, 1, 5);
  check(iFIFO.hasValues(4), false, 1, 6);
  check(iFIFO.hasValues(5), false, 1, 7);
  check(iFIFO.hasValues(6), false, 1, 8);
  check(iFIFO.hasValues(7), false, 1, 9);
  check(iFIFO.hasFreeCapacity(), true, 1, 10);
  check(iFIFO.hasFreeCapacity(0), true, 1, 11);
  check(iFIFO.hasFreeCapacity(1), true, 1, 12);
  check(iFIFO.hasFreeCapacity(5), true, 1, 13);
  check(iFIFO.hasFreeCapacity(6), false, 1, 14);
  check(iFIFO.getFreeCapacity(), 5, 1, 15);
  check(iFIFO.getUsedCapacity(), 1, 1, 16);
  check(iFIFO.valueAtPos(0), 0, 1, 17);
  check(iFIFO.valueAtPos(1), 101, 1, 18);
  check(iFIFO.valueAtPos(2), 0, 1, 19);

  check(iFIFO.putValue(102, false), true, 2, 0);
  // Content: 101, 102
  check(iFIFO.hasValue(), true, 2, 1);
  check(iFIFO.hasValues(0), true, 2, 2);
  check(iFIFO.hasValues(1), true, 2, 3);
  check(iFIFO.hasValues(2), true, 2, 4);
  check(iFIFO.hasValues(3), false, 2, 5);
  check(iFIFO.hasFreeCapacity(), true, 2, 6);
  check(iFIFO.hasFreeCapacity(0), true, 2, 7);
  check(iFIFO.hasFreeCapacity(1), true, 2, 8);
  check(iFIFO.hasFreeCapacity(5), false, 2, 9);
  check(iFIFO.hasFreeCapacity(6), false, 2, 10);
  check(iFIFO.getFreeCapacity(), 4, 2, 11);
  check(iFIFO.getUsedCapacity(), 2, 2, 12);
  check(iFIFO.valueAtPos(0), 0, 2, 13);
  check(iFIFO.valueAtPos(1), 101, 2, 13);
  check(iFIFO.valueAtPos(2), 102, 2, 14);
  check(iFIFO.valueAtPos(3), 0, 2, 15);

  check(iFIFO.getValue(), 101, 3, 0);
  // Content: 102
  check(iFIFO.hasValue(), true, 3, 1);
  check(iFIFO.hasValues(0), true, 3, 2);
  check(iFIFO.hasValues(1), true, 3, 3);
  check(iFIFO.hasValues(2), false, 3, 4);
  check(iFIFO.hasFreeCapacity(), true, 3, 5);
  check(iFIFO.hasFreeCapacity(0), true, 3, 6);
  check(iFIFO.hasFreeCapacity(1), true, 3, 7);
  check(iFIFO.hasFreeCapacity(5), true, 3, 8);
  check(iFIFO.hasFreeCapacity(6), false, 3, 9);
  check(iFIFO.getFreeCapacity(), 5, 3, 10);
  check(iFIFO.getUsedCapacity(), 1, 3, 11);
  check(iFIFO.valueAtPos(0), 0, 3, 12);
  check(iFIFO.valueAtPos(1), 102, 3, 13);
  check(iFIFO.valueAtPos(2), 0, 3, 14);

  check(iFIFO.putValue(103, false), true, 4, 0);
  check(iFIFO.putValue(104, false), true, 4, 1);
  check(iFIFO.getValue(), 102, 4, 2);
  check(iFIFO.getValue(), 103, 4, 3);
  check(iFIFO.getValue(), 104, 4, 4);

  // Content (mid of buffer): empty
  check(iFIFO.hasValue(), false, 5, 0);
  check(iFIFO.hasValues(0), true, 5, 1);
  check(iFIFO.hasValues(1), false, 5, 2);
  check(iFIFO.hasValues(6), false, 5, 3);
  check(iFIFO.hasFreeCapacity(), true, 5, 4);
  check(iFIFO.hasFreeCapacity(0), true, 5, 5);
  check(iFIFO.hasFreeCapacity(1), true, 5, 6);
  check(iFIFO.hasFreeCapacity(5), true, 5, 7);
  check(iFIFO.hasFreeCapacity(6), true, 5, 8);
  check(iFIFO.getFreeCapacity(), 6, 5, 9);
  check(iFIFO.getUsedCapacity(), 0, 5, 10);

  check(iFIFO.putValue(105, false), true, 6, 0);
  // Content (mid of buffer): 105
  check(iFIFO.hasValue(), true, 6, 1);
  check(iFIFO.hasValues(0), true, 6, 2);
  check(iFIFO.hasValues(1), true, 6, 3);
  check(iFIFO.hasValues(2), false, 6, 4);
  check(iFIFO.hasValues(3), false, 6, 5);
  check(iFIFO.hasValues(4), false, 6, 6);
  check(iFIFO.hasValues(5), false, 6, 7);
  check(iFIFO.hasValues(6), false, 6, 8);
  check(iFIFO.hasValues(7), false, 6, 9);
  check(iFIFO.hasFreeCapacity(), true, 6, 10);
  check(iFIFO.hasFreeCapacity(0), true, 6, 11);
  check(iFIFO.hasFreeCapacity(1), true, 6, 12);
  check(iFIFO.hasFreeCapacity(2), true, 6, 13);
  check(iFIFO.hasFreeCapacity(3), true, 6, 14);
  check(iFIFO.hasFreeCapacity(4), true, 6, 15);
  check(iFIFO.hasFreeCapacity(5), true, 6, 16);
  check(iFIFO.hasFreeCapacity(6), false, 6, 17);
  check(iFIFO.hasFreeCapacity(7), false, 6, 18);
  check(iFIFO.getFreeCapacity(), 5, 6, 19);
  check(iFIFO.getUsedCapacity(), 1, 6, 20);
  check(iFIFO.valueAtPos(0), 0, 6, 21);
  check(iFIFO.valueAtPos(1), 105, 6, 22);
  check(iFIFO.valueAtPos(2), 0, 6, 23);

  check(iFIFO.putValue(106, false), true, 7, 0);
  // Content (mid of buffer): 105, 106
  check(iFIFO.hasValue(), true, 7, 1);
  check(iFIFO.hasValues(0), true, 7, 2);
  check(iFIFO.hasValues(1), true, 7, 3);
  check(iFIFO.hasValues(2), true, 7, 4);
  check(iFIFO.hasValues(3), false, 7, 5);
  check(iFIFO.hasValues(4), false, 7, 6);
  check(iFIFO.hasValues(5), false, 7, 7);
  check(iFIFO.hasValues(6), false, 7, 8);
  check(iFIFO.hasValues(7), false, 7, 9);
  check(iFIFO.hasFreeCapacity(), true, 7, 10);
  check(iFIFO.hasFreeCapacity(0), true, 7, 11);
  check(iFIFO.hasFreeCapacity(1), true, 7, 12);
  check(iFIFO.hasFreeCapacity(2), true, 7, 13);
  check(iFIFO.hasFreeCapacity(3), true, 7, 14);
  check(iFIFO.hasFreeCapacity(4), true, 7, 15);
  check(iFIFO.hasFreeCapacity(5), false, 7, 16);
  check(iFIFO.hasFreeCapacity(6), false, 7, 17);
  check(iFIFO.hasFreeCapacity(7), false, 7, 18);
  check(iFIFO.getFreeCapacity(), 4, 7, 19);
  check(iFIFO.getUsedCapacity(), 2, 7, 20);
  check(iFIFO.valueAtPos(0), 0, 7, 21);
  check(iFIFO.valueAtPos(1), 105, 7, 22);
  check(iFIFO.valueAtPos(2), 106, 7, 23);
  check(iFIFO.valueAtPos(3), 0, 7, 24);

  check(iFIFO.putValue(107, false), true, 8, 0);
  check(iFIFO.putValue(108, false), true, 8, 1);
  check(iFIFO.putValue(109, false), true, 8, 2);
  // Content (mid of buffer): 105, 106, 107, 108, 109
  check(iFIFO.hasValue(), true, 8, 3);
  check(iFIFO.hasValues(0), true, 8, 4);
  check(iFIFO.hasValues(1), true, 8, 5);
  check(iFIFO.hasValues(2), true, 8, 6);
  check(iFIFO.hasValues(3), true, 8, 7);
  check(iFIFO.hasValues(4), true, 8, 8);
  check(iFIFO.hasValues(5), true, 8, 9);
  check(iFIFO.hasValues(6), false, 8, 10);
  check(iFIFO.hasValues(7), false, 8, 11);
  check(iFIFO.hasFreeCapacity(), true, 8, 12);
  check(iFIFO.hasFreeCapacity(0), true, 8, 13);
  check(iFIFO.hasFreeCapacity(1), true, 8, 14);
  check(iFIFO.hasFreeCapacity(2), false, 8, 15);
  check(iFIFO.hasFreeCapacity(3), false, 8, 16);
  check(iFIFO.hasFreeCapacity(4), false, 8, 17);
  check(iFIFO.hasFreeCapacity(5), false, 8, 18);
  check(iFIFO.hasFreeCapacity(6), false, 8, 19);
  check(iFIFO.hasFreeCapacity(7), false, 8, 20);
  check(iFIFO.getFreeCapacity(), 1, 8, 21);
  check(iFIFO.getUsedCapacity(), 5, 8, 22);
  check(iFIFO.valueAtPos(0), 0, 8, 23);
  check(iFIFO.valueAtPos(1), 105, 8, 24);
  check(iFIFO.valueAtPos(2), 106, 8, 25);
  check(iFIFO.valueAtPos(3), 107, 8, 26);
  check(iFIFO.valueAtPos(4), 108, 8, 27);
  check(iFIFO.valueAtPos(5), 109, 8, 28);
  check(iFIFO.valueAtPos(6), 0, 8, 29);

  check(iFIFO.putValue(30000, false), true, 9, 0);
  // Content (mid of buffer): 105, 106, 107, 108, 109, 30000
  check(iFIFO.hasValue(), true, 9, 1);
  check(iFIFO.hasValues(0), true, 9, 2);
  check(iFIFO.hasValues(1), true, 9, 3);
  check(iFIFO.hasValues(2), true, 9, 4);
  check(iFIFO.hasValues(3), true, 9, 5);
  check(iFIFO.hasValues(4), true, 9, 6);
  check(iFIFO.hasValues(5), true, 9, 7);
  check(iFIFO.hasValues(6), true, 9, 8);
  check(iFIFO.hasValues(7), false, 9, 9);
  check(iFIFO.hasFreeCapacity(), false, 9, 10);
  check(iFIFO.hasFreeCapacity(0), true, 9, 11);
  check(iFIFO.hasFreeCapacity(1), false, 9, 12);
  check(iFIFO.hasFreeCapacity(2), false, 9, 13);
  check(iFIFO.hasFreeCapacity(3), false, 9, 14);
  check(iFIFO.hasFreeCapacity(4), false, 9, 15);
  check(iFIFO.hasFreeCapacity(5), false, 9, 16);
  check(iFIFO.hasFreeCapacity(6), false, 9, 17);
  check(iFIFO.hasFreeCapacity(7), false, 9, 18);
  check(iFIFO.getFreeCapacity(), 0, 9, 19);
  check(iFIFO.getUsedCapacity(), 6, 9, 20);
  check(iFIFO.valueAtPos(0), 0, 9, 21);
  check(iFIFO.valueAtPos(1), 105, 9, 22);
  check(iFIFO.valueAtPos(2), 106, 9, 23);
  check(iFIFO.valueAtPos(3), 107, 9, 24);
  check(iFIFO.valueAtPos(4), 108, 9, 25);
  check(iFIFO.valueAtPos(5), 109, 9, 26);
  check(iFIFO.valueAtPos(6), 30000, 9, 27);
  check(iFIFO.valueAtPos(7), 0, 9, 28);

  check(iFIFO.putValue(-111, false), false, 10, 0);
  // Content (mid of buffer): 105, 106, 107, 108, 109, 30000
  check(iFIFO.getFreeCapacity(), 0, 10, 1);
  check(iFIFO.getUsedCapacity(), 6, 10, 2);
  check(iFIFO.valueAtPos(0), 0, 10, 3);
  check(iFIFO.valueAtPos(1), 105, 10, 4);
  check(iFIFO.valueAtPos(2), 106, 10, 5);
  check(iFIFO.valueAtPos(3), 107, 10, 6);
  check(iFIFO.valueAtPos(4), 108, 10, 7);
  check(iFIFO.valueAtPos(5), 109, 10, 8);
  check(iFIFO.valueAtPos(6), 30000, 10, 9);
  check(iFIFO.valueAtPos(7), 0, 10, 10);

  check(iFIFO.putValue(-111, true), true, 11, 0);
  // Content (mid of buffer): 105, 106, 107, 108, 109, -111
  check(iFIFO.getFreeCapacity(), 0, 11, 1);
  check(iFIFO.getUsedCapacity(), 6, 11, 2);
  check(iFIFO.valueAtPos(0), 0, 11, 3);
  check(iFIFO.valueAtPos(1), 105, 11, 4);
  check(iFIFO.valueAtPos(2), 106, 11, 5);
  check(iFIFO.valueAtPos(3), 107, 11, 6);
  check(iFIFO.valueAtPos(4), 108, 11, 7);
  check(iFIFO.valueAtPos(5), 109, 11, 8);
  check(iFIFO.valueAtPos(6), -111, 11, 9);
  check(iFIFO.valueAtPos(7), 0, 10, 11);

  // Performance
  Serial.println("\r\nPerformance");

  // reset
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.reset();
  }
  _diff = micros()-_micros;
  Serial.print("reset 10000: ");
  Serial.println(_diff);

  // put linear
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.reset();
    iFIFO.putValue(100, false);
    iFIFO.putValue(101, false);
    iFIFO.putValue(102, false);
    iFIFO.putValue(103, false);
    iFIFO.putValue(104, false);
    iFIFO.putValue(105, false);
  }
  _diff = micros()-_micros;
  Serial.print("putValue 60000: ");
  Serial.println(_diff);

  // put with override
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.putValue(106, true);
  }
  _diff = micros()-_micros;
  Serial.print("putValue 10000 over: ");
  Serial.println(_diff);

  // put without override
  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.putValue(106, false);
  }
  _diff = micros()-_micros;
  Serial.print("putValue 10000 no over: ");
  Serial.println(_diff);

  // valueAtPos (from Start)
  iFIFO.reset();
  iFIFO.putValue(101,true);
  iFIFO.putValue(102,true);
  iFIFO.putValue(103,true);
  iFIFO.putValue(104,true);
  iFIFO.putValue(105,true);
  iFIFO.putValue(106,true);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.valueAtPos(1);
    iFIFO.valueAtPos(2);
    iFIFO.valueAtPos(3);
    iFIFO.valueAtPos(4);
    iFIFO.valueAtPos(5);
    iFIFO.valueAtPos(6);
  }
  _diff = micros()-_micros;
  Serial.print("valueAt Start 60000: ");
  Serial.println(_diff);

  // valueAtPos (from Middle)
  iFIFO.reset();
  iFIFO.putValue(101,true);
  iFIFO.putValue(102,true);
  iFIFO.putValue(103,true);
  iFIFO.putValue(104,true);
  iFIFO.putValue(105,true);
  iFIFO.putValue(106,true);
  iFIFO.getValue();
  iFIFO.getValue();
  iFIFO.getValue();
  iFIFO.putValue(107,true);
  iFIFO.putValue(108,true);
  iFIFO.putValue(109,true);

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.valueAtPos(1);
    iFIFO.valueAtPos(2);
    iFIFO.valueAtPos(3);
    iFIFO.valueAtPos(4);
    iFIFO.valueAtPos(5);
    iFIFO.valueAtPos(6);
  }
  _diff = micros()-_micros;
  Serial.print("valueAt Mid 60000: ");
  Serial.println(_diff);

  // putValue and getLast (from Middle)
  iFIFO.reset();
  iFIFO.putValue(101,true);
  iFIFO.putValue(102,true);
  iFIFO.putValue(103,true);
  iFIFO.putValue(104,true);
  iFIFO.getValue();
  iFIFO.getValue();
  iFIFO.getValue();
  iFIFO.getValue();

  _micros = micros();
  for (int i=0 ; i<10000; i++){
    iFIFO.putValue(101,true);
    iFIFO.putValue(102,true);
    iFIFO.putValue(103,true);
    iFIFO.putValue(104,true);
    iFIFO.putValue(105,true);
    iFIFO.putValue(106,true);
    iFIFO.getValue();
    iFIFO.getValue();
    iFIFO.getValue();
    iFIFO.getValue();
    iFIFO.getValue();
    iFIFO.getValue();
  }
  _diff = micros()-_micros;
  Serial.print("putValue/getLast Mid 120000: ");
  Serial.println(_diff);

  // Füllen für Ausgabe
  iFIFO.putValue(97,true);
  iFIFO.putValue(98,true);
  iFIFO.putValue(99,true);
  iFIFO.putValue(100,true);
  iFIFO.putValue(101,true);
  iFIFO.putValue(102,true);
}

// the loop function runs over and over again forever
void loop() {
  current = micros();

  if(iFIFO.hasValue() && current - lastOutput > 50){
    lastOutput = current;
    Serial.print((char)iFIFO.getValue());
  }

}

