
#ifndef jm_PCF8574_h
#define jm_PCF8574_h

// PCF8574 - PCF8574A
// ==================
// Remote 8-bit I/O expander for I2C-bus with interrupt

// http://www.ti.com/lit/ml/scyb031/scyb031.pdf
// http://www.ti.com/lit/ds/symlink/pcf8574.pdf
// https://www.nxp.com/docs/en/data-sheet/PCF8574_PCF8574A.pdf

// PCF8574 I2C address: 0x20..0x27
// PCF8574A I2C address: 0x38..0x3F

// Quasi-bidirectional I/Os: P0..P7

#include <Arduino.h> 			// HIGH, LOW, OUTPUT, INPUT...
#include <Wire.h>				// TwoWire class, Wire object

class jm_PCF8574
{
protected:

	TwoWire & _wire;			// Arduino Wire I2C bus
	byte _i2c_address;			// Device I2C address
	bool _connected;			// Device ready and not errored

	byte _port_mode;			// Digital I/O mode P0-P7
	byte _port_shadow;			// Digital I/O copy P0-P7

	bool _begin(byte port_mode, byte port_shadow); // return OK

public:

	TwoWire & wire() {return _wire;}
	byte i2c_address() {return _i2c_address;}
	bool connected() {return _connected;}

	byte port_mode() {return _port_mode;}
	void port_mode(byte port_mode) {_port_mode = port_mode;}

	byte port_shadow() {return _port_shadow;}
	void port_shadow(byte port_shadow) {_port_shadow = port_shadow;}

	static byte pin_mask(byte pin) {return (1 << pin);} // get mask of pin
	static int mask_pin(byte mask) // retrieve pin of mask or -1 (fail)
		{int pin; for (pin=-1; mask>0; mask>>=1, pin++); return pin;}

	jm_PCF8574();
	jm_PCF8574(byte i2c_address);
	jm_PCF8574(byte i2c_address, TwoWire & wire);

	operator bool() {return connected();}

	bool begin(); // return OK
	bool begin(byte port_mode, byte port_shadow); // return OK
	bool begin(byte i2c_address); // return OK
	bool begin(byte i2c_address, byte port_mode, byte port_shadow); // return OK
	bool end(); // return OK

	byte port_input(); // input port
	void port_output(byte value); // output port
	void port_output(const byte *data, size_t quantity); // output port

	byte pin_mode(byte pin); // get mode of pin
	void pin_mode(byte pin, byte mode); // set mode of pin

	bool pin_input(byte pin); // input pin
	void pin_output(byte pin, bool state); // output pin
	void pin_set(byte pin); // set pin
	void pin_clear(byte pin); // clear pin
	void pin_toggle(byte pin); // toogle pin

	bool wait(uint16_t us);

	// Arduino Digital...

	void pinMode(byte pin, byte mode);
	int digitalRead(byte pin);
	void digitalWrite(byte pin, byte value);

	// jm_Pin...

	static void obj_pinMode(void * obj, byte pin, byte mode);
	static int obj_digitalRead(void * obj, byte pin);
	static void obj_digitalWrite(void * obj, byte pin, byte value);
};

#endif // jm_PCF8574_h
