
#include <jm_PCF8574.h>

// ---------------------------------------------------------------------------

bool jm_PCF8574::_begin(byte port_mode, byte port_shadow) // return OK
{
	if (_i2c_address == 0x00) return false; // I2C address unset ? !OK

	// set outputs to a known state...

	_port_mode = port_mode; // set mode of all INPUTs (bit==0) and OUTPUTs (bit==1)
	_port_shadow = port_shadow | ~port_mode; // set copy of all OUTPUTs and force HIGH all INPUTs
	// remember that OUTPUT acts as OPEN_DRAIN and INPUT acts as INPUT_PULLUP.

	_connected = true;

	_wire.beginTransmission(_i2c_address);
	_wire.write(_port_shadow);
	_connected &= (_wire.endTransmission() == (byte) 0);

	return _connected;
}

//------------------------------------------------------------------------------

jm_PCF8574::jm_PCF8574() :
	_wire(Wire),				// default Arduino Wire I2C bus
	_i2c_address(0x00),			// undefine Device I2C address
	_connected(false),			// Device not connected (disconnected)
	_port_mode(0x00),			// Digital I/O mode P0-P7 as INPUT
	_port_shadow(0xFF)			// Digital I/O data P0-P7 as INPUT
{
}

jm_PCF8574::jm_PCF8574(byte i2c_address) :
	_wire(Wire),				// default Arduino Wire I2C bus
	_i2c_address(i2c_address),	// define Device I2C address
	_connected(false),			// Device not connected (disconnected)
	_port_mode(0x00),			// Digital I/O mode P0-P7 as INPUT
	_port_shadow(0xFF)			// Digital I/O data P0-P7 as INPUT
{
}

jm_PCF8574::jm_PCF8574(byte i2c_address, TwoWire & wire) :
	_wire(wire),				// select Arduino Wire I2C bus
	_i2c_address(i2c_address),	// define Device I2C address
	_connected(false),			// Device not connected (disconnected)
	_port_mode(0x00),			// Digital I/O mode P0-P7 as INPUT
	_port_shadow(0xFF)			// Digital I/O data P0-P7 as INPUT
{
}

// ---------------------------------------------------------------------------

bool jm_PCF8574::begin() // return OK
{
	return begin(0x00, 0xFF); // default all pins as INPUT (INPUT_PULLUP)
}

bool jm_PCF8574::begin(byte port_mode, byte port_shadow) // return OK
{
	if (_connected) return false; // allready connected ? !OK

	return _begin(port_mode, port_shadow);
}

bool jm_PCF8574::begin(byte i2c_address) // return OK
{
	return begin(i2c_address, 0x00, 0xFF); // default all pins as INPUT (INPUT_PULLUP)
}

bool jm_PCF8574::begin(byte i2c_address, byte port_mode, byte port_shadow) // return OK
{
	if (_connected) return false; // allready connected ? !OK

	_i2c_address = i2c_address;

	return _begin(port_mode, port_shadow);
}

bool jm_PCF8574::end() // return OK
{
	if (!_connected) return false; // already disconnected ? !OK

	_connected = false;

	return true;
}

// ---------------------------------------------------------------------------

byte jm_PCF8574::port_input() // input port
{
	if (!_connected) return _port_shadow;

	if (_wire.requestFrom(_i2c_address, (byte) 1) != (byte) 1) {_connected = false; return _port_shadow;}

	_port_shadow = _wire.read(); // update shadow; never fails just after Wire.requestFrom()

	return _port_shadow;
}

void jm_PCF8574::port_output(byte value) // output port
{
	if (!_connected) return;

	_wire.beginTransmission(_i2c_address);

//	if (_wire.write(value) != 1) {_connected = false; return;}
	_wire.write(value); // never fails just after Wire.beginTransmission()

	if (_wire.endTransmission() != (byte) 0) {_connected = false; return;}

	_port_shadow = value; // update shadow
}

void jm_PCF8574::port_output(const byte *data, size_t quantity) // output port
{
	if (!_connected) return;

	_wire.beginTransmission(_i2c_address);

	if (_wire.write(data, quantity) != quantity) {_connected = false; return;}

	if (_wire.endTransmission() != (byte) 0) {_connected = false; return;}

	if (quantity > 0) _port_shadow = data[quantity - 1]; // update shadow
}

// ---------------------------------------------------------------------------

byte jm_PCF8574::pin_mode(byte pin) // get mode of pin
{
	if (_port_mode & (1 << pin))
		return 1; // OPEN_DRAIN
	else
		return 0; // INPUT_PULLUP
}

void jm_PCF8574::pin_mode(byte pin, byte mode) // set mode of pin
{
	if (mode == 1)
		_port_mode |= (1 << pin); // OPEN_DRAIN
	else
		_port_mode &= ~(1 << pin); // INPUT_PULLUP
}

bool jm_PCF8574::pin_input(byte pin) // input pin
{
	byte port_data = port_input();

	byte pin_mask = (1 << pin);
	byte pin_data = (port_data & pin_mask);

	return ((bool) pin_data);
}

//void jm_PCF8574::pin_output(byte pin, bool state) // output pin
//{
//	byte pin_mask = (1 << pin);
//	byte pin_mode = pin_mask; // pin mode as OUTPUT
//	byte pin_data = (state ? pin_mask : 0);
//
//	_port_mode |= pin_mode; // port mode updated
//
//	port_output(pin_data | (_port_shadow & ~pin_mask));
//}

void jm_PCF8574::pin_output(byte pin, bool state) // output pin
{
	if (state)
		pin_set(pin);
	else
		pin_clear(pin);
}

void jm_PCF8574::pin_set(byte pin) // set pin
{
	byte pin_mask = (1 << pin);
	byte pin_mode = pin_mask; // OPEN_DRAIN

	_port_mode |= pin_mode; // port mode updated

	port_output(_port_shadow | pin_mask);
}

void jm_PCF8574::pin_clear(byte pin) // clear pin
{
	byte pin_mask = (1 << pin);
	byte pin_mode = pin_mask; // OPEN_DRAIN

	_port_mode |= pin_mode; // port mode updated

	port_output(_port_shadow & ~pin_mask);
}

void jm_PCF8574::pin_toggle(byte pin) // toogle pin
{
	byte pin_mask = (1 << pin);
	byte pin_mode = pin_mask; // OPEN_DRAIN

	_port_mode |= pin_mode; // port mode updated

	port_output(_port_shadow ^ pin_mask);
}

// ---------------------------------------------------------------------------

bool jm_PCF8574::wait(uint16_t us)
{
	if (us > 100) delayMicroseconds(us - 100);

	return true;
}

// ---------------------------------------------------------------------------

// Arduino Digital...

void jm_PCF8574::pinMode(byte pin, byte mode)
{
	if (mode == OUTPUT)
		pin_mode(pin, 1); // OPEN_DRAIN
	else
		pin_mode(pin, 0); // INPUT_PULLUP
}

int jm_PCF8574::digitalRead(byte pin)
{
	bool state = pin_input(pin);
	return (_connected ? (state ? HIGH : LOW) : LOW);
}

void jm_PCF8574::digitalWrite(byte pin, byte value)
{
	pin_output(pin, (value==HIGH));
}

// ---------------------------------------------------------------------------

// jm_Pin...

// static functions...

void jm_PCF8574::obj_pinMode(void * obj, byte pin, byte mode)
{
	if ((jm_PCF8574 *) obj) ((jm_PCF8574 *) obj)->pinMode(pin, mode);
}

int jm_PCF8574::obj_digitalRead(void * obj, byte pin)
{
	if ((jm_PCF8574 *) obj) return ((jm_PCF8574 *) obj)->digitalRead(pin); else return LOW;
}

void jm_PCF8574::obj_digitalWrite(void * obj, byte pin, byte value)
{
	if ((jm_PCF8574 *) obj) ((jm_PCF8574 *) obj)->digitalWrite(pin, value);
}

//------------------------------------------------------------------------------

// END.
