#include <Arduino.h>

#include "AnalogControlPanel.h"

/* =========================================================================
Exponential smoothing example.
Exponential smoothing is a way of "smoothing" a time series of readings by
applying more weight to the average of past readings than to the latest reading.

We take readings every 2000 milliseconds and print the current reading and
the smoothed reading.

Each reading
========================================================================= */

#define LDR_SUPPLY_PIN  3
#define LDR_READ_PIN   A3

// =============================================================================
// Reading Intervals: when to take readings
unsigned long nextReadingTime = 0;
static const unsigned long readingInterval = 2000;  // milliseconds

// =============================================================================
// Exponential Smoothing:
// Alpha: the weight we give to the current reading vs. 
// the smoothed history to date, which is weighted (1 - alpha).
const static float alpha = 0.1;
static float smoothed = -1.0; // impossible value first.

float expSmooth(float latest)
{
    if (smoothed < 0)
        {smoothed = latest; return latest;}
    
    smoothed = ((1.0 - alpha) * smoothed) + (alpha * latest);
    return smoothed;
}

// =============================================================================
void setup()
{
    Serial.begin(9600);
    InternalADC.begin();
    InternalADC.speed2x();  // each read takes about 55 microseconds not 110.
    nextReadingTime = millis() + readingInterval;
}

void loop()
{
    if (millis() >= nextReadingTime)
    {
        nextReadingTime += readingInterval;
        float currentReading  = getCurrentReading();
        float smoothedReading = expSmooth(currentReading);
        printCurrentAndSmoothed(currentReading, smoothedReading);
    }
    // Do other stuff as required.
}

// =============================================================================

int medianfilter(int* threeRdgs)
{
    // Find which is the minimum and which the Maximum,
    // and return the one that isn't either of those.
    int8_t m = 0, M = 0;
    int mVal = threeRdgs[0];
    int MVal = threeRdgs[0];
    if (threeRdgs[1] < mVal) {m = 1; mVal = threeRdgs[1];}
    if (threeRdgs[1] > MVal) {M = 1; MVal = threeRdgs[1];}
    if (threeRdgs[2] < mVal) {m = 2;}
    if (threeRdgs[2] > MVal) {M = 2;}

    if ((2 != m) && (2 != M)) return threeRdgs[2];
    if ((1 != m) && (1 != M)) return threeRdgs[1];
    return threeRdgs[0];
}

float medianOf3(void)
{
    // Median-filter three readings for better noise rejection.
    int rdgs[3] {0, 0, 0};
    for (size_t i = 0; i < (sizeof(rdgs) / sizeof(rdgs[0])); i++)
        {rdgs[i] = InternalADC.read();}
    return (float) medianfilter(rdgs);
}

float getCurrentReading(void)
{
    // Supply power to the LDR
    pinMode(LDR_SUPPLY_PIN, OUTPUT);
    digitalWrite(LDR_SUPPLY_PIN, HIGH); // supply LDR with power

    // LDRs take a while to stabilise - milliseconds.
    delay(50);

    InternalADC.usePin(LDR_READ_PIN);

    float theReading = medianOf3();

    InternalADC.freePin(LDR_READ_PIN);

    // Stop supplying power.
    digitalWrite(LDR_SUPPLY_PIN, LOW);
    pinMode(LDR_SUPPLY_PIN, INPUT);

    return theReading;
}

void printCurrentAndSmoothed(float curr, float smooth)
{
    // Print results.
    static bool headerPrinted = false;
    if (!headerPrinted)
    {
        Serial.println(F("t (seconds)\tcurrent\tsmoothed"));
        headerPrinted = true;
    }
    Serial.print(millis() / 1000);
    Serial.print('\t');
    Serial.print(curr, 3);
    Serial.print('\t');
    Serial.print(smooth, 3);
    Serial.println();
}