#if defined(ARDUINO_ARCH_AVR)
#include "AVRStepperPins.h"
#include "FastAccelStepper.h"
#ifdef SIMULATOR
#include <avr/sleep.h>
#endif

// As in StepperDemo for Motor 1 on AVR
#define dirPinStepperX 5
#define enablePinStepperX 6
#define stepPinStepperX stepPinStepper1A

// As in StepperDemo for Motor 2 on AVR
#define dirPinStepperY 7
#define enablePinStepperY 8
#define stepPinStepperY stepPinStepper1B

FastAccelStepperEngine engine = FastAccelStepperEngine();
FastAccelStepper *stepperX = NULL;
FastAccelStepper *stepperY = NULL;

struct control_s {
  uint64_t time;
  uint16_t phi;
  uint16_t drift;  // time delta in ticks
};
struct control_s controlX = {.time = 0, .phi = 0, .drift = 0};
struct control_s controlY = {.time = 0, .phi = 90, .drift = 0};

// This means, the circle is executed in 360*4ms = 1440ms
#ifndef TIMESTEP_TICKS
#define TIMESTEP_TICKS (TICKS_PER_S / 250)
#endif

// Table generated by python:
// [int(math.sin(x*math.pi/180)*1600) for x in range(0,91)]
int16_t steps[91] = {
    0,    27,   55,   83,   111,  139,  167,  194,  222,  250,  277,  305,
    332,  359,  387,  414,  441,  467,  494,  520,  547,  573,  599,  625,
    650,  676,  701,  726,  751,  775,  799,  824,  847,  871,  894,  917,
    940,  962,  985,  1006, 1028, 1049, 1070, 1091, 1111, 1131, 1150, 1170,
    1189, 1207, 1225, 1243, 1260, 1277, 1294, 1310, 1326, 1341, 1356, 1371,
    1385, 1399, 1412, 1425, 1438, 1450, 1461, 1472, 1483, 1493, 1503, 1512,
    1521, 1530, 1538, 1545, 1552, 1558, 1565, 1570, 1575, 1580, 1584, 1588,
    1591, 1593, 1596, 1597, 1599, 1599, 1600};

void setup() {
  Serial.begin(115200);
  Serial.println("Init");
  engine.init();
  stepperX = engine.stepperConnectToPin(stepPinStepperX);
  stepperY = engine.stepperConnectToPin(stepPinStepperY);
  if (!stepperX || !stepperY) {
    while (0 == 0) {
      Serial.println("Cannot initialize steppers");
    }
  }
  stepperX->setDirectionPin(dirPinStepperX);
  stepperX->setEnablePin(enablePinStepperX);
  stepperX->setAutoEnable(false);
  stepperY->setDirectionPin(dirPinStepperY);
  stepperY->setEnablePin(enablePinStepperY);
  stepperY->setAutoEnable(false);

  stepperY->setCurrentPosition(steps[90]);

  // Without auto enable, need to enable manually both steppers
  stepperX->enableOutputs();
  stepperY->enableOutputs();

  // Fill the queue with 1ms delay and start the queues.
  stepperX->moveTimed(0, TICKS_PER_S / 1000, NULL, false);
  stepperY->moveTimed(0, TICKS_PER_S / 1000, NULL, false);

  // Start the queues. Currently best but not perfect synchronicity
  stepperX->moveTimed(0, 0, NULL, true);
  stepperY->moveTimed(0, 0, NULL, true);
}

void moveStepper(FastAccelStepper *stepper, struct control_s *control) {
  uint16_t phi = control->phi % 360;

  // The table contains only one quadrant
  uint16_t index = phi;
  bool negate = false;
  if (index > 180) {
    negate = true;
    index = 360 - index;
  }
  if (index > 90) {
    index = 180 - index;
  }
  int16_t position = steps[index];
  if (negate) {
    position = -position;
  }
  int32_t current_position = stepper->getPositionAfterCommandsCompleted();
  int32_t steps = position - current_position;
  uint32_t actual;
  MoveTimedResultCode rc;
  uint32_t duration = TIMESTEP_TICKS + control->drift;
  rc = stepper->moveTimed(steps, duration, &actual, true);
  switch (rc) {
    case MOVE_TIMED_EMPTY:
      Serial.print("Empty:");
      Serial.println(stepper->getStepPin() == dirPinStepperX ? 'X' : 'Y');
      /* fallthrough */
    case MOVE_TIMED_OK:
      control->drift = duration - actual;
      control->time += actual;
      control->phi += 1;
      break;
    case MOVE_TIMED_BUSY:
      // Serial.println("Busy");
      break;
    case MOVE_TIMED_TOO_LARGE_ERROR:
      Serial.println("Too large");
      break;
    case MoveTimedResultCode::ErrorTicksTooLow:
      Serial.print("Ticks too low:");
      Serial.print(duration / steps);
      Serial.print(" steps:");
      Serial.println(steps);
    default:
      Serial.println(toString(rc));
  }
}

uint32_t last_millis = 0;

void loop() {
  if (last_millis != millis()) {
    Serial.print(uint32_t(controlX.time));
    Serial.print(' ');
    Serial.print(uint32_t(controlY.time));
    Serial.print(' ');
    Serial.print(stepperX->getPositionAfterCommandsCompleted());
    Serial.print(' ');
    Serial.print(stepperY->getPositionAfterCommandsCompleted());
    Serial.print(' ');
    Serial.println(millis());
    last_millis = millis();
  }

  if (controlX.time < controlY.time) {
    moveStepper(stepperX, &controlX);
  } else {
    moveStepper(stepperY, &controlY);
  }
  if ((controlX.phi != 361) && (controlY.phi != 451)) {
    return;
  }
  // Full round should be completed
#ifdef SIMULATOR
  delay(1000);
  noInterrupts();
  sleep_cpu();
#endif
}
#else
void setup() {}
void loop() {}
#endif
