/*
 * SPDX-FileCopyrightText: 2024-2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * @file
 * @brief ESP LCD: ILI9881C
 */

#pragma once

#include <stdint.h>
#include "soc/soc_caps.h"

#if SOC_MIPI_DSI_SUPPORTED
#include "esp_lcd_panel_vendor.h"
#include "esp_lcd_mipi_dsi.h"

#ifdef __cplusplus
extern "C" {
#endif

#define ESP_LCD_ILI9881C_VER_MAJOR    (1)
#define ESP_LCD_ILI9881C_VER_MINOR    (0)
#define ESP_LCD_ILI9881C_VER_PATCH    (1)

/**
 * @brief Create LCD panel for model ILI9881C
 *
 * @note  Vendor specific initialization can be different between manufacturers, should consult the LCD supplier for initialization sequence code.
 *
 * @param[in]  io LCD panel IO handle
 * @param[in]  panel_dev_config General panel device configuration
 * @param[out] ret_panel Returned LCD panel handle
 * @return
 *      - ESP_ERR_INVALID_ARG   if parameter is invalid
 *      - ESP_OK                on success
 *      - Otherwise             on fail
 */
esp_err_t esp_lcd_new_panel_ili9881c(const esp_lcd_panel_io_handle_t io, const esp_lcd_panel_dev_config_t *panel_dev_config,
                                     esp_lcd_panel_handle_t *ret_panel);

/**
 * @brief MIPI-DSI bus configuration structure
 *
 */
#define ILI9881C_PANEL_BUS_DSI_2CH_CONFIG()              \
    {                                                    \
        .bus_id = 0,                                     \
        .num_data_lanes = 2,                             \
        .phy_clk_src = MIPI_DSI_PHY_CLK_SRC_DEFAULT,     \
        .lane_bit_rate_mbps = 1000,                      \
    }

/**
 * @brief MIPI-DBI panel IO configuration structure
 *
 */
#define ILI9881C_PANEL_IO_DBI_CONFIG()  \
    {                                   \
        .virtual_channel = 0,           \
        .lcd_cmd_bits = 8,              \
        .lcd_param_bits = 8,            \
    }

/**
 * @brief MIPI DPI configuration structure
 *
 * @note  refresh_rate = (dpi_clock_freq_mhz * 1000000) / (h_res + hsync_pulse_width + hsync_back_porch + hsync_front_porch)
 *                                                      / (v_res + vsync_pulse_width + vsync_back_porch + vsync_front_porch)
 *
 * @param[in] px_format Pixel format of the panel
 *
 */
#define ILI9881C_800_1280_PANEL_60HZ_DPI_CONFIG(px_format) \
    {                                                      \
        .dpi_clk_src = MIPI_DSI_DPI_CLK_SRC_DEFAULT,       \
        .dpi_clock_freq_mhz = 80,                          \
        .virtual_channel = 0,                              \
        .pixel_format = px_format,                         \
        .num_fbs = 1,                                      \
        .video_timing = {                                  \
            .h_size = 800,                                 \
            .v_size = 1280,                                \
            .hsync_back_porch = 140,                       \
            .hsync_pulse_width = 40,                       \
            .hsync_front_porch = 40,                       \
            .vsync_back_porch = 16,                        \
            .vsync_pulse_width = 4,                        \
            .vsync_front_porch = 16,                       \
        },                                                 \
        .flags.use_dma2d = true,                           \
    }

#ifdef __cplusplus
}
#endif
#endif // SOC_MIPI_DSI_SUPPORTED
