#!/usr/bin/env python3
"""Builds 'bdfconv' from u8g2 source tree and converts .bdf files
(X11 bitmap format) to u8g2-format source files."""

import argparse
import collections
import json
import logging
import ok_logging_setup
import ok_subprocess_defaults
import re
import textwrap
from natsort import natsorted, ns
from pathlib import Path

ok_logging_setup.install()

top_dir = Path(__file__).parent.parent
try: top_dir = top_dir.relative_to(Path.cwd())
except ValueError: pass

parser = argparse.ArgumentParser()
parser.add_argument("--u8g2_repo", type=Path, default=top_dir / ".." / "u8g2")
args = parser.parse_args()
subproc = ok_subprocess_defaults.SubprocessDefaults()

#
# build bdfconv
#

build_dir = top_dir / "build.tmp"
build_dir.mkdir(exist_ok=True)

bdfconv_dir = args.u8g2_repo / "tools" / "font" / "bdfconv"
bdfconv_sources = list(natsorted(bdfconv_dir.glob("*.c"), alg=ns.I | ns.P))
if not bdfconv_sources: ok_logging_setup.exit(f"No *.c: {bdfconv_dir}")
print(bdfconv_sources)

bdfconv = build_dir / "bdfconv"
logging.info(f"\n🏗️ Building: {bdfconv}")
subproc.run("gcc", "-O4", f"-I{bdfconv_dir}", *bdfconv_sources, "-o", bdfconv)

#
# build conversion list
#

bdf_dir = top_dir / "produced_bdfs"
bdf_paths = list(natsorted(bdf_dir.glob("*.bdf"), alg=ns.I | ns.P))
if not bdf_paths: ok_logging_setup.exit(f"No *.bdf: {bdf_dir}")

Font = collections.namedtuple("Font", "bdf_path name symbol")

fonts = []
latest_version = {}
for bdf in bdf_paths:
    m = re.match(r"((\w+)-v\d[\w.]*)?(-\d+)?\.bdf", bdf.name)
    if not m: ok_logging_setup.exit(f"Bad name: {bdf}")
    symbol, unversioned = (re.sub(r"\W+", "_", m[g]) + "_u8g2" for g in (1, 2))
    fonts.append(Font(bdf, m[1], symbol))
    latest_version[unversioned] = symbol

    # TODO: make 7-bit-ASCII reduced version? other reduced versions?

#
# write header file
#

out_source_dir = top_dir / "src"
header_path = out_source_dir / "everyday_pixel_fonts.h"
logging.info(f"🇭 Writing: {header_path}")

guard_macro = "INCLUDED_" + re.sub(r"\W+", "_", header_path.name.upper())
header_path.write_text(textwrap.dedent(f"""
    /* Generated by {Path(__file__).name} */

    #ifndef {guard_macro}
    #define {guard_macro}

    #include <stdint.h>

    #ifdef __cplusplus
    extern "C" {{
    #endif

    /* latest version of each font */
    {"\n    ".join(f"#define {n} {lv}" for n, lv in latest_version.items())}

    /* all versions of all fonts */
    {"\n    ".join(f"extern uint8_t const {f.symbol}[];" for f in fonts)}

    /* array of fonts for testing-- referencing this will link in all fonts! */
    struct everyday_entry {{ const char *name; const uint8_t *u8g2_font; }};
    extern const struct everyday_entry all_everyday_fonts[];

    #ifdef __cplusplus
    }}  /* extern "C" */
    #endif

    #endif  /* {guard_macro} */
""").lstrip("\n"))

#
# convert fonts
#

logging.info(f"🗑️ Cleaning: {out_source_dir}/*_u8g2.c*")
[p.unlink() for p in out_source_dir.glob("*_u8g2.c")]

for font in fonts:
    tmp_path = build_dir / "bdfconv-out.c"
    out_path = out_source_dir / f"{font.symbol}.c"

    logging.info(f"\n🔠 Creating: {out_path}")
    subproc.run(
        bdfconv, "-b0", "-f1", f"-n{font.symbol}",
        f"-o{tmp_path}", font.bdf_path
    )

    header = textwrap.dedent(f"""
        #include \"{header_path.name}\"

        #ifdef __GNUC__
        #define U8G2_FONT_SECTION(name) __attribute__((section(".text." name)))
        #else
        #define U8G2_FONT_SECTION(name)
        #endif

    """).lstrip("\n")
    out_path.write_text(header + tmp_path.read_text())

#
# write all-fonts-list source file
#

all_fonts_path = out_source_dir / "all_everyday_pixel_fonts.c"
logging.info(f"\n🌐 Writing: {all_fonts_path}")

all_fonts_path.write_text(textwrap.dedent(f"""
    /* Generated by {Path(__file__).name} */

    #include "{header_path.name}"

    #include <stddef.h>

    const struct everyday_entry all_everyday_fonts[] = {{
      {",\n      ".join(
          f"{{ {json.dumps(font.name)}, {font.symbol} }}"
          for font in fonts
      )},
      {{ NULL, NULL }}
    }};
""").lstrip("\n"))

logging.info(f"🏁 Done updating {len(fonts)} u8g2 font files")
