/*  libFLAC - Free Lossless Audio Codec library
    Copyright (C) 2000-2009  Josh Coalson
    Copyright (C) 2011-2016  Xiph.Org Foundation

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    - Neither the name of the Xiph.org Foundation nor the names of its
    contributors may be used to endorse or promote products derived from
    this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
    ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
    A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR
    CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
    EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
    PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
    PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
    LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
    NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef FLAC__STREAM_DECODER_H
#define FLAC__STREAM_DECODER_H

//#include <stdio.h> /* for FILE */
#include "export.h"
#include "format.h"

#ifdef __cplusplus
extern "C" {
#endif


/** \file include/FLAC/stream_decoder.h

    \brief
    This module contains the functions which implement the stream
    decoder.

    See the detailed documentation in the
    \link flac_stream_decoder stream decoder \endlink module.
*/

/** \defgroup flac_decoder FLAC/ \*_decoder.h: decoder interfaces
    \ingroup flac

    \brief
    This module describes the decoder layers provided by libFLAC.

    The stream decoder can be used to decode complete streams either from
    the client via callbacks, or directly from a file, depending on how
    it is initialized.  When decoding via callbacks, the client provides
    callbacks for reading FLAC data and writing decoded samples, and
    handling metadata and errors.  If the client also supplies seek-related
    callback, the decoder function for sample-accurate seeking within the
    FLAC input is also available.  When decoding from a file, the client
    needs only supply a filename or open \c FILE* and write/metadata/error
    callbacks; the rest of the callbacks are supplied internally.  For more
    info see the \link flac_stream_decoder stream decoder \endlink module.
*/

/** \defgroup flac_stream_decoder FLAC/stream_decoder.h: stream decoder interface
    \ingroup flac_decoder

    \brief
    This module contains the functions which implement the stream
    decoder.

    The stream decoder can decode native FLAC, and optionally Ogg FLAC
    (check FLAC_API_SUPPORTS_OGG_FLAC) streams and files.

    The basic usage of this decoder is as follows:
    - The program creates an instance of a decoder using
     FLAC__stream_decoder_new().
    - The program overrides the default settings using
     FLAC__stream_decoder_set_*() functions.
    - The program initializes the instance to validate the settings and
     prepare for decoding using
     - FLAC__stream_decoder_init_stream() or FLAC__stream_decoder_init_FILE()
       or FLAC__stream_decoder_init_file() for native FLAC,
     - FLAC__stream_decoder_init_ogg_stream() or FLAC__stream_decoder_init_ogg_FILE()
       or FLAC__stream_decoder_init_ogg_file() for Ogg FLAC
    - The program calls the FLAC__stream_decoder_process_*() functions
     to decode data, which subsequently calls the callbacks.
    - The program finishes the decoding with FLAC__stream_decoder_finish(),
     which flushes the input and output and resets the decoder to the
     uninitialized state.
    - The instance may be used again or deleted with
     FLAC__stream_decoder_delete().

    In more detail, the program will create a new instance by calling
    FLAC__stream_decoder_new(), then call FLAC__stream_decoder_set_*()
    functions to override the default decoder options, and call
    one of the FLAC__stream_decoder_init_*() functions.

    There are three initialization functions for native FLAC, one for
    setting up the decoder to decode FLAC data from the client via
    callbacks, and two for decoding directly from a FLAC file.

    For decoding via callbacks, use FLAC__stream_decoder_init_stream().
    You must also supply several callbacks for handling I/O.  Some (like
    seeking) are optional, depending on the capabilities of the input.

    For decoding directly from a file, use FLAC__stream_decoder_init_FILE()
    or FLAC__stream_decoder_init_file().  Then you must only supply an open
    \c FILE* or filename and fewer callbacks; the decoder will handle
    the other callbacks internally.

    There are three similarly-named init functions for decoding from Ogg
    FLAC streams.  Check \c FLAC_API_SUPPORTS_OGG_FLAC to find out if the
    library has been built with Ogg support.

    Once the decoder is initialized, your program will call one of several
    functions to start the decoding process:

    - FLAC__stream_decoder_process_single() - Tells the decoder to process at
     most one metadata block or audio frame and return, calling either the
     metadata callback or write callback, respectively, once.  If the decoder
     loses sync it will return with only the error callback being called.
    - FLAC__stream_decoder_process_until_end_of_metadata() - Tells the decoder
     to process the stream from the current location and stop upon reaching
     the first audio frame.  The client will get one metadata, write, or error
     callback per metadata block, audio frame, or sync error, respectively.
    - FLAC__stream_decoder_process_until_end_of_stream() - Tells the decoder
     to process the stream from the current location until the read callback
     returns FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM or
     FLAC__STREAM_DECODER_READ_STATUS_ABORT.  The client will get one metadata,
     write, or error callback per metadata block, audio frame, or sync error,
     respectively.

    When the decoder has finished decoding (normally or through an abort),
    the instance is finished by calling FLAC__stream_decoder_finish(), which
    ensures the decoder is in the correct state and frees memory.  Then the
    instance may be deleted with FLAC__stream_decoder_delete() or initialized
    again to decode another stream.

    Seeking is exposed through the FLAC__stream_decoder_seek_absolute() method.
    At any point after the stream decoder has been initialized, the client can
    call this function to seek to an exact sample within the stream.
    Subsequently, the first time the write callback is called it will be
    passed a (possibly partial) block starting at that sample.

    If the client cannot seek via the callback interface provided, but still
    has another way of seeking, it can flush the decoder using
    FLAC__stream_decoder_flush() and start feeding data from the new position
    through the read callback.

    The stream decoder also provides MD5 signature checking.  If this is
    turned on before initialization, FLAC__stream_decoder_finish() will
    report when the decoded MD5 signature does not match the one stored
    in the STREAMINFO block.  MD5 checking is automatically turned off
    (until the next FLAC__stream_decoder_reset()) if there is no signature
    in the STREAMINFO block or when a seek is attempted.

    The FLAC__stream_decoder_set_metadata_*() functions deserve special
    attention.  By default, the decoder only calls the metadata_callback for
    the STREAMINFO block.  These functions allow you to tell the decoder
    explicitly which blocks to parse and return via the metadata_callback
    and/or which to skip.  Use a FLAC__stream_decoder_set_metadata_respond_all(),
    FLAC__stream_decoder_set_metadata_ignore() ... or FLAC__stream_decoder_set_metadata_ignore_all(),
    FLAC__stream_decoder_set_metadata_respond() ... sequence to exactly specify
    which blocks to return.  Remember that metadata blocks can potentially
    be big (for example, cover art) so filtering out the ones you don't
    use can reduce the memory requirements of the decoder.  Also note the
    special forms FLAC__stream_decoder_set_metadata_respond_application(id)
    and FLAC__stream_decoder_set_metadata_ignore_application(id) for
    filtering APPLICATION blocks based on the application ID.

    STREAMINFO and SEEKTABLE blocks are always parsed and used internally, but
    they still can legally be filtered from the metadata_callback.

    \note
    The "set" functions may only be called when the decoder is in the
    state FLAC__STREAM_DECODER_UNINITIALIZED, i.e. after
    FLAC__stream_decoder_new() or FLAC__stream_decoder_finish(), but
    before FLAC__stream_decoder_init_*().  If this is the case they will
    return \c true, otherwise \c false.

    \note
    FLAC__stream_decoder_finish() resets all settings to the constructor
    defaults, including the callbacks.

    \{
*/


/** State values for a FLAC__StreamDecoder

    The decoder's state can be obtained by calling FLAC__stream_decoder_get_state().
*/
typedef enum {

    FLAC__STREAM_DECODER_SEARCH_FOR_METADATA = 0,
    /**< The decoder is ready to search for metadata. */

    FLAC__STREAM_DECODER_READ_METADATA,
    /**< The decoder is ready to or is in the process of reading metadata. */

    FLAC__STREAM_DECODER_SEARCH_FOR_FRAME_SYNC,
    /** < The decoder is ready to or is in the process of searching for the
        frame sync code.
    */

    FLAC__STREAM_DECODER_READ_FRAME,
    /**< The decoder is ready to or is in the process of reading a frame. */

    FLAC__STREAM_DECODER_END_OF_STREAM,
    /**< The decoder has reached the end of the stream. */

    FLAC__STREAM_DECODER_OGG_ERROR,
    /**< An error occurred in the underlying Ogg layer.  */

    FLAC__STREAM_DECODER_SEEK_ERROR,
    /** < An error occurred while seeking.  The decoder must be flushed
        with FLAC__stream_decoder_flush() or reset with
        FLAC__stream_decoder_reset() before decoding can continue.
    */

    FLAC__STREAM_DECODER_ABORTED,
    /**< The decoder was aborted by the read or write callback. */

    FLAC__STREAM_DECODER_MEMORY_ALLOCATION_ERROR,
    /** < An error occurred allocating memory.  The decoder is in an invalid
        state and can no longer be used.
    */

    FLAC__STREAM_DECODER_UNINITIALIZED
    /** < The decoder is in the uninitialized state; one of the
        FLAC__stream_decoder_init_*() functions must be called before samples
        can be processed.
    */

} FLAC__StreamDecoderState;

/** Maps a FLAC__StreamDecoderState to a C string.

    Using a FLAC__StreamDecoderState as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderStateString[];


/** Possible return values for the FLAC__stream_decoder_init_*() functions.
*/
typedef enum {

    FLAC__STREAM_DECODER_INIT_STATUS_OK = 0,
    /**< Initialization was successful. */

    FLAC__STREAM_DECODER_INIT_STATUS_UNSUPPORTED_CONTAINER,
    /** < The library was not compiled with support for the given container
        format.
    */

    FLAC__STREAM_DECODER_INIT_STATUS_INVALID_CALLBACKS,
    /**< A required callback was not supplied. */

    FLAC__STREAM_DECODER_INIT_STATUS_MEMORY_ALLOCATION_ERROR,
    /**< An error occurred allocating memory. */

    FLAC__STREAM_DECODER_INIT_STATUS_ERROR_OPENING_FILE,
    /** < fopen() failed in FLAC__stream_decoder_init_file() or
        FLAC__stream_decoder_init_ogg_file(). */

    FLAC__STREAM_DECODER_INIT_STATUS_ALREADY_INITIALIZED
    /** < FLAC__stream_decoder_init_*() was called when the decoder was
        already initialized, usually because
        FLAC__stream_decoder_finish() was not called.
    */

} FLAC__StreamDecoderInitStatus;

/** Maps a FLAC__StreamDecoderInitStatus to a C string.

    Using a FLAC__StreamDecoderInitStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderInitStatusString[];


/** Return values for the FLAC__StreamDecoder read callback.
*/
typedef enum {

    FLAC__STREAM_DECODER_READ_STATUS_CONTINUE,
    /**< The read was OK and decoding can continue. */

    FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM,
    /** < The read was attempted while at the end of the stream.  Note that
        the client must only return this value when the read callback was
        called when already at the end of the stream.  Otherwise, if the read
        itself moves to the end of the stream, the client should still return
        the data and \c FLAC__STREAM_DECODER_READ_STATUS_CONTINUE, and then on
        the next read callback it should return
        \c FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM with a byte count
        of \c 0.
    */

    FLAC__STREAM_DECODER_READ_STATUS_ABORT
    /**< An unrecoverable error occurred.  The decoder will return from the process call. */

} FLAC__StreamDecoderReadStatus;

/** Maps a FLAC__StreamDecoderReadStatus to a C string.

    Using a FLAC__StreamDecoderReadStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderReadStatusString[];


/** Return values for the FLAC__StreamDecoder seek callback.
*/
typedef enum {

    FLAC__STREAM_DECODER_SEEK_STATUS_OK,
    /**< The seek was OK and decoding can continue. */

    FLAC__STREAM_DECODER_SEEK_STATUS_ERROR,
    /**< An unrecoverable error occurred.  The decoder will return from the process call. */

    FLAC__STREAM_DECODER_SEEK_STATUS_UNSUPPORTED
    /**< Client does not support seeking. */

} FLAC__StreamDecoderSeekStatus;

/** Maps a FLAC__StreamDecoderSeekStatus to a C string.

    Using a FLAC__StreamDecoderSeekStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderSeekStatusString[];


/** Return values for the FLAC__StreamDecoder tell callback.
*/
typedef enum {

    FLAC__STREAM_DECODER_TELL_STATUS_OK,
    /**< The tell was OK and decoding can continue. */

    FLAC__STREAM_DECODER_TELL_STATUS_ERROR,
    /**< An unrecoverable error occurred.  The decoder will return from the process call. */

    FLAC__STREAM_DECODER_TELL_STATUS_UNSUPPORTED
    /**< Client does not support telling the position. */

} FLAC__StreamDecoderTellStatus;

/** Maps a FLAC__StreamDecoderTellStatus to a C string.

    Using a FLAC__StreamDecoderTellStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderTellStatusString[];


/** Return values for the FLAC__StreamDecoder length callback.
*/
typedef enum {

    FLAC__STREAM_DECODER_LENGTH_STATUS_OK,
    /**< The length call was OK and decoding can continue. */

    FLAC__STREAM_DECODER_LENGTH_STATUS_ERROR,
    /**< An unrecoverable error occurred.  The decoder will return from the process call. */

    FLAC__STREAM_DECODER_LENGTH_STATUS_UNSUPPORTED
    /**< Client does not support reporting the length. */

} FLAC__StreamDecoderLengthStatus;

/** Maps a FLAC__StreamDecoderLengthStatus to a C string.

    Using a FLAC__StreamDecoderLengthStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderLengthStatusString[];


/** Return values for the FLAC__StreamDecoder write callback.
*/
typedef enum {

    FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE,
    /**< The write was OK and decoding can continue. */

    FLAC__STREAM_DECODER_WRITE_STATUS_ABORT
    /**< An unrecoverable error occurred.  The decoder will return from the process call. */

} FLAC__StreamDecoderWriteStatus;

/** Maps a FLAC__StreamDecoderWriteStatus to a C string.

    Using a FLAC__StreamDecoderWriteStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderWriteStatusString[];


/** Possible values passed back to the FLAC__StreamDecoder error callback.
    \c FLAC__STREAM_DECODER_ERROR_STATUS_LOST_SYNC is the generic catch-
    all.  The rest could be caused by bad sync (false synchronization on
    data that is not the start of a frame) or corrupted data.  The error
    itself is the decoder's best guess at what happened assuming a correct
    sync.  For example \c FLAC__STREAM_DECODER_ERROR_STATUS_BAD_HEADER
    could be caused by a correct sync on the start of a frame, but some
    data in the frame header was corrupted.  Or it could be the result of
    syncing on a point the stream that looked like the starting of a frame
    but was not.  \c FLAC__STREAM_DECODER_ERROR_STATUS_UNPARSEABLE_STREAM
    could be because the decoder encountered a valid frame made by a future
    version of the encoder which it cannot parse, or because of a false
    sync making it appear as though an encountered frame was generated by
    a future encoder.
*/
typedef enum {

    FLAC__STREAM_DECODER_ERROR_STATUS_LOST_SYNC,
    /**< An error in the stream caused the decoder to lose synchronization. */

    FLAC__STREAM_DECODER_ERROR_STATUS_BAD_HEADER,
    /**< The decoder encountered a corrupted frame header. */

    FLAC__STREAM_DECODER_ERROR_STATUS_FRAME_CRC_MISMATCH,
    /**< The frame's data did not match the CRC in the footer. */

    FLAC__STREAM_DECODER_ERROR_STATUS_UNPARSEABLE_STREAM
    /**< The decoder encountered reserved fields in use in the stream. */

} FLAC__StreamDecoderErrorStatus;

/** Maps a FLAC__StreamDecoderErrorStatus to a C string.

    Using a FLAC__StreamDecoderErrorStatus as the index to this array
    will give the string equivalent.  The contents should not be modified.
*/
extern FLAC_API const char * const FLAC__StreamDecoderErrorStatusString[];


/***********************************************************************

    class FLAC__StreamDecoder

 ***********************************************************************/

struct FLAC__StreamDecoderProtected;
struct FLAC__StreamDecoderPrivate;
/** The opaque structure definition for the stream decoder type.
    See the \link flac_stream_decoder stream decoder module \endlink
    for a detailed description.
*/
typedef struct {
    struct FLAC__StreamDecoderProtected *protected_; /* avoid the C++ keyword 'protected' */
    struct FLAC__StreamDecoderPrivate *private_; /* avoid the C++ keyword 'private' */
} FLAC__StreamDecoder;

/** Signature for the read callback.

    A function pointer matching this signature must be passed to
    FLAC__stream_decoder_init*_stream(). The supplied function will be
    called when the decoder needs more input data.  The address of the
    buffer to be filled is supplied, along with the number of bytes the
    buffer can hold.  The callback may choose to supply less data and
    modify the byte count but must be careful not to overflow the buffer.
    The callback then returns a status code chosen from
    FLAC__StreamDecoderReadStatus.

    Here is an example of a read callback for stdio streams:
    \code
    FLAC__StreamDecoderReadStatus read_cb(const FLAC__StreamDecoder *decoder, FLAC__byte buffer[], size_t *bytes, void *client_data)
    {
     FILE *file = ((MyClientData*)client_data)->file;
     if(*bytes > 0) {
 *     *bytes = fread(buffer, sizeof(FLAC__byte), *bytes, file);
       if(ferror(file))
         return FLAC__STREAM_DECODER_READ_STATUS_ABORT;
       else if(*bytes == 0)
         return FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM;
       else
         return FLAC__STREAM_DECODER_READ_STATUS_CONTINUE;
     }
     else
       return FLAC__STREAM_DECODER_READ_STATUS_ABORT;
    }
    \endcode

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  buffer   A pointer to a location for the callee to store
                    data to be decoded.
    \param  bytes    A pointer to the size of the buffer.  On entry
                    to the callback, it contains the maximum number
                    of bytes that may be stored in \a buffer.  The
                    callee must set it to the actual number of bytes
                    stored (0 in case of error or end-of-stream) before
                    returning.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
    \retval FLAC__StreamDecoderReadStatus
      The callee's return status.  Note that the callback should return
      \c FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM if and only if
      zero bytes were read and there is no more data to be read.
*/
typedef FLAC__StreamDecoderReadStatus(*FLAC__StreamDecoderReadCallback)(const FLAC__StreamDecoder *decoder, FLAC__byte buffer[], size_t *bytes, void *client_data);

/** Signature for the seek callback.

    A function pointer matching this signature may be passed to
    FLAC__stream_decoder_init*_stream().  The supplied function will be
    called when the decoder needs to seek the input stream.  The decoder
    will pass the absolute byte offset to seek to, 0 meaning the
    beginning of the stream.

    Here is an example of a seek callback for stdio streams:
    \code
    FLAC__StreamDecoderSeekStatus seek_cb(const FLAC__StreamDecoder *decoder, FLAC__uint64 absolute_byte_offset, void *client_data)
    {
     FILE *file = ((MyClientData*)client_data)->file;
     if(file == stdin)
       return FLAC__STREAM_DECODER_SEEK_STATUS_UNSUPPORTED;
     else if(fseeko(file, (off_t)absolute_byte_offset, SEEK_SET) < 0)
       return FLAC__STREAM_DECODER_SEEK_STATUS_ERROR;
     else
       return FLAC__STREAM_DECODER_SEEK_STATUS_OK;
    }
    \endcode

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  absolute_byte_offset  The offset from the beginning of the stream
                                 to seek to.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
    \retval FLAC__StreamDecoderSeekStatus
      The callee's return status.
*/
typedef FLAC__StreamDecoderSeekStatus(*FLAC__StreamDecoderSeekCallback)(const FLAC__StreamDecoder *decoder, FLAC__uint64 absolute_byte_offset, void *client_data);

/** Signature for the tell callback.

    A function pointer matching this signature may be passed to
    FLAC__stream_decoder_init*_stream().  The supplied function will be
    called when the decoder wants to know the current position of the
    stream.  The callback should return the byte offset from the
    beginning of the stream.

    Here is an example of a tell callback for stdio streams:
    \code
    FLAC__StreamDecoderTellStatus tell_cb(const FLAC__StreamDecoder *decoder, FLAC__uint64 *absolute_byte_offset, void *client_data)
    {
     FILE *file = ((MyClientData*)client_data)->file;
     off_t pos;
     if(file == stdin)
       return FLAC__STREAM_DECODER_TELL_STATUS_UNSUPPORTED;
     else if((pos = ftello(file)) < 0)
       return FLAC__STREAM_DECODER_TELL_STATUS_ERROR;
     else {
 *     *absolute_byte_offset = (FLAC__uint64)pos;
       return FLAC__STREAM_DECODER_TELL_STATUS_OK;
     }
    }
    \endcode

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  absolute_byte_offset  A pointer to storage for the current offset
                                 from the beginning of the stream.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
    \retval FLAC__StreamDecoderTellStatus
      The callee's return status.
*/
typedef FLAC__StreamDecoderTellStatus(*FLAC__StreamDecoderTellCallback)(const FLAC__StreamDecoder *decoder, FLAC__uint64 *absolute_byte_offset, void *client_data);

/** Signature for the length callback.

    A function pointer matching this signature may be passed to
    FLAC__stream_decoder_init*_stream().  The supplied function will be
    called when the decoder wants to know the total length of the stream
    in bytes.

    Here is an example of a length callback for stdio streams:
    \code
    FLAC__StreamDecoderLengthStatus length_cb(const FLAC__StreamDecoder *decoder, FLAC__uint64 *stream_length, void *client_data)
    {
     FILE *file = ((MyClientData*)client_data)->file;
     struct stat filestats;

     if(file == stdin)
       return FLAC__STREAM_DECODER_LENGTH_STATUS_UNSUPPORTED;
     else if(fstat(fileno(file), &filestats) != 0)
       return FLAC__STREAM_DECODER_LENGTH_STATUS_ERROR;
     else {
 *     *stream_length = (FLAC__uint64)filestats.st_size;
       return FLAC__STREAM_DECODER_LENGTH_STATUS_OK;
     }
    }
    \endcode

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  stream_length  A pointer to storage for the length of the stream
                          in bytes.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
    \retval FLAC__StreamDecoderLengthStatus
      The callee's return status.
*/
typedef FLAC__StreamDecoderLengthStatus(*FLAC__StreamDecoderLengthCallback)(const FLAC__StreamDecoder *decoder, FLAC__uint64 *stream_length, void *client_data);

/** Signature for the EOF callback.

    A function pointer matching this signature may be passed to
    FLAC__stream_decoder_init*_stream().  The supplied function will be
    called when the decoder needs to know if the end of the stream has
    been reached.

    Here is an example of a EOF callback for stdio streams:
    FLAC__bool eof_cb(const FLAC__StreamDecoder *decoder, void *client_data)
    \code
    {
     FILE *file = ((MyClientData*)client_data)->file;
     return feof(file)? true : false;
    }
    \endcode

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
    \retval FLAC__bool
      \c true if the currently at the end of the stream, else \c false.
*/
typedef FLAC__bool(*FLAC__StreamDecoderEofCallback)(const FLAC__StreamDecoder *decoder, void *client_data);

/** Signature for the write callback.

    A function pointer matching this signature must be passed to one of
    the FLAC__stream_decoder_init_*() functions.
    The supplied function will be called when the decoder has decoded a
    single audio frame.  The decoder will pass the frame metadata as well
    as an array of pointers (one for each channel) pointing to the
    decoded audio.

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  frame    The description of the decoded frame.  See
                    FLAC__Frame.
    \param  buffer   An array of pointers to decoded channels of data.
                    Each pointer will point to an array of signed
                    samples of length \a frame->header.blocksize.
                    Channels will be ordered according to the FLAC
                    specification; see the documentation for the
                    <A HREF="../format.html#frame_header">frame header</A>.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
    \retval FLAC__StreamDecoderWriteStatus
      The callee's return status.
*/
typedef FLAC__StreamDecoderWriteStatus(*FLAC__StreamDecoderWriteCallback)(const FLAC__StreamDecoder *decoder, const FLAC__Frame *frame, const FLAC__int32 * const buffer[], void *client_data);

/** Signature for the metadata callback.

    A function pointer matching this signature must be passed to one of
    the FLAC__stream_decoder_init_*() functions.
    The supplied function will be called when the decoder has decoded a
    metadata block.  In a valid FLAC file there will always be one
    \c STREAMINFO block, followed by zero or more other metadata blocks.
    These will be supplied by the decoder in the same order as they
    appear in the stream and always before the first audio frame (i.e.
    write callback).  The metadata block that is passed in must not be
    modified, and it doesn't live beyond the callback, so you should make
    a copy of it with FLAC__metadata_object_clone() if you will need it
    elsewhere.  Since metadata blocks can potentially be large, by
    default the decoder only calls the metadata callback for the
    \c STREAMINFO block; you can instruct the decoder to pass or filter
    other blocks with FLAC__stream_decoder_set_metadata_*() calls.

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  metadata The decoded metadata block.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
*/
typedef void (*FLAC__StreamDecoderMetadataCallback)(const FLAC__StreamDecoder *decoder, const FLAC__StreamMetadata *metadata, void *client_data);

/** Signature for the error callback.

    A function pointer matching this signature must be passed to one of
    the FLAC__stream_decoder_init_*() functions.
    The supplied function will be called whenever an error occurs during
    decoding.

    \note In general, FLAC__StreamDecoder functions which change the
    state should not be called on the \a decoder while in the callback.

    \param  decoder  The decoder instance calling the callback.
    \param  status   The error encountered by the decoder.
    \param  client_data  The callee's client data set through
                        FLAC__stream_decoder_init_*().
*/
typedef void (*FLAC__StreamDecoderErrorCallback)(const FLAC__StreamDecoder *decoder, FLAC__StreamDecoderErrorStatus status, void *client_data);


/***********************************************************************

    Class constructor/destructor

 ***********************************************************************/

/** Create a new stream decoder instance.  The instance is created with
    default settings; see the individual FLAC__stream_decoder_set_*()
    functions for each setting's default.

    \retval FLAC__StreamDecoder
      \c NULL if there was an error allocating memory, else the new instance.
*/
FLAC_API FLAC__StreamDecoder *FLAC__stream_decoder_new(void);

/** Free a decoder instance.  Deletes the object pointed to by \a decoder.

    \param decoder  A pointer to an existing decoder.
    \assert
      \code decoder != NULL \endcode
*/
FLAC_API void FLAC__stream_decoder_delete(FLAC__StreamDecoder *decoder);


/***********************************************************************

    Public class method prototypes

 ***********************************************************************/

/** Set the serial number for the FLAC stream within the Ogg container.
    The default behavior is to use the serial number of the first Ogg
    page.  Setting a serial number here will explicitly specify which
    stream is to be decoded.

    \note
    This does not need to be set for native FLAC decoding.

    \default \c use serial number of first page
    \param  decoder        A decoder instance to set.
    \param  serial_number  See above.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_ogg_serial_number(FLAC__StreamDecoder *decoder, long serial_number);

/** Set the "MD5 signature checking" flag.  If \c true, the decoder will
    compute the MD5 signature of the unencoded audio data while decoding
    and compare it to the signature from the STREAMINFO block, if it
    exists, during FLAC__stream_decoder_finish().

    MD5 signature checking will be turned off (until the next
    FLAC__stream_decoder_reset()) if there is no signature in the
    STREAMINFO block or when a seek is attempted.

    Clients that do not use the MD5 check should leave this off to speed
    up decoding.

    \default \c false
    \param  decoder  A decoder instance to set.
    \param  value    Flag value (see above).
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_md5_checking(FLAC__StreamDecoder *decoder, FLAC__bool value);

/** Direct the decoder to pass on all metadata blocks of type \a type.

    \default By default, only the \c STREAMINFO block is returned via the
            metadata callback.
    \param  decoder  A decoder instance to set.
    \param  type     See above.
    \assert
      \code decoder != NULL \endcode
      \a type is valid
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_metadata_respond(FLAC__StreamDecoder *decoder, FLAC__MetadataType type);

/** Direct the decoder to pass on all APPLICATION metadata blocks of the
    given \a id.

    \default By default, only the \c STREAMINFO block is returned via the
            metadata callback.
    \param  decoder  A decoder instance to set.
    \param  id       See above.
    \assert
      \code decoder != NULL \endcode
      \code id != NULL \endcode
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_metadata_respond_application(FLAC__StreamDecoder *decoder, const FLAC__byte id[4]);

/** Direct the decoder to pass on all metadata blocks of any type.

    \default By default, only the \c STREAMINFO block is returned via the
            metadata callback.
    \param  decoder  A decoder instance to set.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_metadata_respond_all(FLAC__StreamDecoder *decoder);

/** Direct the decoder to filter out all metadata blocks of type \a type.

    \default By default, only the \c STREAMINFO block is returned via the
            metadata callback.
    \param  decoder  A decoder instance to set.
    \param  type     See above.
    \assert
      \code decoder != NULL \endcode
      \a type is valid
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_metadata_ignore(FLAC__StreamDecoder *decoder, FLAC__MetadataType type);

/** Direct the decoder to filter out all APPLICATION metadata blocks of
    the given \a id.

    \default By default, only the \c STREAMINFO block is returned via the
            metadata callback.
    \param  decoder  A decoder instance to set.
    \param  id       See above.
    \assert
      \code decoder != NULL \endcode
      \code id != NULL \endcode
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_metadata_ignore_application(FLAC__StreamDecoder *decoder, const FLAC__byte id[4]);

/** Direct the decoder to filter out all metadata blocks of any type.

    \default By default, only the \c STREAMINFO block is returned via the
            metadata callback.
    \param  decoder  A decoder instance to set.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if the decoder is already initialized, else \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_set_metadata_ignore_all(FLAC__StreamDecoder *decoder);

/** Get the current decoder state.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__StreamDecoderState
      The current decoder state.
*/
FLAC_API FLAC__StreamDecoderState FLAC__stream_decoder_get_state(const FLAC__StreamDecoder *decoder);

/** Get the current decoder state as a C string.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval const char
      The decoder state as a C string.  Do not modify the contents.
*/
FLAC_API const char *FLAC__stream_decoder_get_resolved_state_string(const FLAC__StreamDecoder *decoder);

/** Get the "MD5 signature checking" flag.
    This is the value of the setting, not whether or not the decoder is
    currently checking the MD5 (remember, it can be turned off automatically
    by a seek).  When the decoder is reset the flag will be restored to the
    value returned by this function.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      See above.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_get_md5_checking(const FLAC__StreamDecoder *decoder);

/** Get the total number of samples in the stream being decoded.
    Will only be valid after decoding has started and will contain the
    value from the \c STREAMINFO block.  A value of \c 0 means "unknown".

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval uint32_t
      See above.
*/
FLAC_API FLAC__uint64 FLAC__stream_decoder_get_total_samples(const FLAC__StreamDecoder *decoder);

/** Get the current number of channels in the stream being decoded.
    Will only be valid after decoding has started and will contain the
    value from the most recently decoded frame header.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval uint32_t
      See above.
*/
FLAC_API uint32_t FLAC__stream_decoder_get_channels(const FLAC__StreamDecoder *decoder);

/** Get the current channel assignment in the stream being decoded.
    Will only be valid after decoding has started and will contain the
    value from the most recently decoded frame header.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__ChannelAssignment
      See above.
*/
FLAC_API FLAC__ChannelAssignment FLAC__stream_decoder_get_channel_assignment(const FLAC__StreamDecoder *decoder);

/** Get the current sample resolution in the stream being decoded.
    Will only be valid after decoding has started and will contain the
    value from the most recently decoded frame header.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval uint32_t
      See above.
*/
FLAC_API uint32_t FLAC__stream_decoder_get_bits_per_sample(const FLAC__StreamDecoder *decoder);

/** Get the current sample rate in Hz of the stream being decoded.
    Will only be valid after decoding has started and will contain the
    value from the most recently decoded frame header.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval uint32_t
      See above.
*/
FLAC_API uint32_t FLAC__stream_decoder_get_sample_rate(const FLAC__StreamDecoder *decoder);

/** Get the current blocksize of the stream being decoded.
    Will only be valid after decoding has started and will contain the
    value from the most recently decoded frame header.

    \param  decoder  A decoder instance to query.
    \assert
      \code decoder != NULL \endcode
    \retval uint32_t
      See above.
*/
FLAC_API uint32_t FLAC__stream_decoder_get_blocksize(const FLAC__StreamDecoder *decoder);

/** Returns the decoder's current read position within the stream.
    The position is the byte offset from the start of the stream.
    Bytes before this position have been fully decoded.  Note that
    there may still be undecoded bytes in the decoder's read FIFO.
    The returned position is correct even after a seek.

    \warning This function currently only works for native FLAC,
             not Ogg FLAC streams.

    \param  decoder   A decoder instance to query.
    \param  position  Address at which to return the desired position.
    \assert
      \code decoder != NULL \endcode
      \code position != NULL \endcode
    \retval FLAC__bool
      \c true if successful, \c false if the stream is not native FLAC,
      or there was an error from the 'tell' callback or it returned
      \c FLAC__STREAM_DECODER_TELL_STATUS_UNSUPPORTED.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_get_decode_position(const FLAC__StreamDecoder *decoder, FLAC__uint64 *position);

/** Initialize the decoder instance to decode native FLAC streams.

    This flavor of initialization sets up the decoder to decode from a
    native FLAC stream. I/O is performed via callbacks to the client.
    For decoding from a plain file via filename or open FILE*,
    FLAC__stream_decoder_init_file() and FLAC__stream_decoder_init_FILE()
    provide a simpler interface.

    This function should be called after FLAC__stream_decoder_new() and
    FLAC__stream_decoder_set_*() but before any of the
    FLAC__stream_decoder_process_*() functions.  Will set and return the
    decoder state, which will be FLAC__STREAM_DECODER_SEARCH_FOR_METADATA
    if initialization succeeded.

    \param  decoder            An uninitialized decoder instance.
    \param  read_callback      See FLAC__StreamDecoderReadCallback.  This
                              pointer must not be \c NULL.
    \param  seek_callback      See FLAC__StreamDecoderSeekCallback.  This
                              pointer may be \c NULL if seeking is not
                              supported.  If \a seek_callback is not \c NULL then a
                              \a tell_callback, \a length_callback, and \a eof_callback must also be supplied.
                              Alternatively, a dummy seek callback that just
                              returns \c FLAC__STREAM_DECODER_SEEK_STATUS_UNSUPPORTED
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  tell_callback      See FLAC__StreamDecoderTellCallback.  This
                              pointer may be \c NULL if not supported by the client.  If
                              \a seek_callback is not \c NULL then a
                              \a tell_callback must also be supplied.
                              Alternatively, a dummy tell callback that just
                              returns \c FLAC__STREAM_DECODER_TELL_STATUS_UNSUPPORTED
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  length_callback    See FLAC__StreamDecoderLengthCallback.  This
                              pointer may be \c NULL if not supported by the client.  If
                              \a seek_callback is not \c NULL then a
                              \a length_callback must also be supplied.
                              Alternatively, a dummy length callback that just
                              returns \c FLAC__STREAM_DECODER_LENGTH_STATUS_UNSUPPORTED
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  eof_callback       See FLAC__StreamDecoderEofCallback.  This
                              pointer may be \c NULL if not supported by the client.  If
                              \a seek_callback is not \c NULL then a
                              \a eof_callback must also be supplied.
                              Alternatively, a dummy length callback that just
                              returns \c false
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  write_callback     See FLAC__StreamDecoderWriteCallback.  This
                              pointer must not be \c NULL.
    \param  metadata_callback  See FLAC__StreamDecoderMetadataCallback.  This
                              pointer may be \c NULL if the callback is not
                              desired.
    \param  error_callback     See FLAC__StreamDecoderErrorCallback.  This
                              pointer must not be \c NULL.
    \param  client_data        This value will be supplied to callbacks in their
                              \a client_data argument.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__StreamDecoderInitStatus
      \c FLAC__STREAM_DECODER_INIT_STATUS_OK if initialization was successful;
      see FLAC__StreamDecoderInitStatus for the meanings of other return values.
*/
FLAC_API FLAC__StreamDecoderInitStatus FLAC__stream_decoder_init_stream(
    FLAC__StreamDecoder *decoder,
    FLAC__StreamDecoderReadCallback read_callback,
    FLAC__StreamDecoderSeekCallback seek_callback,
    FLAC__StreamDecoderTellCallback tell_callback,
    FLAC__StreamDecoderLengthCallback length_callback,
    FLAC__StreamDecoderEofCallback eof_callback,
    FLAC__StreamDecoderWriteCallback write_callback,
    FLAC__StreamDecoderMetadataCallback metadata_callback,
    FLAC__StreamDecoderErrorCallback error_callback,
    void *client_data
);

/** Initialize the decoder instance to decode Ogg FLAC streams.

    This flavor of initialization sets up the decoder to decode from a
    FLAC stream in an Ogg container. I/O is performed via callbacks to the
    client.  For decoding from a plain file via filename or open FILE*,
    FLAC__stream_decoder_init_ogg_file() and FLAC__stream_decoder_init_ogg_FILE()
    provide a simpler interface.

    This function should be called after FLAC__stream_decoder_new() and
    FLAC__stream_decoder_set_*() but before any of the
    FLAC__stream_decoder_process_*() functions.  Will set and return the
    decoder state, which will be FLAC__STREAM_DECODER_SEARCH_FOR_METADATA
    if initialization succeeded.

    \note Support for Ogg FLAC in the library is optional.  If this
    library has been built without support for Ogg FLAC, this function
    will return \c FLAC__STREAM_DECODER_INIT_STATUS_UNSUPPORTED_CONTAINER.

    \param  decoder            An uninitialized decoder instance.
    \param  read_callback      See FLAC__StreamDecoderReadCallback.  This
                              pointer must not be \c NULL.
    \param  seek_callback      See FLAC__StreamDecoderSeekCallback.  This
                              pointer may be \c NULL if seeking is not
                              supported.  If \a seek_callback is not \c NULL then a
                              \a tell_callback, \a length_callback, and \a eof_callback must also be supplied.
                              Alternatively, a dummy seek callback that just
                              returns \c FLAC__STREAM_DECODER_SEEK_STATUS_UNSUPPORTED
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  tell_callback      See FLAC__StreamDecoderTellCallback.  This
                              pointer may be \c NULL if not supported by the client.  If
                              \a seek_callback is not \c NULL then a
                              \a tell_callback must also be supplied.
                              Alternatively, a dummy tell callback that just
                              returns \c FLAC__STREAM_DECODER_TELL_STATUS_UNSUPPORTED
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  length_callback    See FLAC__StreamDecoderLengthCallback.  This
                              pointer may be \c NULL if not supported by the client.  If
                              \a seek_callback is not \c NULL then a
                              \a length_callback must also be supplied.
                              Alternatively, a dummy length callback that just
                              returns \c FLAC__STREAM_DECODER_LENGTH_STATUS_UNSUPPORTED
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  eof_callback       See FLAC__StreamDecoderEofCallback.  This
                              pointer may be \c NULL if not supported by the client.  If
                              \a seek_callback is not \c NULL then a
                              \a eof_callback must also be supplied.
                              Alternatively, a dummy length callback that just
                              returns \c false
                              may also be supplied, all though this is slightly
                              less efficient for the decoder.
    \param  write_callback     See FLAC__StreamDecoderWriteCallback.  This
                              pointer must not be \c NULL.
    \param  metadata_callback  See FLAC__StreamDecoderMetadataCallback.  This
                              pointer may be \c NULL if the callback is not
                              desired.
    \param  error_callback     See FLAC__StreamDecoderErrorCallback.  This
                              pointer must not be \c NULL.
    \param  client_data        This value will be supplied to callbacks in their
                              \a client_data argument.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__StreamDecoderInitStatus
      \c FLAC__STREAM_DECODER_INIT_STATUS_OK if initialization was successful;
      see FLAC__StreamDecoderInitStatus for the meanings of other return values.
*/
FLAC_API FLAC__StreamDecoderInitStatus FLAC__stream_decoder_init_ogg_stream(
    FLAC__StreamDecoder *decoder,
    FLAC__StreamDecoderReadCallback read_callback,
    FLAC__StreamDecoderSeekCallback seek_callback,
    FLAC__StreamDecoderTellCallback tell_callback,
    FLAC__StreamDecoderLengthCallback length_callback,
    FLAC__StreamDecoderEofCallback eof_callback,
    FLAC__StreamDecoderWriteCallback write_callback,
    FLAC__StreamDecoderMetadataCallback metadata_callback,
    FLAC__StreamDecoderErrorCallback error_callback,
    void *client_data
);

/** Initialize the decoder instance to decode native FLAC files.

    This flavor of initialization sets up the decoder to decode from a
    plain native FLAC file.  For non-stdio streams, you must use
    FLAC__stream_decoder_init_stream() and provide callbacks for the I/O.

    This function should be called after FLAC__stream_decoder_new() and
    FLAC__stream_decoder_set_*() but before any of the
    FLAC__stream_decoder_process_*() functions.  Will set and return the
    decoder state, which will be FLAC__STREAM_DECODER_SEARCH_FOR_METADATA
    if initialization succeeded.

    \param  decoder            An uninitialized decoder instance.
    \param  file               An open FLAC file.  The file should have been
                              opened with mode \c "rb" and rewound.  The file
                              becomes owned by the decoder and should not be
                              manipulated by the client while decoding.
                              Unless \a file is \c stdin, it will be closed
                              when FLAC__stream_decoder_finish() is called.
                              Note however that seeking will not work when
                              decoding from \c stdin since it is not seekable.
    \param  write_callback     See FLAC__StreamDecoderWriteCallback.  This
                              pointer must not be \c NULL.
    \param  metadata_callback  See FLAC__StreamDecoderMetadataCallback.  This
                              pointer may be \c NULL if the callback is not
                              desired.
    \param  error_callback     See FLAC__StreamDecoderErrorCallback.  This
                              pointer must not be \c NULL.
    \param  client_data        This value will be supplied to callbacks in their
                              \a client_data argument.
    \assert
      \code decoder != NULL \endcode
      \code file != NULL \endcode
    \retval FLAC__StreamDecoderInitStatus
      \c FLAC__STREAM_DECODER_INIT_STATUS_OK if initialization was successful;
      see FLAC__StreamDecoderInitStatus for the meanings of other return values.
*/
#if 0
FLAC_API FLAC__StreamDecoderInitStatus FLAC__stream_decoder_init_FILE(
    FLAC__StreamDecoder *decoder,
    FILE *file,
    FLAC__StreamDecoderWriteCallback write_callback,
    FLAC__StreamDecoderMetadataCallback metadata_callback,
    FLAC__StreamDecoderErrorCallback error_callback,
    void *client_data
);
#endif
/** Initialize the decoder instance to decode Ogg FLAC files.

    This flavor of initialization sets up the decoder to decode from a
    plain Ogg FLAC file.  For non-stdio streams, you must use
    FLAC__stream_decoder_init_ogg_stream() and provide callbacks for the I/O.

    This function should be called after FLAC__stream_decoder_new() and
    FLAC__stream_decoder_set_*() but before any of the
    FLAC__stream_decoder_process_*() functions.  Will set and return the
    decoder state, which will be FLAC__STREAM_DECODER_SEARCH_FOR_METADATA
    if initialization succeeded.

    \note Support for Ogg FLAC in the library is optional.  If this
    library has been built without support for Ogg FLAC, this function
    will return \c FLAC__STREAM_DECODER_INIT_STATUS_UNSUPPORTED_CONTAINER.

    \param  decoder            An uninitialized decoder instance.
    \param  file               An open FLAC file.  The file should have been
                              opened with mode \c "rb" and rewound.  The file
                              becomes owned by the decoder and should not be
                              manipulated by the client while decoding.
                              Unless \a file is \c stdin, it will be closed
                              when FLAC__stream_decoder_finish() is called.
                              Note however that seeking will not work when
                              decoding from \c stdin since it is not seekable.
    \param  write_callback     See FLAC__StreamDecoderWriteCallback.  This
                              pointer must not be \c NULL.
    \param  metadata_callback  See FLAC__StreamDecoderMetadataCallback.  This
                              pointer may be \c NULL if the callback is not
                              desired.
    \param  error_callback     See FLAC__StreamDecoderErrorCallback.  This
                              pointer must not be \c NULL.
    \param  client_data        This value will be supplied to callbacks in their
                              \a client_data argument.
    \assert
      \code decoder != NULL \endcode
      \code file != NULL \endcode
    \retval FLAC__StreamDecoderInitStatus
      \c FLAC__STREAM_DECODER_INIT_STATUS_OK if initialization was successful;
      see FLAC__StreamDecoderInitStatus for the meanings of other return values.
*/
#if 0
FLAC_API FLAC__StreamDecoderInitStatus FLAC__stream_decoder_init_ogg_FILE(
    FLAC__StreamDecoder *decoder,
    FILE *file,
    FLAC__StreamDecoderWriteCallback write_callback,
    FLAC__StreamDecoderMetadataCallback metadata_callback,
    FLAC__StreamDecoderErrorCallback error_callback,
    void *client_data
);
#endif
/** Initialize the decoder instance to decode native FLAC files.

    This flavor of initialization sets up the decoder to decode from a plain
    native FLAC file.  If POSIX fopen() semantics are not sufficient, (for
    example, with Unicode filenames on Windows), you must use
    FLAC__stream_decoder_init_FILE(), or FLAC__stream_decoder_init_stream()
    and provide callbacks for the I/O.

    This function should be called after FLAC__stream_decoder_new() and
    FLAC__stream_decoder_set_*() but before any of the
    FLAC__stream_decoder_process_*() functions.  Will set and return the
    decoder state, which will be FLAC__STREAM_DECODER_SEARCH_FOR_METADATA
    if initialization succeeded.

    \param  decoder            An uninitialized decoder instance.
    \param  filename           The name of the file to decode from.  The file will
                              be opened with fopen().  Use \c NULL to decode from
                              \c stdin.  Note that \c stdin is not seekable.
    \param  write_callback     See FLAC__StreamDecoderWriteCallback.  This
                              pointer must not be \c NULL.
    \param  metadata_callback  See FLAC__StreamDecoderMetadataCallback.  This
                              pointer may be \c NULL if the callback is not
                              desired.
    \param  error_callback     See FLAC__StreamDecoderErrorCallback.  This
                              pointer must not be \c NULL.
    \param  client_data        This value will be supplied to callbacks in their
                              \a client_data argument.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__StreamDecoderInitStatus
      \c FLAC__STREAM_DECODER_INIT_STATUS_OK if initialization was successful;
      see FLAC__StreamDecoderInitStatus for the meanings of other return values.
*/
#if 0
FLAC_API FLAC__StreamDecoderInitStatus FLAC__stream_decoder_init_file(
    FLAC__StreamDecoder *decoder,
    const char *filename,
    FLAC__StreamDecoderWriteCallback write_callback,
    FLAC__StreamDecoderMetadataCallback metadata_callback,
    FLAC__StreamDecoderErrorCallback error_callback,
    void *client_data
);
#endif
/** Initialize the decoder instance to decode Ogg FLAC files.

    This flavor of initialization sets up the decoder to decode from a plain
    Ogg FLAC file.  If POSIX fopen() semantics are not sufficient, (for
    example, with Unicode filenames on Windows), you must use
    FLAC__stream_decoder_init_ogg_FILE(), or FLAC__stream_decoder_init_ogg_stream()
    and provide callbacks for the I/O.

    This function should be called after FLAC__stream_decoder_new() and
    FLAC__stream_decoder_set_*() but before any of the
    FLAC__stream_decoder_process_*() functions.  Will set and return the
    decoder state, which will be FLAC__STREAM_DECODER_SEARCH_FOR_METADATA
    if initialization succeeded.

    \note Support for Ogg FLAC in the library is optional.  If this
    library has been built without support for Ogg FLAC, this function
    will return \c FLAC__STREAM_DECODER_INIT_STATUS_UNSUPPORTED_CONTAINER.

    \param  decoder            An uninitialized decoder instance.
    \param  filename           The name of the file to decode from.  The file will
                              be opened with fopen().  Use \c NULL to decode from
                              \c stdin.  Note that \c stdin is not seekable.
    \param  write_callback     See FLAC__StreamDecoderWriteCallback.  This
                              pointer must not be \c NULL.
    \param  metadata_callback  See FLAC__StreamDecoderMetadataCallback.  This
                              pointer may be \c NULL if the callback is not
                              desired.
    \param  error_callback     See FLAC__StreamDecoderErrorCallback.  This
                              pointer must not be \c NULL.
    \param  client_data        This value will be supplied to callbacks in their
                              \a client_data argument.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__StreamDecoderInitStatus
      \c FLAC__STREAM_DECODER_INIT_STATUS_OK if initialization was successful;
      see FLAC__StreamDecoderInitStatus for the meanings of other return values.
*/
#if 0
FLAC_API FLAC__StreamDecoderInitStatus FLAC__stream_decoder_init_ogg_file(
    FLAC__StreamDecoder *decoder,
    const char *filename,
    FLAC__StreamDecoderWriteCallback write_callback,
    FLAC__StreamDecoderMetadataCallback metadata_callback,
    FLAC__StreamDecoderErrorCallback error_callback,
    void *client_data
);
#endif
/** Finish the decoding process.
    Flushes the decoding buffer, releases resources, resets the decoder
    settings to their defaults, and returns the decoder state to
    FLAC__STREAM_DECODER_UNINITIALIZED.

    In the event of a prematurely-terminated decode, it is not strictly
    necessary to call this immediately before FLAC__stream_decoder_delete()
    but it is good practice to match every FLAC__stream_decoder_init_*()
    with a FLAC__stream_decoder_finish().

    \param  decoder  An uninitialized decoder instance.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if MD5 checking is on AND a STREAMINFO block was available
      AND the MD5 signature in the STREAMINFO block was non-zero AND the
      signature does not match the one computed by the decoder; else
      \c true.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_finish(FLAC__StreamDecoder *decoder);

/** Flush the stream input.
    The decoder's input buffer will be cleared and the state set to
    \c FLAC__STREAM_DECODER_SEARCH_FOR_FRAME_SYNC.  This will also turn
    off MD5 checking.

    \param  decoder  A decoder instance.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c true if successful, else \c false if a memory allocation
      error occurs (in which case the state will be set to
      \c FLAC__STREAM_DECODER_MEMORY_ALLOCATION_ERROR).
*/
FLAC_API FLAC__bool FLAC__stream_decoder_flush(FLAC__StreamDecoder *decoder);

/** Reset the decoding process.
    The decoder's input buffer will be cleared and the state set to
    \c FLAC__STREAM_DECODER_SEARCH_FOR_METADATA.  This is similar to
    FLAC__stream_decoder_finish() except that the settings are
    preserved; there is no need to call FLAC__stream_decoder_init_*()
    before decoding again.  MD5 checking will be restored to its original
    setting.

    If the decoder is seekable, or was initialized with
    FLAC__stream_decoder_init*_FILE() or FLAC__stream_decoder_init*_file(),
    the decoder will also attempt to seek to the beginning of the file.
    If this rewind fails, this function will return \c false.  It follows
    that FLAC__stream_decoder_reset() cannot be used when decoding from
    \c stdin.

    If the decoder was initialized with FLAC__stream_encoder_init*_stream()
    and is not seekable (i.e. no seek callback was provided or the seek
    callback returns \c FLAC__STREAM_DECODER_SEEK_STATUS_UNSUPPORTED), it
    is the duty of the client to start feeding data from the beginning of
    the stream on the next FLAC__stream_decoder_process_*() call.

    \param  decoder  A decoder instance.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c true if successful, else \c false if a memory allocation occurs
      (in which case the state will be set to
      \c FLAC__STREAM_DECODER_MEMORY_ALLOCATION_ERROR) or a seek error
      occurs (the state will be unchanged).
*/
FLAC_API FLAC__bool FLAC__stream_decoder_reset(FLAC__StreamDecoder *decoder);

/** Decode one metadata block or audio frame.
    This version instructs the decoder to decode a either a single metadata
    block or a single frame and stop, unless the callbacks return a fatal
    error or the read callback returns
    \c FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM.

    As the decoder needs more input it will call the read callback.
    Depending on what was decoded, the metadata or write callback will be
    called with the decoded metadata block or audio frame.

    Unless there is a fatal read error or end of stream, this function
    will return once one whole frame is decoded.  In other words, if the
    stream is not synchronized or points to a corrupt frame header, the
    decoder will continue to try and resync until it gets to a valid
    frame, then decode one frame, then return.  If the decoder points to
    a frame whose frame CRC in the frame footer does not match the
    computed frame CRC, this function will issue a
    FLAC__STREAM_DECODER_ERROR_STATUS_FRAME_CRC_MISMATCH error to the
    error callback, and return, having decoded one complete, although
    corrupt, frame.  (Such corrupted frames are sent as silence of the
    correct length to the write callback.)

    \param  decoder  An initialized decoder instance.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if any fatal read, write, or memory allocation error
      occurred (meaning decoding must stop), else \c true; for more
      information about the decoder, check the decoder state with
      FLAC__stream_decoder_get_state().
*/
FLAC_API FLAC__bool FLAC__stream_decoder_process_single(FLAC__StreamDecoder *decoder);

/** Decode until the end of the metadata.
    This version instructs the decoder to decode from the current position
    and continue until all the metadata has been read, or until the
    callbacks return a fatal error or the read callback returns
    \c FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM.

    As the decoder needs more input it will call the read callback.
    As each metadata block is decoded, the metadata callback will be called
    with the decoded metadata.

    \param  decoder  An initialized decoder instance.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if any fatal read, write, or memory allocation error
      occurred (meaning decoding must stop), else \c true; for more
      information about the decoder, check the decoder state with
      FLAC__stream_decoder_get_state().
*/
FLAC_API FLAC__bool FLAC__stream_decoder_process_until_end_of_metadata(FLAC__StreamDecoder *decoder);

/** Decode until the end of the stream.
    This version instructs the decoder to decode from the current position
    and continue until the end of stream (the read callback returns
    \c FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM), or until the
    callbacks return a fatal error.

    As the decoder needs more input it will call the read callback.
    As each metadata block and frame is decoded, the metadata or write
    callback will be called with the decoded metadata or frame.

    \param  decoder  An initialized decoder instance.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if any fatal read, write, or memory allocation error
      occurred (meaning decoding must stop), else \c true; for more
      information about the decoder, check the decoder state with
      FLAC__stream_decoder_get_state().
*/
FLAC_API FLAC__bool FLAC__stream_decoder_process_until_end_of_stream(FLAC__StreamDecoder *decoder);

/** Skip one audio frame.
    This version instructs the decoder to 'skip' a single frame and stop,
    unless the callbacks return a fatal error or the read callback returns
    \c FLAC__STREAM_DECODER_READ_STATUS_END_OF_STREAM.

    The decoding flow is the same as what occurs when
    FLAC__stream_decoder_process_single() is called to process an audio
    frame, except that this function does not decode the parsed data into
    PCM or call the write callback.  The integrity of the frame is still
    checked the same way as in the other process functions.

    This function will return once one whole frame is skipped, in the
    same way that FLAC__stream_decoder_process_single() will return once
    one whole frame is decoded.

    This function can be used in more quickly determining FLAC frame
    boundaries when decoding of the actual data is not needed, for
    example when an application is separating a FLAC stream into frames
    for editing or storing in a container.  To do this, the application
    can use FLAC__stream_decoder_skip_single_frame() to quickly advance
    to the next frame, then use
    FLAC__stream_decoder_get_decode_position() to find the new frame
    boundary.

    This function should only be called when the stream has advanced
    past all the metadata, otherwise it will return \c false.

    \param  decoder  An initialized decoder instance not in a metadata
                    state.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c false if any fatal read, write, or memory allocation error
      occurred (meaning decoding must stop), or if the decoder
      is in the FLAC__STREAM_DECODER_SEARCH_FOR_METADATA or
      FLAC__STREAM_DECODER_READ_METADATA state, else \c true; for more
      information about the decoder, check the decoder state with
      FLAC__stream_decoder_get_state().
*/
FLAC_API FLAC__bool FLAC__stream_decoder_skip_single_frame(FLAC__StreamDecoder *decoder);

/** Flush the input and seek to an absolute sample.
    Decoding will resume at the given sample.  Note that because of
    this, the next write callback may contain a partial block.  The
    client must support seeking the input or this function will fail
    and return \c false.  Furthermore, if the decoder state is
    \c FLAC__STREAM_DECODER_SEEK_ERROR, then the decoder must be flushed
    with FLAC__stream_decoder_flush() or reset with
    FLAC__stream_decoder_reset() before decoding can continue.

    \param  decoder  A decoder instance.
    \param  sample   The target sample number to seek to.
    \assert
      \code decoder != NULL \endcode
    \retval FLAC__bool
      \c true if successful, else \c false.
*/
FLAC_API FLAC__bool FLAC__stream_decoder_seek_absolute(FLAC__StreamDecoder *decoder, FLAC__uint64 sample);

/** Return client_data from decoder.
     The data pointed to by the pointer should not be modified.

    \param  decoder  A decoder instance.
    \retval const void
       The callee's client data set through FLAC__stream_decoder_init_*().
       Do not modify the contents.
*/
FLAC_API const void *FLAC__get_decoder_client_data(FLAC__StreamDecoder *decoder);

/* \} */

#ifdef __cplusplus
}
#endif

#endif
