/*
    Inkplate7_Daily_Weather_Station example for Soldered Inkplate7
    For this example you will need only USB cable and Inkplate7.
    Select "Soldered Inkplate7" from Tools -> Board menu.
    Don't have "Soldered Inkplate7" option? Follow our tutorial and add it:
    https://soldered.com/learn/add-inkplate-6-board-definition-to-arduino-ide/

    This example will show you how you can use Inkplate 7 to display API data,
    e.g. OpenWeather public weather API for real time data. It shows the forecast
    weather for 4 days. What happens here is basically ESP32 connects to WiFi and
    sends an API call and the server returns data in JSON format containing data
    about weather, then using the library ArduinoJson we extract icons and temperatures
    from JSON data and show it on Inkplate 7. After displaying the weather,
    ESP32 goes to sleep and wakes up every DELAY_MS milliseconds to show new weather
    (you can change the time interval).

    IMPORTANT:
    Make sure to change your desired city, timezone and wifi credentials below
    Also have ArduinoJSON installed in your Arduino libraries
    Want to learn more about Inkplate? Visit www.inkplate.io
    Looking to get support? Write on our forums: https://forum.soldered.com/
    11 April 2023 by Soldered

    In order to convert your images into a format compatible with Inkplate
    use the Soldered Image Converter available at:
    https://github.com/SolderedElectronics/Soldered-Image-Converter/releases
*/

// Next 3 lines are a precaution, you can ignore those, and the example would also work without them
#ifndef ARDUINO_INKPLATE7
#error "Wrong board selection for this example, please select Soldered Inkplate7 in the boards menu."
#endif

// ---------- CHANGE HERE  -------------:

// City search query
char city[128] = "OSIJEK";

// Coordinates sent to the api
char lat[] = "45.5510548";
char lon[] = "18.695463";

// Change to your wifi ssid and password
char ssid[] = "";
char pass[] = "";

// Change to your api key, if you don't have one, head over to:
// https://openweathermap.org/guide , register and copy the key provided
char apiKey[] = "";

// ----------------------------------

// Include Inkplate library to the sketch
#include "Inkplate.h"

// Header file for easier code readability
#include "Network.h"

// Including fonts used
#include "Fonts/Inter12pt7b.h"
#include "Fonts/Inter30pt7b.h"
#include "Fonts/Inter8pt7b.h"

// Including icons generated by the py file
#include "icons.h"

// Delay between API calls
#define DELAY_MS (uint32_t)10 * 60 * 1000 // 10 minute time

// Inkplate object
Inkplate display;

// All our network functions are in this object, see Network.h
Network network;

// Constants used for drawing icons
char abbrs[9][16] = {"01d", "02d", "03d", "04d", "09d", "10d", "11d", "13d", "50d"};

const uint8_t *logos[9] = {icon_01d, icon_02d, icon_03d, icon_04d, icon_09d, icon_10d, icon_11d, icon_13d, icon_50d};

const uint8_t *s_logos[9] = {icon_s_01d, icon_s_02d, icon_s_03d, icon_s_04d, icon_s_09d,
                             icon_s_10d, icon_s_11d, icon_s_13d, icon_s_50d};

char abbr1[16];
char abbr2[16];
char abbr3[16];
char abbr4[16];

// Variables for storing temperature
char temps[8][4] = {
    "-",
    "-",
    "-",
    "-",
};

uint8_t hours = 0;

// Variables for storing current time and weather info
char currentTemp[16] = "-";
char currentWind[16] = "-";

char currentTime[16] = "--:--";

char currentWeather[32] = "-";
char currentWeatherAbbr[8] = "01d";

// Functions defined below
void drawWeather();
void drawCurrent();
void drawTemps();
void drawCity();
void drawTime();

void setup()
{
    // Begin serial and display
    Serial.begin(115200);
    display.begin();

    // Calling our begin from network.h file
    network.begin(ssid, pass);

    // Get all relevant data, see Network.cpp for info
    Serial.print("Fetching data");
    while (!network.getData(lat, lon, apiKey, city, temps[0], temps[1], temps[2], temps[3], currentTemp, currentWind,
                            currentTime, currentWeather, currentWeatherAbbr, abbr1, abbr2, abbr3, abbr4, &hours))
    {
        Serial.print('.');
        delay(500);
    }
    Serial.println();
    network.getTime(currentTime);

    // Draw data, see functions below for info
    Serial.println("Drawing on the screen");
    drawWeather();
    drawCurrent();
    drawTemps();
    drawCity();
    drawTime();

    // Next line actually draw all on the screen
    display.display();

    Serial.println("Going to sleep, bye!");

    // Activate wakeup timer
    esp_sleep_enable_timer_wakeup(1000L * DELAY_MS);

    // Start deep sleep (this function does not return). Program stops here.
    esp_deep_sleep_start();
}

void loop()
{
    // Never here! If you are using deep sleep, the whole program should be in setup() because the board restarts each
    // time. loop() must be empty!
}

// Function for drawing weather info
void drawWeather()
{
    // Searching for weather state abbreviation
    for (int i = 0; i < 9; ++i)
    {
        // If found draw specified icon
        if (strcmp(abbrs[i], currentWeatherAbbr) == 0)
        {
            display.drawBitmap(116, 8, logos[i], 96, 96, INKPLATE7_BLACK);
        }
    }

    // Draw weather state
    display.setTextColor(INKPLATE7_BLACK);
    display.setFont(&Inter12pt7b);
    display.setTextSize(1);
    display.setCursor(130, 118);
    display.println(currentWeather);
}

// Function for drawing current time
void drawTime()
{
    // Drawing current time
    display.setTextColor(INKPLATE7_BLACK);
    display.setFont(&Inter12pt7b);
    display.setTextSize(1);

    display.setCursor(E_INK_WIDTH - 14 * strlen(currentTime), 20);
    display.println(currentTime);
}

// Function for drawing city name
void drawCity()
{
    // Drawing city name
    display.setTextColor(INKPLATE7_BLACK);
    display.setFont(&Inter12pt7b);
    display.setTextSize(1);

    display.setCursor(E_INK_WIDTH / 2 - 7 * strlen(city), E_INK_HEIGHT - 18);
    display.println(city);
}

// Function for drawing temperatures
void drawTemps()
{
    // Drawing 4 rectangles in which temperatures will be written
    int rectWidth = 135;
    int rectSpacing = (E_INK_WIDTH - rectWidth * 4) / 5;
    int yRectangleOffset = 135;
    int rectHeight = 195;

    for (int i = 0; i < 4; i++)
    {
        display.drawRect((i + 1) * rectSpacing + i * rectWidth, yRectangleOffset, rectWidth, rectHeight, BLACK);
    }

    // Drawing days into rectangles
    int textMargin = 10;

    display.setFont(&Inter12pt7b);
    display.setTextSize(1);
    display.setTextColor(INKPLATE7_BLACK);

    int dayOffset = 15;

    for (int i = 0; i < 4; i++)
    {
        if (i == 0)
        {
            display.setCursor((i + 1) * rectSpacing + i * rectWidth + textMargin + 25,
                              yRectangleOffset + textMargin + dayOffset);
            display.println("Today");
        }
        else
        {
            display.setCursor((i + 1) * rectSpacing + i * rectWidth + textMargin + 35,
                              yRectangleOffset + textMargin + dayOffset);
            display.println(wDays[hours + i > 6 ? hours + i - 6 : hours + i]);
        }
    }

    int tempOffset = 97;

    // Drawing temperature values into rectangles
    display.setFont(&Inter30pt7b);
    display.setTextSize(1);
    display.setTextColor(INKPLATE7_RED);

    for (int i = 0; i < 4; i++)
    {
        display.setCursor((i + 1) * rectSpacing + i * rectWidth + textMargin,
                          yRectangleOffset + textMargin + tempOffset);
        display.print(temps[i]);
        display.println(F("C"));
    }

    // Drawing icons into rectangles
    int iconOffset = 130;

    for (int i = 0; i < 9; ++i)
    {
        // If found draw specified icon
        if (strcmp(abbr1, abbrs[i]) == 0)
            display.drawBitmap(1 * rectSpacing + 0 * rectWidth + textMargin + 35,
                               yRectangleOffset + textMargin + iconOffset, s_logos[i], 48, 48, INKPLATE7_BLACK,
                               INKPLATE7_WHITE);
    }

    for (int i = 0; i < 9; ++i)
    {
        // If found draw specified icon
        if (strcmp(abbr2, abbrs[i]) == 0)
            display.drawBitmap(2 * rectSpacing + 1 * rectWidth + textMargin + 35,
                               yRectangleOffset + textMargin + iconOffset, s_logos[i], 48, 48, INKPLATE7_BLACK,
                               INKPLATE7_WHITE);
    }

    for (int i = 0; i < 9; ++i)
    {
        // If found draw specified icon
        if (strcmp(abbr3, abbrs[i]) == 0)
            display.drawBitmap(3 * rectSpacing + 2 * rectWidth + textMargin + 35,
                               yRectangleOffset + textMargin + iconOffset, s_logos[i], 48, 48, INKPLATE7_BLACK,
                               INKPLATE7_WHITE);
    }

    for (int i = 0; i < 9; ++i)
    {
        // If found draw specified icon
        if (strcmp(abbr4, abbrs[i]) == 0)
            display.drawBitmap(4 * rectSpacing + 3 * rectWidth + textMargin + 35,
                               yRectangleOffset + textMargin + iconOffset, s_logos[i], 48, 48, INKPLATE7_BLACK,
                               INKPLATE7_WHITE);
    }
}

// Current weather drawing function
void drawCurrent()
{
    // Drawing current information

    // Temperature:
    display.setFont(&Inter30pt7b);
    display.setTextSize(1);

    display.drawTextWithShadow(275, 78, currentTemp, INKPLATE7_BLACK, INKPLATE7_RED);

    int x = display.getCursorX();
    int y = display.getCursorY();

    display.setFont(&Inter12pt7b);
    display.setTextSize(1);

    display.setCursor(x, y);
    display.println(F("C"));

    // Wind:
    display.setFont(&Inter30pt7b);
    display.setTextSize(1);

    display.drawTextWithShadow(440, 78, currentWind, INKPLATE7_BLACK, INKPLATE7_RED);

    x = display.getCursorX();
    y = display.getCursorY();

    display.setFont(&Inter12pt7b);
    display.setTextSize(1);

    display.setCursor(x, y);
    display.println(F("m/s"));

    // Labels underneath
    display.setFont(&Inter8pt7b);
    display.setTextSize(1);

    display.setCursor(257, 118);
    display.println(F("TEMPERATURE"));

    display.setCursor(428, 118);
    display.println(F("WIND SPEED"));
}
