#!/bin/bash
# Unix shell script to gzip JavaScript and HTML files individually
# Creates separate .gz files for each source file in ../data directory

# Get the directory where this script is located
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR" || exit 1

# Define paths relative to script directory
DATA_DIR="../data"

echo "Compressing JavaScript and HTML files with gzip..."
echo "Script directory: $SCRIPT_DIR"
echo "Data directory: $DATA_DIR"
echo ""

# Create ../data directory if it doesn't exist
if [ ! -d "$DATA_DIR" ]; then
    echo "Creating $DATA_DIR directory..."
    mkdir -p "$DATA_DIR"
    echo ""
fi

# Check if gzip is available
if ! command -v gzip &> /dev/null; then
    echo "Error: gzip command not found!"
    echo "Please install gzip package"
    exit 1
fi

# Copy entire directory to data with rsync or cp
echo "Copying source files to $DATA_DIR..."
if command -v rsync &> /dev/null; then
    rsync -av --delete \
        --exclude='.git' \
        --exclude='build*.js' \
        --exclude='build*.sh' \
        --exclude='build*.bat' \
        --exclude='*.md' \
        --exclude='*.gz' \
        . "$DATA_DIR/" > /dev/null 2>&1
else
    cp -r . "$DATA_DIR/"
    # Remove unnecessary files
    rm -f "$DATA_DIR/build-bundle.js"
    rm -f "$DATA_DIR/build_data.sh"
    rm -f "$DATA_DIR/build_data.bat"
    rm -f "$DATA_DIR"/*.md
    rm -f "$DATA_DIR"/*.gz
fi

echo "Copy complete!"
echo ""

# Navigate to data directory and gzip all js and html files
cd "$DATA_DIR" || exit 1

echo "Compressing files in $DATA_DIR..."
count=0

# Gzip all .js and .html files
for file in $(find . -name "*.js" -o -name "*.html" -o -name "*.ico" 2>/dev/null | sort); do
    # Skip if directory
    [ -d "$file" ] && continue

    # Skip build and shell scripts
    case "$file" in
        *build*.js|*build*.sh|*build*.bat|*.md|*.gz)
            continue
            ;;
    esac

    echo "  Compressing $file..."
    if gzip -9 "$file"; then
        ((count++))
    fi
done

echo ""
echo "Gzip compression completed!"
echo "Total files processed: $count"
echo ""

# Clean up non-gz files (keep only .gz files)
echo "Cleaning up non-compressed files..."
cleanup_count=0

for file in $(find . -type f ! -name "*.gz" ! -name "data_index.html" 2>/dev/null); do
    case "$file" in
        */.git/*|./.|*/.*)
            continue
            ;;
    esac

    if [ -f "$file" ]; then
        echo "  Removing $file"
        rm "$file"
        ((cleanup_count++))
    fi
done

echo "Removed $cleanup_count non-compressed files"
echo ""

# Rename data_index.html.gz to index.html.gz if it exists
if [ -f "data_index.html.gz" ]; then
    echo "Renaming data_index.html.gz to index.html.gz..."
    mv "data_index.html.gz" "index.html.gz"
fi

echo ""
echo "Final .gz file count:"
find . -name "*.gz" -type f 2>/dev/null | wc -l
echo ""
echo "Deployment structure ready:"
echo "  $DATA_DIR/               - All compressed .js and .html files"
echo "  $DATA_DIR/jsfreechart/   - JSFreeChart library files (compressed)"
echo ""
echo "Files are ready for web server deployment with Content-Encoding: gzip header"
