/*
 * Arduino-Http-Requests Library
 * File: Http.h
 * 
 * Copyright (c) 2025 Dominik Werner
 * https://github.com/dowerner/Arduino-Http-Requests
 *
 * This file is part of the Arduino-Http-Requests library and is licensed
 * under the MIT License. See LICENSE file for details.
 */

#pragma once

#include "LinkedList.h"
#include "HttpRequest.h"
#include "HttpResponse.h"
#include "HttpCallback.h"
#include "UrlParsing.h"
#include "HttpResponseParsing.h"

#define HTTP_RESPONSE_BUFFER_SIZE 1024
#define RESPONSE_TIMEOUT_MS 60000

/**
 * Base class for all specific HTTP request implementers.
 */
template <typename TClient>
class Http {
public:

    /**
     * @brief Sends an HTTP GET request to the specified URL.
     *
     * @param url The URL to request.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus get(const char* url, RequestCompletedCallback* onRequestCompleted) {
        return get(String(url), onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP GET request to the specified URL.
     *
     * @param url The URL to request.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus get(const String& url, RequestCompletedCallback* onRequestCompleted) {
        return sendRequest(url, onRequestCompleted, "GET");
    }

    /**
     * @brief Sends an HTTP POST request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param body The body of the POST request as a JSON string.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus post(const char* url, const char* body, RequestCompletedCallback* onRequestCompleted) {
        return post(String(url), String(body), onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP POST request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param body The body of the POST request as a JSON string.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus post(const char* url, const String& body, RequestCompletedCallback* onRequestCompleted) {
        return post(String(url), body, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP POST request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param body The body of the POST request as a JSON document.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus post(const char* url, const JsonDocument& body, RequestCompletedCallback* onRequestCompleted) {
        return post(String(url), body, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP POST request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param body The body of the POST request as a JSON document.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus post(const String& url, const JsonDocument& body, RequestCompletedCallback* onRequestCompleted) {
        String serializedBody;
        serializeJson(body, serializedBody);
        return post(url, serializedBody, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP POST request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param body The body of the POST request as a JSON string.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus post(const String& url, const String& body, RequestCompletedCallback* onRequestCompleted) {
        String commands[] = { 
            String("Content-Type: application/json"),
            String("Content-Length: ") + String(body.length()),
            String(),
            body
        };
        return sendRequest(url, onRequestCompleted, "POST", commands, 4);
    }

    /**
     * @brief Sends an HTTP PUT request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be put.
     * @param body The body of the PUT request, as a JSON document.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus put(const char* url, const JsonDocument& body, RequestCompletedCallback* onRequestCompleted) {
        return put(String(url), body, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP PUT request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be put.
     * @param body The body of the PUT request, as a JSON document.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus put(const String& url, const JsonDocument& body, RequestCompletedCallback* onRequestCompleted) {
        String serializedBody;
        serializeJson(body, serializedBody);
        return put(url, serializedBody, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP PUT request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be put.
     * @param body The body of the PUT request, as a JSON string.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus put(const char* url, const char* body, RequestCompletedCallback* onRequestCompleted) {
        return put(String(url), String(body), onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP PUT request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be put.
     * @param body The body of the PUT request, as a JSON string.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus put(const char* url, const String& body, RequestCompletedCallback* onRequestCompleted) {
        return put(String(url), body, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP PUT request with a JSON body to the specified URL.
     *
     * @param url The URL to which data will be put.
     * @param body The body of the PUT request, as a JSON string.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus put(const String& url, const String& body, RequestCompletedCallback* onRequestCompleted) {
        String commands[] = { 
            String("Content-Type: application/json"),
            String("Content-Length: ") + String(body.length()),
            String(),
            body
        };
        return sendRequest(url, onRequestCompleted, "PUT", commands, 4);
    }

    /**
     * @brief Sends an HTTP DELETE request to the specified URL.
     *
     * @param url The URL from which a resource will be deleted.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus del(const char* url, RequestCompletedCallback* onRequestCompleted) {
        return del(String(url), onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP DELETE request to the specified URL.
     *
     * @param url The URL from which a resource will be deleted.
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus del(const String& url, RequestCompletedCallback* onRequestCompleted) {
        return sendRequest(url, onRequestCompleted, "DELETE");
    }

    /**
     * @brief Sends an HTTP POST request with a form string body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param formString The form string to be posted (e.g. username=test&password=mypass1234)
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus postAsForm(const char* url, const char* formString, RequestCompletedCallback* onRequestCompleted) {
        return postAsForm(String(url), formString, onRequestCompleted);
    }

    /**
     * @brief Sends an HTTP POST request with a form string body to the specified URL.
     *
     * @param url The URL to which data will be posted.
     * @param formString The form string to be posted (e.g. username=test&password=mypass1234)
     * @param onRequestCompleted Callback function invoked when the response is received.
     * @return HttpRequstStatus Status indicating the current status of the request.
     */
    HttpRequstStatus postAsForm(const String& url, const char* formString, RequestCompletedCallback* onRequestCompleted) {
        String formData = String(formString);
        String commands[] = { 
            String("Content-Type: application/x-www-form-urlencoded"),
            String("Content-Length: ") + String(formData.length()),
            String(),
            formData
        };
        return sendRequest(url, onRequestCompleted, "POST", commands, 4);
    }

    virtual String getLocalIP() = 0;  // Pure virtual function - must be implemented by derived classes

    void loop() {
        size_t requestCount = pendingRequests->getSize();

        if (requestCount == 0) return;

        unsigned long ts = millis();

        for (size_t i = 0; i < requestCount; ++i) {
            HttpRequest<TClient>* request = nullptr;
            if (!pendingRequests->get(i, request) || request == nullptr || !request->client->available()) {
                // check if response timed out
                unsigned long requestDurationMs = ts - request->requestStartTS;
                if (requestDurationMs > RESPONSE_TIMEOUT_MS) {
                    if (request->callback != nullptr) {
                        HttpResponse timeoutResponse;
                        timeoutResponse.status = HttpRequstStatus::NoResponse;
                        timeoutResponse.responseCode = 0;
                        request->callback(timeoutResponse);
                    }

                    // Remove and delete the timed out request
                    pendingRequests->removeAt(i);
                    delete request;
                    
                    // Don't increment i since the next item is now at the same index
                    --i;
                    --requestCount;  // Update the count since we removed an item
                }
                continue;
            }

            // Reset state for this request
            size_t bytesRead = 0;
            String responseText = String();
            char localRespBuffer[HTTP_RESPONSE_BUFFER_SIZE];
            char currentChar;

            // Process this request
            while (request->client->available()) {
                currentChar = request->client->read();
                localRespBuffer[bytesRead++] = currentChar;

                if (bytesRead == HTTP_RESPONSE_BUFFER_SIZE) {
                    responseText.concat(localRespBuffer);
                    bytesRead = 0;
                }
            }
            
            // Add remaining bytes
            if (bytesRead > 0) {
                responseText += String(localRespBuffer, bytesRead);
            }
            
            // Serial.println(responseText);
            HttpResponse response = HttpResponseParsing::parseResponse(responseText);
            response.status = HttpRequstStatus::Completed;

            if (request->callback != nullptr) {
                request->callback(response);
            }

            // Remove and delete the completed request
            pendingRequests->removeAt(i);
            delete request;
            
            // Don't increment i since the next item is now at the same index
            --i;
            --requestCount;  // Update the count since we removed an item
        }
    }

    Http<TClient>() {
        pendingRequests = new List<HttpRequest<TClient>*>();
    }
    
    ~Http<TClient>() {
        // Clean up all HttpRequest objects
        HttpRequest<TClient>* request;
        while (pendingRequests->getSize() > 0) {
            if (pendingRequests->get(0, request)) {
                delete request;  // This will delete the HttpRequest and its client
                pendingRequests->removeAt(0);
            }
        }
        delete pendingRequests;
    }
    
private:
    List<HttpRequest<TClient>*>* pendingRequests;

    HttpRequstStatus sendRequest(const String& url, RequestCompletedCallback* onRequestCompleted, const char* method) {
        return sendRequest(url, onRequestCompleted, method, nullptr, 0);
    }

    HttpRequstStatus sendRequest(const String& url, RequestCompletedCallback* onRequestCompleted, const char* method, String clientCommands[], int16_t commandCount) {
        ParsedUrl parsedUrl = UrlParsing::parseUrl(url);
        if (parsedUrl.failed) {
            return HttpRequstStatus::Failed_InvalidUrl;
        }

        // Create and add request to the list
        HttpRequest<TClient>* request = new HttpRequest<TClient>();
        request->requestStartTS = millis();
        request->client = new TClient();
        request->callback = onRequestCompleted;
        pendingRequests->add(request);
        
        if (!request->client->connect(parsedUrl.host.c_str(), parsedUrl.port)) {
            return HttpRequstStatus::Failed_UnableToConnectToServer;
        }

        String command = String(method) + String(" ") + parsedUrl.path + String(" HTTP/1.1");
        String hostPart = String("Host: ") + getLocalIP();

        request->client->println(command.c_str());
        request->client->println(hostPart.c_str());
        request->client->println("Connection: close");

        // Serial.println("Custom commands\n");
        if (clientCommands != nullptr) {
            for (int16_t i = 0; i < commandCount; ++i) {
                request->client->println(clientCommands[i].c_str());
                // Serial.println(clientCommands[i]);
            }
        }
        request->client->println();

        return HttpRequstStatus::Sent;
    }

};
