#include "./arduino_percent.hpp"

namespace {
    constexpr char CODE[] = "0123456789ABCDEF";
    constexpr char UNRESERVED[] = "-._~0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

    uint8_t indexOf(char search) {
        if ('`' < search) {
            search -= ' ';
        }

        for (uint8_t i = 0; i < 16; i++) {
            if (::CODE[i] == search) {
                return i;
            }
        }

        return 0xFF;
    }

    bool isUnreserved(char search) {
        for (const auto &v: ::UNRESERVED) {
            if (v == search) {
                return true;
            }
        }

        return false;
    }
}

/**
* @brief Convert URL-unsafe string to percent-encoded string.
*/
void percent::encode(const char* input, char* output) {
    while (*input != '\0') {
        if (::isUnreserved(*input)) {
            *output++ = *input;
        } else {
            *output++ = '%';
            *output++ = ::CODE[*input >> 0x04];
            *output++ = ::CODE[*input & 0x0F];
        }

        input++;
    }

    *output = '\0';
}

/**
* @brief Calculate number of output characters.
*/
size_t percent::encodeLength(const char* input) {
    size_t length = 0;

    while (*input != '\0') {
        length += ::isUnreserved(*input++) ? 1 : 3;
    }

    return length + 1;
}

/**
* @brief Convert percent-encoded string to URL-unsafe string.
*/
void percent::decode(const char* input, char* output) {
    while (*input != '\0') {
        if (*input == '%') {
            *output++ = (::indexOf(*++input) << 4) + ::indexOf(*++input);
        } else {
            *output++ = *input;
        }

        input++;
    }

    *output = '\0';
}

/**
* @brief Calculate number of output characters.
*/
size_t percent::decodeLength(const char* input) {
    size_t length = 0;

    while (*input != '\0') {
        input += *input == '%' ? 3 : 1;
        length++;
    }

    return length + 1;
}