#ifndef MOBILER_NMV_H
#define MOBILER_NMV_H

#include <Arduino.h>
#include <PID_v2.h>
#include <math.h>

#define MAX_POINTS 20

// ---------------------- Structs ----------------------
struct Pose {
    float x;
    float y;
    float heading;
};

struct WheelVelocities {
    float velL_deg;
    float velR_deg;
    bool goalReached;
};


struct PathData {
    float goalX[MAX_POINTS];
    float goalY[MAX_POINTS];
    int length;
};

struct AccAvg {
    float x;
    float y;
    float z;
    int samples;
};



// ---------------------- Class ----------------------
class mobileR_NMV {
  public:
    mobileR_NMV();

    // ----- Motor & encoder attach -----
    //void attach_motors(int pinL, int pinR);
    //void attach_encoders(int pinL, int pinR);
    
    void attach_motors(char left, int pinL, char right, int pinR);
    void attach_encoders(char left, int pinL, char right, int pinR);
    
    //void pulseLimit(char type1, int value1, char type2, int value2);
    // Pulse limit function using string labels
void pulseLimit(const char* type1, int value1, const char* type2, int value2);



    // ----- Encoder calibration -----
    void caliEncoder(char wheel, long counts, float revs);

    // ----- PID setup -----
    void setPID_L(double Kp, double Ki, double Kd = 0, double minOut = 33, double maxOut = 70, int sampleTime = 5);
    void setPID_R(double Kp, double Ki, double Kd = 0, double minOut = 33, double maxOut = 70, int sampleTime = 5);

    // ----- Odometer calibration -----
    void caliODOM(float wheelR, float wheelB);

    // ----- Obstacle recovery -----
    void navi_obst_recovery(unsigned long stage1_ms, unsigned long stage2_ms, unsigned long stage3_ms);
    void navi_caliP(float k_val);

    // ----- Velocity updates -----
    double updateVelocityL();
    double updateVelocityR();

    // ----- PID compute -----
    double computePID_L(double currentVel, double targetVel);
    double computePID_R(double currentVel, double targetVel);

    // ----- Odometry -----
    Pose odometer(float velL_deg, float velR_deg, bool obstacleMode = false, int sensorValue = 0, int threshold = 0);

    // ----- Navigation -----
    WheelVelocities navigate(Pose rPose, float goalX, float goalY);

    // ----- Encoder ISR handlers -----
    void handleEncoderL();
    void handleEncoderR();
    
    // ----- Parsing functions -----
    PathData handlePath(const String& data);
    AccAvg handleACC(const String& data);


  private:
    static mobileR_NMV* instance;  // pointer for static ISR
    static void ISR_encoderL();
    static void ISR_encoderR();

    // Motor pins
    int motorL_pin, motorR_pin;
    int encL_pin, encR_pin;

    // Robot params
    float wheelRadius;
    float wheelBase;
    
    int minPulse = 33;   // default min PWM
    int maxPulse = 70;   // default max PWM

    // PID controllers
    double Kp_L, Ki_L, Kd_L, pidOutputL, filtered_vel_L, targetL;
    double Kp_R, Ki_R, Kd_R, pidOutputR, filtered_vel_R, targetR;
    PID PID_L;
    PID PID_R;

    // Encoder variables
    volatile long encoderCountL, encoderCountR;
    long lastCountL, lastCountR;
    unsigned long lastVelTimeL, lastVelTimeR;
    double degPerPulseL, degPerPulseR;

    // Pose
    float x, y, theta;
    unsigned long lastOdomTime;

    // Obstacle recovery
    bool recoveryActive;
    unsigned long recoveryStart;
    unsigned long stageTimes[3];
    int recoveryStage;

    // Navigation
    float kP_omega;
};

#endif

