#include "DimuthuRobotLib.h"
#include <math.h>

// Robot constants
const double Y_Rest = 70.0;
const double Z_Rest = -70.0;
const double J2L = 57.0;
const double J3L = 80.0;

SpiderBot_NMV::SpiderBot_NMV() {
  sequenceLen = 0;
  moveSpeed = 1.0;
  lastUpdate = 0;
  for (int i = 0; i < MAX_LEGS; i++) {
    legSequence[i] = i;   
    phase[i] = 0.0;
    gaitPointsCount[i] = 0;
    phaseOffset[i] = 0.0;
  }
}

void SpiderBot_NMV::attachLeg(LegName leg, int servo1Pin, int servo2Pin) {
    legServo1[(int)leg].attach(servo1Pin, 500, 2500);
    legServo2[(int)leg].attach(servo2Pin, 500, 2500);
    
    // Attach R/Tibia servos for all legs on their respective pins
   legServo3[0].attach(7);  // Leg FL
   legServo3[0].write(40);

   legServo3[1].attach(13);  // Leg FR
   legServo3[1].write(40);

   legServo3[2].attach(10);  // Leg BL
   legServo3[2].write(50);

   legServo3[3].attach(2);  // Leg BR
   legServo3[3].write(140);
   
   // myservo.write(50);   //7           // tell servo to go to position in variable 'pos'
   // myservo2.write(20); //13
   // myservo3.write(80); //10
   // myservo4.write(100+20); //2  

}


// ---------------------
// SET FOR ONE LEG
// ---------------------
void SpiderBot_NMV::setGaitPoints(int legIndex, float X[], float Y[], float Z[], int pointsCount) {
  gaitPointsCount[legIndex] = pointsCount;
  for (int i = 0; i < pointsCount; i++) {
    gaitX[legIndex][i] = X[i];
    gaitY[legIndex][i] = Y[i];
    gaitZ[legIndex][i] = Z[i];
  }
}

// ---------------------
// SET SAME GAIT FOR ALL LEGS
// ---------------------
/*
void SpiderBot_NMV::setGaitPoints(float X[], float Y[], float Z[], int pointsCount) {
  for (int leg = 0; leg < MAX_LEGS; leg++) {
    gaitPointsCount[leg] = pointsCount;
    for (int i = 0; i < pointsCount; i++) {
      gaitX[leg][i] = X[i];
      gaitY[leg][i] = Y[i];
      gaitZ[leg][i] = Z[i];
    }
  }
}
*/

void SpiderBot_NMV::setGaitPoints(LegName leg, float X[], float Y[], float Z[], int pointsCount) {
    int legIndex = (int)leg; // cast enum to int
    if (legIndex < 0 || legIndex >= MAX_LEGS) return;

    gaitPointsCount[legIndex] = pointsCount;
    for (int i = 0; i < pointsCount; i++) {
        gaitX[legIndex][i] = X[i];
        gaitY[legIndex][i] = Y[i];
        gaitZ[legIndex][i] = Z[i];
    }
}



void SpiderBot_NMV::setLegSequence(int sequence[], int len) {
  sequenceLen = len;
  for (int i = 0; i < len; i++) legSequence[i] = sequence[i];
}

void SpiderBot_NMV::setCycleTime(float cycleTimeMs) {
    // dt = run() interval in ms (fixed 20ms)
    const float dt = 20.0;

    // Use the first leg to calculate moveSpeed
    int points = gaitPointsCount[0]; // assume all legs have same points

    if (points <= 0) {
        moveSpeed = 1.0; // default fallback
        return;
    }

    // moveSpeed = points / (cycleTime / dt)
    moveSpeed = points / (cycleTimeMs / dt);
}


void SpiderBot_NMV::setPhaseOffset(LegName leg, float offset) {
    int legIndex = (int)leg;  // cast enum to int for array indexing
    if (legIndex < 0 || legIndex >= MAX_LEGS) return;

    phaseOffset[legIndex] = offset;
    phase[legIndex] = offset * gaitPointsCount[legIndex];
}
/*
LegXYZ SpiderBot_NMV::runSingleLeg(LegName leg) {
    unsigned long now = millis();
    if (now - lastUpdate < 20) return lastOutput; // 50Hz update
    lastUpdate = now;

    int legIndex = (int)leg;
    if (gaitPointsCount[legIndex] == 0) return lastOutput;

    // advance phase
    phase[legIndex] += moveSpeed;
    if (phase[legIndex] >= gaitPointsCount[legIndex]) {
        phase[legIndex] -= gaitPointsCount[legIndex]; // wrap around for cyclic motion
    }

    int p1 = int(phase[legIndex]);
    int p2 = (p1 + 1) % gaitPointsCount[legIndex];
    float frac = phase[legIndex] - p1;

    LegXYZ out;
    out.leg = legIndex;
    out.X = gaitX[legIndex][p1] + frac * (gaitX[legIndex][p2] - gaitX[legIndex][p1]);
    out.Y = gaitY[legIndex][p1] + frac * (gaitY[legIndex][p2] - gaitY[legIndex][p1]);
    out.Z = gaitZ[legIndex][p1] + frac * (gaitZ[legIndex][p2] - gaitZ[legIndex][p1]);

    lastOutput = out;
    return out;
}
*/

/*
LegXYZ SpiderBot_NMV::run() {
  unsigned long now = millis();
  if (now - lastUpdate < 20) return lastOutput; // keep last output if called too fast
  lastUpdate = now;

  static int seqIndex = 0;
  int leg = legSequence[seqIndex];

  if (gaitPointsCount[leg] == 0) return lastOutput;

  phase[leg] += moveSpeed;
  if (phase[leg] >= gaitPointsCount[leg]) {
    phase[leg] = 0;
    seqIndex = (seqIndex + 1) % sequenceLen;
  }

  int p1 = int(phase[leg]);
  int p2 = (p1 + 1) % gaitPointsCount[leg];
  float frac = phase[leg] - p1;

  LegXYZ out;
  out.leg = leg;
  out.X = gaitX[leg][p1] + frac * (gaitX[leg][p2] - gaitX[leg][p1]);
  out.Y = gaitY[leg][p1] + frac * (gaitY[leg][p2] - gaitY[leg][p1]);
  out.Z = gaitZ[leg][p1] + frac * (gaitZ[leg][p2] - gaitZ[leg][p1]);

  lastOutput = out;
  return out;
}
*/

void SpiderBot_NMV::run(LegXYZ outLegs[MAX_LEGS]) {
    unsigned long now = millis();
    if (now - lastUpdate < 20) return; // 50Hz update
    lastUpdate = now;

    for (int i = 0; i < MAX_LEGS; i++) {
        if (gaitPointsCount[i] == 0) continue;

        // advance phase
        phase[i] += moveSpeed;
        if (phase[i] >= gaitPointsCount[i]) phase[i] -= gaitPointsCount[i];

        int p1 = int(phase[i]);
        int p2 = (p1 + 1) % gaitPointsCount[i];
        float frac = phase[i] - p1;

        outLegs[i].leg = i;
        outLegs[i].X = gaitX[i][p1] + frac * (gaitX[i][p2] - gaitX[i][p1]);
        outLegs[i].Y = gaitY[i][p1] + frac * (gaitY[i][p2] - gaitY[i][p1]);
        outLegs[i].Z = gaitZ[i][p1] + frac * (gaitZ[i][p2] - gaitZ[i][p1]);
    }
}

LegXYZ SpiderBot_NMV::runLeg(LegName leg) {
    unsigned long now = millis();
    if (now - lastUpdate < 20) return lastOutput; // 50Hz update
    lastUpdate = now;

    int i = (int)leg;
    if (gaitPointsCount[i] == 0) return lastOutput; // fallback

    phase[i] += moveSpeed;
    if (phase[i] >= gaitPointsCount[i]) phase[i] -= gaitPointsCount[i];

    int p1 = int(phase[i]);
    int p2 = (p1 + 1) % gaitPointsCount[i];
    float frac = phase[i] - p1;

    LegXYZ out;
    out.leg = i;
    out.X = gaitX[i][p1] + frac * (gaitX[i][p2] - gaitX[i][p1]);
    out.Y = gaitY[i][p1] + frac * (gaitY[i][p2] - gaitY[i][p1]);
    out.Z = gaitZ[i][p1] + frac * (gaitZ[i][p2] - gaitZ[i][p1]);

    lastOutput = out;
    return out;
}





void SpiderBot_NMV::updateLeg(const LegIK &ik, int leg) {
  float J1 = ik.J1;
  float J2 = ik.J2;

  switch (leg) {
    case 0:
      legServo1[leg].write(120 - J1);
      legServo2[leg].write(80  - J2);
      break;
    case 1:
      legServo1[leg].write(50 + J1);
      legServo2[leg].write(50 + J2);
      break;
    case 2:
      legServo1[leg].write(120 + J1);
      legServo2[leg].write(105 - J2);
      break;
    case 3:
      legServo1[leg].write(50 - J1);
      legServo2[leg].write(30 + J2);
      break;
  }
}


LegIK SpiderBot_NMV::calcIK(float X, float Y, float Z) {
  LegIK ik;

  Y += Y_Rest;
  Z += Z_Rest;

  ik.J1 = atan2(X, Y) * 180.0 / PI;

  double H = sqrt(X*X + Y*Y);
  double L = sqrt(H*H + Z*Z);

  double B = acos((L*L + J2L*J2L - J3L*J3L) / (2 * L * J2L)) * 180.0 / PI;
  double A = atan2(Z, H) * 180.0 / PI;

  ik.J2 = B + A;

  return ik;
}

