#include "DimuthuRobotLib.h"
#include <Arduino.h>

Quadcopter_NMV::Quadcopter_NMV(int pinFL, int pinFR, int pinBR, int pinBL) {
    motorPins[FL] = pinFL;
    motorPins[FR] = pinFR;
    motorPins[BR] = pinBR;
    motorPins[BL] = pinBL;
}

// Attach motor with default mixing + trim
void Quadcopter_NMV::attach_motor(MotorID mID, int minS, int maxS) {
    minSpeed[mID] = minS;
    maxSpeed[mID] = maxS;

    motorMix[mID][0] = (mID<2?-1:1); // default pitch
    motorMix[mID][1] = (mID%2==0?-1:1); // default roll
    motorMix[mID][2] = (mID<2?-1:1);    // default yaw
    motorMix[mID][3] = motorTrim[mID]; // trim

    motors[mID].attach(motorPins[mID]);
}

// Attach motor with custom mix + trim
void Quadcopter_NMV::attach_motor(MotorID mID, float mix[4], int minS, int maxS) {
    minSpeed[mID] = minS;
    maxSpeed[mID] = maxS;

    for(int i=0;i<4;i++) motorMix[mID][i] = mix[i];

    motors[mID].attach(motorPins[mID]);
}

// Set trims
void Quadcopter_NMV::setTrim(MotorID mID, float trimValue) { motorTrim[mID] = trimValue; motorMix[mID][3]=trimValue; }
void Quadcopter_NMV::setTrims(float tFL,float tFR,float tBR,float tBL){
    setTrim(FL,tFL); setTrim(FR,tFR); setTrim(BR,tBR); setTrim(BL,tBL);
}

// IMU init
void Quadcopter_NMV::imu_init() {
    Wire.begin();
    mpu.initialize();
    if(!mpu.testConnection()) { while(1); }
    mpu.dmpInitialize();
    mpu.setDMPEnabled(true);
    dmpReady = true;
    lastTime = millis();
}

// IMU read
IMUData Quadcopter_NMV::imu_read() {
    IMUData data = {0,0,0};
    if(!dmpReady) return data;
    if(mpu.dmpGetCurrentFIFOPacket(fifoBuffer)){
        mpu.dmpGetQuaternion(&q, fifoBuffer);
        mpu.dmpGetGravity(&gravity,&q);
        mpu.dmpGetYawPitchRoll(ypr,&q,&gravity);
        data.yaw = ypr[0]*180/M_PI;
        data.pitch = ypr[1]*180/M_PI;
        data.roll = ypr[2]*180/M_PI;
    }
    return data;
}

// Set PID
void Quadcopter_NMV::setPID_pitch(float params[2]){ pidPitch[0]=params[0]; pidPitch[1]=params[1]; }
void Quadcopter_NMV::setPID_roll(float params[2]){ pidRoll[0]=params[0]; pidRoll[1]=params[1]; }
void Quadcopter_NMV::setPID_yaw(float params[2]){ pidYaw[0]=params[0]; pidYaw[1]=params[1]; }

// Calculate PID
void Quadcopter_NMV::calPID_pitch(float data, float target){
    float error = target - data;
    unsigned long currentTime = millis();
    float dt = (currentTime - lastTimePitch)/1000.0; // seconds
    if(dt <= 0) dt = 0.001; // avoid divide by zero
    //if(dt < 0.003) dt = 0.003; 
    pidOutPitch = pidPitch[0]*error + pidPitch[1]*(error-prevErrorPitch)/dt;
    //Serial.println(pidOutPitch);
    prevErrorPitch = error;
    lastTimePitch = currentTime;
}

void Quadcopter_NMV::calPID_roll(float data, float target){
    float error = target - data;
    unsigned long currentTime = millis();
    float dt = (currentTime - lastTimeRoll)/1000.0; // seconds
    if(dt <= 0) dt = 0.001;
    //if(dt < 0.003) dt = 0.003;   //
    pidOutRoll = pidRoll[0]*error + pidRoll[1]*(error-prevErrorRoll)/dt;
    prevErrorRoll = error;
    lastTimeRoll = currentTime;

    //Serial.println(pidOutRoll);
}

void Quadcopter_NMV::calPID_yaw(float data, float target){
    float error = target - data;
    unsigned long currentTime = millis();
    float dt = (currentTime - lastTimeYaw)/1000.0; // seconds
    if(dt <= 0) dt = 0.001; // avoid divide by zero
    //if(dt < 0.003) dt = 0.003; 
    pidOutYaw = pidYaw[0]*error + pidYaw[1]*(error-prevErrorYaw)/dt;
    prevErrorYaw = error;
    lastTimeYaw = currentTime;
    
    //Serial.println(pidOutRoll);
}

// Run motor
void Quadcopter_NMV::run_motor(MotorID mID,int baseSpeed){
    float val = baseSpeed + motorMix[mID][0]*pidOutPitch + motorMix[mID][1]*pidOutRoll+ motorMix[mID][2]*pidOutYaw + motorMix[mID][3];
    //Serial.println(baseSpeed);
    //Serial.println("__");
    //Serial.println(motorMix[mID][0]);
    //Serial.println(val);
    //delay(10);
    val = constrain(val,minSpeed[mID],maxSpeed[mID]);
    
    //motors[mID].writeMicroseconds((int)val);
    motors[mID].write(val);
    lastMotorOutput[mID] = val;
    
}

// Run all motors
void Quadcopter_NMV::run_motors(int baseSpeed){
    for(int i=0;i<4;i++) run_motor((MotorID)i, baseSpeed);
}

///*
CommandData Quadcopter_NMV::updateCommands() {
    unsigned long t = millis();

    // -------------------------------
    // 1) READ SERIAL COMMANDS
    // -------------------------------
    while (Serial.available() > 0) {
        char c = Serial.peek();

        // THROTTLE: u, d, l
        if (c == 'u' || c == 'd' || c == 'l') {
            c = Serial.read();

            if (c == 'u') {
                throttle_ramp_target += 5;
                if (hover_throttle + throttle_ramp_target > maxSpeed[0])
                    throttle_ramp_target = maxSpeed[0] - hover_throttle;
            }
            else if (c == 'd') {
                throttle_ramp_target -= 5;
                if (hover_throttle + throttle_ramp_target < minSpeed[0])
                    throttle_ramp_target = minSpeed[0] - hover_throttle;
            }
            else if (c == 'l') {
                landingActive = true;
                landingStart = millis();
                landingStartValue = target_throttle_manual;
                throttle_ramp_target = 0;
                Serial.println("Landing initiated...");
            }
        }

        // MOVEMENT COMMAND: "x y z"
        else {
            String line = Serial.readStringUntil('\n');
            line.trim();
            if (line.length() == 0) break;

            int s1 = line.indexOf(' ');
            int s2 = line.indexOf(' ', s1 + 1);

            if (s1 < 0 || s2 < 0) {
                Serial.println("Invalid format. Use: X Y Z");
                continue;
            }

            float x = line.substring(0, s1).toFloat();
            float y = line.substring(s1 + 1, s2).toFloat();
            float z = line.substring(s2 + 1).toFloat();

            // convert cm → motion duration
            durX = x * 500;  startX = t; moveX = true;
            durY = y * 500;  startY = t; moveY = true;
            durZ = z * 500;  startZ = t; moveZ = true;
        }
    }

    // --------------------------------
    // 2) UPDATE MOVEMENT (X / Y / Z)
    // --------------------------------
    float target_pitch = 0;
    float target_roll  = 0;

    // X axis → pitch
    if (moveX) {
        unsigned long e = t - startX;
        if (e >= durX) moveX = false;
        else {
            float half = durX / 2.0;
            float p = (e <= half) ? (float)e / half : (float)(e - half) / half;
            float shape = (e <= half) ? p : (1.0 - p);
            target_pitch = startTilt * shape;
        }
    }

    // Y axis → roll
    if (moveY) {
        unsigned long e = t - startY;
        if (e >= durY) moveY = false;
        else {
            float half = durY / 2.0;
            float p = (e <= half) ? (float)e / half : (float)(e - half) / half;
            float shape = (e <= half) ? p : (1.0 - p);
            target_roll = startTilt * shape;
        }
    }

    // Z axis → throttle addition
    float target_throttle = hover_throttle + target_throttle_manual;

    if (moveZ) {
    
    //float target_throttle = hover_throttle; // base hover
    
        unsigned long e = t - startZ;

        if (e >= durZ) moveZ = false;

        float half = durZ / 2.0;
        float ramp = 0;

        if (moveZ) {
            ramp = (e <= half)
                   ? hover_lift * (float)e / half
                   : hover_lift * (1.0 - (float)(e - half) / half);
        }

        //target_throttle = hover_throttle + target_throttle_manual + ramp;
        
        target_throttle += ramp+target_throttle_manual;

	// --- Clamp ---
	if (target_throttle > maxSpeed[0]) target_throttle = maxSpeed[0];
	if (target_throttle < minSpeed[0]) target_throttle = minSpeed[0];
    }

    // --------------------------------
    // 3) THROTTLE MANUAL RAMP & LANDING
    // --------------------------------
    if (!landingActive) {
        if (target_throttle_manual < throttle_ramp_target) target_throttle_manual++;
        else if (target_throttle_manual > throttle_ramp_target) target_throttle_manual--;
    }

    if (landingActive) {
        unsigned long e = millis() - landingStart;
        if (e >= landingDuration) {
            target_throttle_manual = 0;
            landingActive = false;
        } else {
            float ratio = 1.0 - (float)e / landingDuration;
            target_throttle_manual = landingStartValue * ratio;
        }
    }

    // Clamp throttle
    if (target_throttle > maxSpeed[0]) target_throttle = maxSpeed[0];
    if (target_throttle < minSpeed[0]) target_throttle = minSpeed[0];
    
    
    
    // -------------------------------
    // 4) RETURN COMMAND STRUCT
    // -------------------------------
    CommandData out;
    out.speed        = target_throttle;
    out.target_pitch = target_pitch;
    out.target_roll  = target_roll;
    
    return out;
}

//*/
void Quadcopter_NMV::armMotors() {
    for (int i = 0; i < 4; i++) {
        motors[i].write(minSpeed[i]);  // usually 1000
    }
    delay(3000); // let ESC beep and arm
}


void Quadcopter_NMV::printMotorOutputs() {
    Serial.print("FL: "); Serial.print(lastMotorOutput[FL]);
    Serial.print("  FR: "); Serial.print(lastMotorOutput[FR]);
    Serial.print("  BR: "); Serial.print(lastMotorOutput[BR]);
    Serial.print("  BL: "); Serial.println(lastMotorOutput[BL]);
}

