#ifdef ARDUINO

#include "roo_transport/link/arduino/link_stream.h"

#include <cstddef>

namespace roo_transport {

LinkStream::LinkStream(Channel& channel, uint32_t my_stream_id)
    : link_(channel, my_stream_id) {}

LinkStream::LinkStream(Link socket) : link_(std::move(socket)) {}

int LinkStream::available() { return in().available(); }

int LinkStream::read() {
  if (available() == 0) {
    yield();
    return -1;
  }
  return in().read();
}

int LinkStream::peek() { return in().peek(); }

size_t LinkStream::readBytes(char* buffer, size_t length) {
  return timedRead((roo::byte*)buffer, length, roo_time::Millis(getTimeout()));
}

size_t LinkStream::readBytes(uint8_t* buffer, size_t length) {
  return timedRead((roo::byte*)buffer, length, roo_time::Millis(getTimeout()));
}

size_t LinkStream::write(uint8_t val) {
  out().write((const roo::byte*)&val, 1);
  return 1;
}

size_t LinkStream::write(const uint8_t* buffer, size_t size) {
  return out().writeFully((const roo::byte*)buffer, size);
}

int LinkStream::availableForWrite() { return out().availableForWrite(); }

void LinkStream::flush() { out().flush(); }

LinkStatus LinkStream::status() const { return link_.status(); }

void LinkStream::awaitConnected() { link_.awaitConnected(); }

bool LinkStream::awaitConnected(roo_time::Duration timeout) {
  return link_.awaitConnected(timeout);
}

size_t LinkStream::timedRead(roo::byte* buf, size_t count,
                             roo_time::Duration timeout) {
  roo_time::Uptime start = roo_time::Uptime::Now();
  size_t total = 0;
  if (in().status() != roo_io::kOk) return -1;
  while (count > 0) {
    for (int i = 0; i < 100; ++i) {
      size_t result = in().tryRead(buf, count);
      if (result == 0) {
        if (in().status() != roo_io::kOk) return -1;
        // link_.channel_->loop();
      } else {
        total += result;
        count -= result;
      }
      if (count == 0) return total;
      yield();
    }
    if (roo_time::Uptime::Now() - start > timeout) break;
    delay(1);
  }
  return total;
}

}  // namespace roo_transport

#endif
