/*
  DCF77RX - Arduino libary receiving and decoding DCF77 frames Copyright (c)
  2025 Wolfgang Schmieder.  All right reserved.

  Contributors:
  - Wolfgang Schmieder

  Project home: https://github.com/dac1e/DCF77RX/

  This library is free software; you can redistribute it and/or modify it
  the terms of the GNU Lesser General Public License as under published
  by the Free Software Foundation; either version 3.0 of the License,
  or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

/**
 * Receive frames from dcf77, to synchronize a software clock.
 */

#include "DCF77RX.h"

/**
 * The clock needs an initial Dcf77 frame to start. Seconds
 * since the last received Dcf77 are calculated via systick
 * from function millis() and used to keep the clock running.
 * The clock needs a Dcf77 frame update at least every
 * 2**32 milliseconds, which is approximately every 49 days.
 * Otherwise there will be a systick overrun and the clock
 * will provide wrong results.
 */

// Set the following macro to true if you want to watch when
// the clock is updated from a received Dcf77 frame.
#define PRINT_DCF77FRAME_EVENT true

#ifdef ARDUINO_ARCH_AVR
static constexpr int DCF77_PIN = 2;
#else
static constexpr int DCF77_PIN = 23;
#endif

class DCF77Clock : public DCF77RX<DCF77_PIN> {
  using baseClass = DCF77RX<DCF77_PIN>;

public:
  DCF77Clock()
    : mLastDcf77Frame(0), mState(INVALID), mSystickAtLastFrame(0) {
  }

  void begin() {
    baseClass::begin();
  }

  /**
   * Read the current time.
   *
   * @param[out] tm The actual time.
   * @param[out] millisec The number of expired milliseconds
   *  within the current second.
   *
   * @return false, as long as no Dcf77 frame was received.
   */
  bool getTime(DCF77::tm& tm, unsigned* millisec) {
    if(mState != INVALID) {
      // Disable interrupts to avoid race condition with onDCF77FrameReceived()
      // which is updating mSystickAtLastFrame and mLastDcf77Frame.
      noInterrupts();
      const uint32_t millisSinceLastFrame = millis() - mSystickAtLastFrame;
      const uint64_t dcf77frame = mLastDcf77Frame;
      interrupts();

      const uint32_t secSinceLastFrame = millisSinceLastFrame / 1000;
      dcf77frame2time(tm, dcf77frame);
      const DCF77::time_t timestamp = DCF77::tm_to_timestamp(tm);

      DCF77::timestamp_to_tm(tm, timestamp + secSinceLastFrame, tm.tm_isdst);
      if(millisec != nullptr) {
        *millisec = millisSinceLastFrame % 1000;
      }
      return true;
    }
    return false;
  }

private:
  /**
   * This function runs within the interrupt context and must
   * be executed quickly in order not to prevent other lower
   * priority interrupts to be serviced.
   */
  void onDCF77FrameReceived(const uint64_t dcf77frame, const uint32_t systick) override {
    mSystickAtLastFrame = systick;
    mLastDcf77Frame = dcf77frame;
    mState = VALID;
  }

  uint32_t mSystickAtLastFrame;
  uint64_t mLastDcf77Frame;

  enum STATE : int8_t {INVALID, VALID};
  STATE mState;
};

DCF77Clock dcf77Clock;

static constexpr size_t PRINTOUT_PERIOD = 1;
static uint32_t counter = 0;
static uint32_t lastSystick = 0;

//The setup function is called once at startup of the sketch
void setup()
{
  Serial.begin(9600);
  Serial.println();
  Serial.println("---------- DCF77Clock -----------");
  Serial.println("First frame may take some minutes");
  dcf77Clock.begin();
  lastSystick = millis() - PRINTOUT_PERIOD * 1000;
}

// The loop function is called in an endless loop
void loop()
{
  const uint32_t systick = millis();
  if(systick - lastSystick >= PRINTOUT_PERIOD * 1000) {
    PrintableDCF77tm tm;
    if(dcf77Clock.getTime(tm, nullptr) ) {
      Serial.print(tm);
      Serial.print(", isdst=");
      Serial.println(tm.tm_isdst);
    } else {
      Serial.print('[');
      Serial.print(counter);
      Serial.print("s]");
      Serial.print(" Waiting for completion of DCF77 frame on Arduino pin ");
      Serial.println(DCF77_PIN);
      counter += PRINTOUT_PERIOD;
    }
    lastSystick = systick;
  }
}
