/* ===========================================================================

Crystalfontz CFA039A0-N-V Arduino library example.

https://www.crystalfontz.com/product/cfa039a0nvdct-480x128-graphic-usb-tft-display-module

Mark Williams (2025)
Crystalfontz America Inc.

Distributed under the "The Unlicense".
http://unlicense.org
This is free and unencumbered software released into the public domain.
For more details, see the website above.

=========================================================================== */

#include <Arduino.h>
#include "CFA039A0-N-V.h"

//////////////////////////////////////////////////////////////////////////////////////////////////////
//PROJECT SETTINGS

//interface selection (select one)
#define IFACE_I2C
//#define IFACE_SPI

//////////////////////////////////////////////////////////////////////////////////////////////////////
// ARDUINO UNO CONNECTION DETAILS

//I2C Interface connections
// SEEEDUNIO <-> CFA039A0-N-V
// SCL <-> H1-P4
// SDA <-> H1-P3
// PIN9 <-> H1-P13 (data ready)
// GND <-> H1-P15 (ground)

//SPI Interface connections
// SEEEDUNIO <-> CFA039A0-N-V
// MOSI/PIN11 <-> H1-P8
// MISO/PIN12 <-> H1-P7
// SCK/PIN13 <-> H1-P10
// PIN10 <-> H1-P9 (slave select)
// PIN9 <-> H1-P12 (data ready)
// GND <-> H1-P15 (ground)

//////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef IFACE_I2C
# ifndef IFACE_SPI
#  ifndef IFACE_SERIAL
#    error YOU MUST SELECT A MODULE INTERFACE ABOVE
#  endif
# endif
#endif

//////////////////////////////////////////////////////////////////////////////////////////////////////

//interface settings
#ifdef IFACE_I2C
#include <Wire.h>
# define IFACE_I2C_ADDR		  0x44
# define IFACE_I2C_SPEED	  100000
# define IFACE_I2C_READY	  9
# define IFACE_I2C_REQ_LEN	4
#endif
#ifdef IFACE_SPI
# include <SPI.h>
# define IFACE_SPI_SPEED	  1000000UL
# define IFACE_SPI_BITFIRST	MSBFIRST
# define IFACE_SPI_MODE		  SPI_MODE0
# define IFACE_SPI_SS		    10
# define IFACE_SPI_READY  	9
# define IFACE_SPI_REQ_LEN	16
#endif

#define IFACE_READY_WAIT	50 /*mS*/

//////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef IFACE_SPI
SPISettings spiSettings = SPISettings(IFACE_SPI_SPEED, IFACE_SPI_BITFIRST, IFACE_SPI_MODE);
#endif

CFA039A0NV cfaModule;

//////////////////////////////////////////////////////////////////////////////////////////////////////

//rainbow RGB values + white and black
#define RAINBOW_COLORS  8
const uint8_t rainbow[RAINBOW_COLORS][3] =
{
  {255, 0, 0},    // Red
  {255, 127, 0},  // Orange
  {255, 255, 0},  // Yellow
  {0, 255, 0},    // Green
  {0, 0, 255},    // Blue
  {75, 0, 130},   // Indigo
  {148, 0, 211},  // Violet
  {255, 255, 255} // White
};

//////////////////////////////////////////////////////////////////////////////////////////////////////

void setup()
{
	//run at power-on/reset of the Seeeduino/Arduino
	//setup interfaces / variables
	
	//debug out
	Serial.begin(115200);
	Serial.write("\n\nsetup()\n");

	//module interface init
#ifdef IFACE_I2C
	Wire.begin();
	pinMode(IFACE_I2C_READY, INPUT_PULLUP); //I2C-Ready
#endif

#ifdef IFACE_SPI
	pinMode(IFACE_SPI_SS, OUTPUT); //SPI-SS
	digitalWrite(IFACE_SPI_SS, HIGH); //SS deselect
	pinMode(IFACE_SPI_READY, INPUT_PULLUP); //SPI-Ready
	SPI.begin();
#endif

  //module init
#ifdef IFACE_I2C  
  cfaModule.begin(Wire, IFACE_I2C_ADDR, IFACE_I2C_READY);
#endif
#ifdef IFACE_SPI
  cfaModule.begin(SPI, spiSettings, IFACE_SPI_SS, IFACE_SPI_READY);
#endif
  cfaModule.enableDebugOutput();

  //ping the module until we get a reply
  Serial.println(F("Attemping to PING the module..."));
  while(1)
  {
    if (cfaModule.cmdPing((uint8_t*)"PING", 4) == CFA039A0NV::CFA_OK)
    {
      //we got a response
      break;
    }
    Serial.println(F("no reply"));
    delay(500);
  }
  cfaModule.clearBuffers();
  Serial.println(F("Ping reply recieved, continuing."));
}

void loop(void)
{
  //main arduino api/framework loop
	CFA039A0NV::packet_t inPacket;
  int currentColor;
  uint8_t R,G,B;
  uint16_t RGB565;
  int currentSize;

  //setup the initial display
  Serial.println(F("Inital display setup"));

  //set the default background color to dark grey
  cfaModule.cmdSetGraphicsBackgroundColor(0x39c7);

  //clear the screen
  cfaModule.cmdClearDisplay();

  //set the display backlight brightness to 100%
  cfaModule.cmdSetDisplayBrightness(100);

  //set the default text color to yellow
  cfaModule.cmdSetGraphicsTextAColor(0, 0xFFE0);

  //write info text on the screen using the default TTF font
  cfaModule.cmdTTFNewText(
    1,                            //new object ID
    "Up/Down Keys Change Color",  //the text
    475, 70,                      //X, Y coords
    CFA039A0NV::JUSTIFY_RIGHT,    //right-justify text
    1,                            //z-index of 1
    CFA039A0NV::TOUCHREPORT_OFF,  //no touch reporting
    NULL, NULL);                  //dont return rendered text width/height

  //write info text on the screen using the default TTF font
  cfaModule.cmdTTFNewText(
    2,                            //new object ID
    "Left/Right Keys Change Size",  //the text
    475, 95,                      //X, Y coords
    CFA039A0NV::JUSTIFY_RIGHT,    //right-justify text
    1,                            //z-index of 1
    CFA039A0NV::TOUCHREPORT_OFF,  //no touch reporting
    NULL, NULL);                  //dont return rendered text width/height
//strcpy_P(&outPacket.data[9], (const char *)PSTR("Left/Right Keys Change Size")); //the text    

  //write info text on the screen using the default TTF font
  cfaModule.cmdTTFNewText(
    3,                            //new object ID
    "X Key Clears the Screen",    //the text
    475, 120,                     //X, Y coords
    CFA039A0NV::JUSTIFY_RIGHT,    //right-justify text
    1,                            //z-index of 1
    CFA039A0NV::TOUCHREPORT_OFF,  //no touch reporting
    NULL, NULL);                  //dont return rendered text width/height

  //create the sketch gfx object, make it cover the full screen
  //no fill color
  cfaModule.cmdSketchNewSurface(
    10,                           //new object ID
    0, 0,                         //X, Y coords
    470, 128,                     //width, height
    false,                        //no background fill (transparent)
    10,                           //z-index of 10
    CFA039A0NV::TOUCHREPORT_OFF); //no touch reporting

  //set arrow key colors to white
  cfaModule.cmdSetKeypadAndIndicatorLEDColor(
    CFA039A0NV::KEY_UP | CFA039A0NV::KEY_DOWN |
    CFA039A0NV::KEY_LEFT | CFA039A0NV::KEY_RIGHT, //set the color of these keys
    CFA039A0NV::INDICATORLED_NONE,  //dont set any indicator led colors
    0xFF, 0xFF, 0XFF);              //R, G, B values

  //set tick key color to off
  cfaModule.cmdSetKeypadAndIndicatorLEDColor(
    CFA039A0NV::KEY_ENTER,          //set the color of these keys
    CFA039A0NV::INDICATORLED_NONE,  //dont set any indicator led colors
    0, 0, 0);                       //R, G, B values

  //set X key color to red
  cfaModule.cmdSetKeypadAndIndicatorLEDColor(
    CFA039A0NV::KEY_CANCEL,         //set the color of these keys
    CFA039A0NV::INDICATORLED_NONE,  //dont set any indicator led colors
    0xFF, 0, 0);                    //R, G, B values

  //set inital indicator led color to red (color 1 in rainbow)
  cfaModule.cmdSetKeypadAndIndicatorLEDColor(
    CFA039A0NV::KEY_NONE,                     //set the color of these keys
    CFA039A0NV::INDICATORLED_ALL,             //dont set any indicator led colors
    rainbow[0][0], rainbow[0][1], rainbow[0][1]); //R, G, B values

  //turn on display-wide touch reporting
  cfaModule.cmdSetTouchscreenReporting(CFA039A0NV::TOUCHREPORT_ALL);

  //turn on keypad reporting
  //report no key presses, report all key releases
  cfaModule.cmdSetKeypadReporting(CFA039A0NV::KEY_NONE, CFA039A0NV::KEY_ALL);

  //finished setting up the display, sit in a main loop and wait for report packets
  currentColor = 0; //red
  currentSize = 10; //10px circle
  //update RGB values for drawing
  R = rainbow[currentColor][0];
  G = rainbow[currentColor][1];
  B = rainbow[currentColor][2];
  //main loop
  while(1)
  {
    //wait until we get a report packet from the module
    //get all report packet types
    cfaModule.getReportPacket((PacketReports_t)0xFF, &inPacket, 0);
		cfaModule.debugPrintPacket("IN", &inPacket);

    //check the incoming packet type
    if (inPacket.command == PRPT_TOUCH)
    {
      //touch screen report packet
      if (inPacket.data[2] & 0b110)
      {
        //it's either a press or drag report packet
        //first convert current RGB rainbow color to RGB565
        RGB565 = ((R & 0b11111000) << 8) | ((G & 0b11111100) << 3) | (B >> 3);
        //draw a circle on the sketch object
        //use the X,Y position from the touch report packet
        uint16_t touchX = inPacket.data[3] | (inPacket.data[4] << 8);
        uint16_t touchY = inPacket.data[5] | (inPacket.data[6] << 8);
        Serial.print(F("TOUCH COORD: "));
        Serial.print(touchX);
        Serial.print("x");
        Serial.println(touchY);
        cfaModule.cmdSketchDrawCircle(
          10,                            //existing sketch gfx object ID number  
          touchX,                        //X center position
          touchY,                        //Y center position
          currentSize,                   //circle radius
          true,                          //fill with color
          RGB565,                        //fill color
          false,                         //no border
          0);                            //border color (not used)
      }
    }
    else if (inPacket.command == PRPT_KEY)
    {
      //it's a  keypad report packet
      if (inPacket.data[0] == 12)
      {
        //cancel key has been pressed & released
        //clear the sketch object by deleting it and re-adding it
        cfaModule.cmdRemoveGraphicsObject(10);

        //re-create the sketch gfx object, make it cover the full screen
        //no fill color
        cfaModule.cmdSketchNewSurface(
          10,                           //new object ID
          0, 0,                         //X, Y coords
          470, 128,                     //width, height
          false,                        //no background fill (transparent)
          10,                           //z-index of 10
          CFA039A0NV::TOUCHREPORT_OFF); //no touch reporting
      }

      if ((inPacket.data[0] == 7) || (inPacket.data[0] == 8))
      {
        //up or down key has been pressed & released
        if (inPacket.data[0] == 8)
        {
          //down key has been pressed, so next color
          currentColor++;
          if (currentColor > RAINBOW_COLORS-1)
            //limit
            currentColor = 0;
        }
        else
        {
          //up key has been pressed, so previous color
          currentColor--;
          if (currentColor < 0)
            //limit
            currentColor = RAINBOW_COLORS-1;
        }
        //debug output
        Serial.print(F("New color index: "));
        Serial.println(currentColor);

        //update RGB values for drawing
        R = rainbow[currentColor][0];
        G = rainbow[currentColor][1];
        B = rainbow[currentColor][2];

        //update indicator leds
        cfaModule.cmdSetKeypadAndIndicatorLEDColor(
          CFA039A0NV::KEY_NONE,                     //set the color of these keys
          CFA039A0NV::INDICATORLED_ALL,             //dont set any indicator led colors
          rainbow[currentColor][0],                 //red value
          rainbow[currentColor][1],                 //green value
          rainbow[currentColor][2]);                //blue value
      }

      if (inPacket.data[0] == 9)
      {
        //left key has been pressed & released
        //decrease draw size by 5px
        currentSize -= 5;
        if (currentSize < 1)
          //limit
          currentSize = 1;
        //debug output
        Serial.print(F("New size: "));
        Serial.println(currentSize);
      }
      if (inPacket.data[0] == 10)
      {
        //right key has been pressed & released
        currentSize += 5;
        if (currentSize > 30)
          //limit
          currentSize = 30;
        //debug output
        Serial.print(F("New size: "));
        Serial.println(currentSize);
      }                
    }
    //continue while(1) loop
  }
  //we never get here
}


